#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <tk.h>

#include "vrml.h"

#define	ACTION_SCAN	0
#define	ACTION_PARSE	1
#define	ACTION_CONT	2
#define	ACTION_SKIP	3
#define	ACTION_END	4


void		SmRedrawCell _ANSI_ARGS_ ((Cell *cell));


/********************************************************************************
 *
 * global data structures
 *
 ********************************************************************************/

static char _buffer[BUFSIZE];
static char *_inptr;
static char *_end;

#ifdef DEBUG
static int tokenCount;
#endif

static State 	rootState;
static SurfaceProperties	rootSurface;
static TransformProperties	rootTransform;

static DecodeTable decodeTable[] = {

    {"AsciiText",		NODE_AsciiText,		Vrml_read_AsciiText,		Vrml_free_AsciiText,		Vrml_render_AsciiText},
    {"Cone",			NODE_Cone,		Vrml_read_Cone,			NULL,				Vrml_render_Cone},
    {"Cube",			NODE_Cube,		Vrml_read_Cube,			NULL,				Vrml_render_Cube},
    {"Cylinder",		NODE_Cylinder,		Vrml_read_Cylinder,		NULL,				Vrml_render_Cylinder},
    {"IndexedFaceSet",		NODE_IndexedFaceSet,	Vrml_read_IndexedFaceSet,	Vrml_free_IndexedFaceSet,	Vrml_render_IndexedFaceSet},
    {"IndexedLineSet",		NODE_IndexedLineSet,	Vrml_read_IndexedLineSet,	Vrml_free_IndexedLineSet,	Vrml_render_IndexedLineSet},
    {"PointSet",		NODE_PointSet,		Vrml_read_PointSet,		NULL,				Vrml_render_PointSet},
    {"Sphere",			NODE_Sphere,		Vrml_read_Sphere,		NULL,				Vrml_render_Sphere},

    {"Coordinate3",		NODE_Coordinate3,	Vrml_read_Coordinate3,		Vrml_free_Coordinate3,		Vrml_render_Coordinate3},
    {"FontStyle",		NODE_FontStyle,		Vrml_read_FontStyle,		NULL,				Vrml_render_FontStyle},
    {"Info",			NODE_Info,		Vrml_read_Info,			Vrml_free_Info,			Vrml_render_Info},
    {"LOD",			NODE_LOD,		Vrml_read_LOD,			Vrml_free_LOD,			Vrml_render_LOD},
    {"Material",		NODE_Material,		Vrml_read_Material,		Vrml_free_Material,		Vrml_render_Material},
    {"MaterialBinding",		NODE_MaterialBinding,	Vrml_read_MaterialBinding,	NULL,				Vrml_render_MaterialBinding},
    {"Normal",			NODE_Normal,		Vrml_read_Normal,		Vrml_free_Normal,		Vrml_render_Normal},
    {"NormalBinding",		NODE_NormalBinding,	Vrml_read_NormalBinding,	NULL,				Vrml_render_NormalBinding},
    {"Texture2",		NODE_Texture2,		Vrml_read_Texture2,		Vrml_free_Texture2,		Vrml_render_Texture2},
    {"Texture2Transform",	NODE_Texture2Transform,	Vrml_read_Texture2Transform,	NULL,				Vrml_render_Texture2Transform},
    {"TextureCoordinate2",	NODE_TextureCoordinate2,Vrml_read_TextureCoordinate2,	Vrml_free_TextureCoordinate2,	Vrml_render_TextureCoordinate2},
    {"ShapeHints",		NODE_ShapeHints,	Vrml_read_ShapeHints,		NULL,				Vrml_render_ShapeHints},

    {"MatrixTransform",		NODE_MatrixTransform,	Vrml_read_MatrixTransform,	NULL,				Vrml_render_MatrixTransform},
    {"Rotation",		NODE_Rotation,		Vrml_read_Rotation,		NULL,				Vrml_render_Rotation},
    {"Scale",			NODE_Scale,		Vrml_read_Scale,		NULL,				Vrml_render_Scale},
    {"Transform",		NODE_Transform,		Vrml_read_Transform,		NULL,				Vrml_render_Transform},
    {"Translation",		NODE_Translation,	Vrml_read_Translation,		NULL,				Vrml_render_Translation},

    {"OrthographicCamera",	NODE_OrthographicCamera,Vrml_read_OrthographicCamera,	NULL,				Vrml_render_OrthographicCamera},
    {"PerspectiveCamera",	NODE_PerspectiveCamera,	Vrml_read_PerspectiveCamera,	NULL,				Vrml_render_PerspectiveCamera},
    {"DirectionalLight",	NODE_DirectionalLight,	Vrml_read_DirectionalLight,	NULL,				Vrml_render_DirectionalLight},
    {"PointLight",		NODE_PointLight,	Vrml_read_PointLight,		NULL,				Vrml_render_PointLight},
    {"SpotLight",		NODE_SpotLight,		Vrml_read_SpotLight,		NULL,				Vrml_render_SpotLight},

    {"Group",			NODE_Group,		Vrml_read_Group,		Vrml_free_Group,		Vrml_render_Group},
    {"Separator",		NODE_Separator,		Vrml_read_Separator,		Vrml_free_Separator,		Vrml_render_Separator},
    {"Switch",			NODE_Switch,		Vrml_read_Switch,		Vrml_free_Switch,		Vrml_render_Switch},
    {"TransformSeparator",	NODE_TransformSeparator,Vrml_read_TransformSeparator,	Vrml_free_TransformSeparator,	Vrml_render_TransformSeparator},
    {"WWWAnchor",		NODE_WWWAnchor,		Vrml_read_WWWAnchor,		Vrml_free_WWWAnchor,		Vrml_render_WWWAnchor},
    
    {"WWWInline",		NODE_WWWInline,		Vrml_read_WWWInline,		Vrml_free_WWWInline,		Vrml_render_WWWInline},

    {"Script",			NODE_Script,		Vrml_read_Script,		Vrml_free_Script,		Vrml_render_Script},

    {NULL,			NODE_EndLOD,			NULL,			NULL,				Vrml_finish_LOD},
    {NULL,			NODE_EndGroup,			NULL,			NULL,				Vrml_finish_Group},
    {NULL,			NODE_EndSeparator,		NULL,			NULL,				Vrml_finish_Separator},
    {NULL,			NODE_EndSwitch,			NULL,			NULL,				Vrml_finish_Switch},
    {NULL,			NODE_EndTransformSeparator,	NULL,			NULL,				Vrml_finish_TransformSeparator},
    {NULL,			NODE_EndWWWAnchor,		NULL,			NULL,				Vrml_finish_WWWAnchor},
    
    {NULL,			0,			NULL,			NULL,				NULL},
};


/********************************************************************************
 *
 * get_char
 *
 * Read in a character from the input stream.
 *
 * Return -1 upon end-of-file.
 *
 *******************************************************************************/

static int
get_char(channel, c)
     Tcl_Channel channel;
     char *c;
{
    int len;

    if (_inptr == _end) {
	if ((len = Tcl_Read(channel, _buffer, BUFSIZE)) < 1) {
	    return -1;
	}
	_inptr = _buffer;
	_end = _buffer + len;
    }
    *c = *_inptr++;
    return 0;
}


/********************************************************************************
 *
 * put_char
 *
 * Return a character to the input stream.
 *
 *******************************************************************************/

static void
put_char(c)
     char c;
{
    *--_inptr = c;
}


/********************************************************************************
 *
 * Vrml_get_token
 *
 * Read in a token from the input stream. Memory for the token is allocated
 * dynamically, and a pointer to the token is returned.
 *
 * Tokens are delimited by white spaces, '{', '}', '[', ']', ',', '|' and '#'.
 *
 * Return:		TOKEN_OPEN_PAREN
 *			TOKEN_CLOSE_PAREN
 *			TOKEN_OPEN_BRACKET
 *			TOKEN_CLOSE_BRACKET
 *			TOKEN_COMMA
 *			TOKEN_BAR
 *			TOKEN_OUT_OF_MEMORY
 *			TOKEN_EOF
 *			TOKEN_END
 *
 ********************************************************************************/

int
Vrml_get_token(channel, token)
     Tcl_Channel channel;
     char **token;
{
    int len, size, skip;
    int quote, escape;
    char c, *ptr, *new, *buffer;

    quote = escape = 0;
    if ((buffer = (char *) ckalloc(BUFSIZE)) == NULL) {
	return TOKEN_OUT_OF_MEMORY;
    }
    ptr = buffer;
    len = -1;
    skip = 1;
    size = BUFSIZE;

    while (get_char(channel, &c) != -1) {

	/* skip initial white spaces */
	if (skip && isspace(c)) continue;

	if (escape || quote) goto insert;
	skip = 0;

	switch(c) {

	  case '#':
	    while (get_char(channel, &c) != -1) {
		if (c == '\n') break;
	    }
	    if (c != '\n') {
		(void) ckfree((void *) buffer);
		return TOKEN_END;
	    }
	    if (len > 0) {
		*(buffer+len) = '\0';
		*token = buffer;

#ifdef DEBUG
		tokenCount++;
#endif

		return TOKEN_WORD;
	    }
	    skip = 1;
	    continue;

	  case 92:
	    escape = 1;
	    if (len == -1) len = 0;
	    continue;

	  case '{':
	  case '}':
	  case '(':
	  case ')':
	  case '[':
	  case ']':
	  case ',':
	  case '|':
	    if (len == -1) {
		(void) ckfree((void *) buffer);
		if (c == '(') return TOKEN_OPEN_PAREN;
		else if (c == ')') return TOKEN_CLOSE_PAREN;
		else if (c == '{') return TOKEN_OPEN_CURLY;
		else if (c == '}') return TOKEN_CLOSE_CURLY;
		else if (c == '[') return TOKEN_OPEN_BRACKET;
		else if (c == ']') return TOKEN_CLOSE_BRACKET;
		else if (c == ',') return TOKEN_COMMA;
		else return TOKEN_BAR;
	    }
	    else {
		put_char(c);
		*(buffer+len) = '\0';
		*token = buffer;

#ifdef DEBUG
		tokenCount++;
#endif

		return TOKEN_WORD;
	    }

	  case '"':
	    quote = 1;
	    len = 0;
	    continue;

	  default:
	    if (isspace(c)) {
		*(buffer+len) = '\0';
		*token = buffer;

#ifdef DEBUG
		tokenCount++;
#endif

		return TOKEN_WORD;
	    }
	    if (len == -1) {
		len  = 0;
	    }
	}

	/* if we get here, we must be dealing with a non-special character */

      insert:
	if (escape || (c != '\\')) {
	    if (!escape && c == '"') {
		*(buffer+len) = '\0';
		*token = buffer;

#ifdef DEBUG
		tokenCount++;
#endif

		return TOKEN_WORD;
	    }
	    if (len == (size - 1)) {
		if ((new = ckrealloc(buffer, size + BUFSIZE)) == NULL) {
		    ckfree((void *) buffer);
		    return TOKEN_OUT_OF_MEMORY;
		}
		buffer = new;
		size += BUFSIZE;
		ptr = buffer + len;
	    }
	    *ptr++ = c;
	    len++;
	    escape = 0;
	}
	else {
	    escape = 1;
	}
    }
    (void) ckfree((void *) buffer);
    return (len == -1) ? TOKEN_END : TOKEN_EOF;
}


/********************************************************************************
 *
 * Vrml_free_token
 *
 * Free a previously allocated token.
 *
 ********************************************************************************/
 
void
Vrml_free_token(token)
     char *token;
{
    (void) ckfree((void *) token);

#ifdef DEBUG
    tokenCount--;
#endif

}


/********************************************************************************
 *
 * check_version
 *
 * Make sure we are reading a Version 1.0 document.
 *
 * Return:		 0 if we are,
 *			-1 if we are not
 *
 ********************************************************************************/

static int
check_version(channel)
     Tcl_Channel channel;
{
    int passed;
    char c, *p;
    char *version = "#VRMLV1.0ascii";

    p = version;
    passed = -1;
    while (get_char(channel, &c) != -1) {
	if (c == '\n') {
	    return passed;
	}

	if (isspace(c)) continue;

	if (passed != 0) {
	    if (c == *p++) {
		if (*p == '\0') {
		    passed = 0;
		}
	    }
	    else {
		return -1;
	    }
	}
    }
    return passed;
}


/********************************************************************************
 *
 * skip_node
 *
 * Return:	TCL_OK if successful,
 *		TCL_ERROR if error is encountered
 *
 ********************************************************************************/

int
skip_node(interp, channel, argv)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
{
    int count, done;
    char *token;

    count = 0;
    done = 0;
    while (!done) {
	switch(Vrml_get_token(channel, &token)) {
	    
	  case TOKEN_OPEN_CURLY:
	    count++;
	    break;

	  case TOKEN_CLOSE_CURLY:
	    count--;
	    if (count == 0) {
		done = 1;
	    }
	    else if (count < 0) {
		Tcl_AppendResult(interp, argv, ": extra '}' in input", (char *) NULL);
		return TCL_ERROR;
	    }
	    break;
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return TCL_ERROR;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    return TCL_ERROR;
	    
	  case TOKEN_WORD:
	    Vrml_free_token(token);
	    break;
	}
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * dispatch_node
 *
 * Return:		Pointer to new node, or
 *			(Node *) NULL when current node is not recognized, or
 *			(Node *) -1 when an error in input is encountered
 *
 ********************************************************************************/

static Node *
dispatch_node(interp, channel, argv, token, new, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     char *token;
     Node *new, **names, **inlines, **textures;
{
    DecodeTable *tab = decodeTable;
    char *name, *type;
    Node *p;

    name = NULL;
    if (!strcmp(token, "USE")) {
	
	switch (Vrml_get_token(channel, &name)) {
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_END:
	  case TOKEN_EOF:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": illegal USE node name", (char *) NULL);
	    return (Node *) -1;
	}

	/* look for node with the given name */

	if (*names == NULL) {
	    Tcl_AppendResult(interp, argv, ": USE node not found \"", name, "\"", (char *) NULL);
	    Vrml_free_token(name);
	    return (Node *) -1;
	}
	else {
	    for (p = *names; p; p = p->namelist) {
		if (!strcmp(p->name, name)) {
		    new->namelist = p;
		    new->name = name;
		    new->code = NODE_USE;
		    return new;
		}
	    }
	    Tcl_AppendResult(interp, argv, ": USE node not found \"", name, "\"", (char *) NULL);
	    Vrml_free_token(name);
	    return (Node *) -1;
	}
    }
    else if (!strcmp(token, "DEF")) {

	/* get name */
	switch (Vrml_get_token(channel, &name)) {
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_END:
	  case TOKEN_EOF:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": illegal DEF node name", (char *) NULL);
	    return (Node *) -1;
	}

	/* get type */
	switch (Vrml_get_token(channel, &type)) {
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    Vrml_free_token(name);
	    return (Node *) -1;

	  case TOKEN_END:
	  case TOKEN_EOF:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    Vrml_free_token(name);
	    return (Node *) -1;
	    
	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": illegal DEF node type", (char *) NULL);
	    Vrml_free_token(name);
	    return (Node *) -1;
	}
    }
    else {
	name = NULL;
	type = token;
    }

    for (; tab->code; tab++) {
	if (tab->type && !strcmp(tab->type, type)) {

#ifdef DEBUG
	    printf("%s: Node %s\n", argv, type);
#endif

	    if (tab->new == NULL) {

#ifdef DEBUG
		printf("%s: no %s reader available; node skipped\n", argv, type);
#endif

		if (name) {
		    Vrml_free_token(name);
		    Vrml_free_token(type);
		}
		return (skip_node(channel, argv) == TCL_OK) ? NULL : (Node *) -1;
	    }
	    else {
		Node *result;
		
		new->code = tab->code;
		result =  (tab->new)(interp, channel, argv, new, names, inlines, textures);

		/* record name <--> node association */
		if (name) {
		    Vrml_free_token(type);
		    if ((result != NULL) && (result != (Node *) -1)) {
			
			/* insert name into name list */
			result->namelist = *names;
			*names = result;
			result->name = name;
		    }
		    else {
			Vrml_free_token(name);
		    }
		}

		/* record nodes that reference external URLs: only original nodes are recorded  */ 
		if ((result != NULL) && (result != (Node *) -1) && (result->origin == NULL) &&
		    ((tab->code == NODE_WWWInline) || (tab->code == NODE_Texture2))) {

		    if (tab->code == NODE_WWWInline) {
			if (result->node.wwwInline.name) {
			    result->inlinelist = *inlines;
			    *inlines = result;
			}
		    }
		    else {
			if (result->node.texture2.filename) {
			    result->texturelist = *textures;
			    *textures = result;
			}
		    }
		}
		return result;
	    }
	}
    }

#ifdef DEBUG
    printf("%s: unknown node \"%s\" skipped\n", argv, type);
#endif

    if (name) {
	Vrml_free_token(name);
	Vrml_free_token(type);
    }
    return (skip_node(interp, channel, argv) == TCL_OK) ? NULL : (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_read_node
 *
 * Return:		Pointer to new node, or
 *			NULL when end of input is reached, or
 *			-1 when an error in input is encountered
 *
 ********************************************************************************/

Node *
Vrml_read_node(interp, channel, argv, names, inlines, textures, recursive, save)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node **names, **inlines, **textures;
     int recursive;
     char *save;
{
    char *token;
    Node *node, *new;

    if ((new = (Node *) ckalloc(sizeof(Node))) == NULL) {
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return ((Node *) -1);
    }

    new->name = NULL;

    new->namelist = NULL;
    new->inlinelist = NULL;
    new->texturelist = NULL;

    new->label = NULL;
    new->code = -1;
    new->parent = NULL;
    new->sibling = NULL;
    new->child = NULL;
    new->toplevel = NULL;
    new->next = NULL;
    new->origin = NULL;

    new->infos = NULL;
    new->scripts = NULL;
    new->lights = NULL;
    new->cameras = NULL;
    new->objects = NULL;
    new->anchors = NULL;
    new->lods = NULL;
    new->map = NULL;

    if (save) {
	token = save;
    }

    while (1) {
	switch(save ? TOKEN_WORD: Vrml_get_token(channel, &token)) {

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    (void) ckfree((void *) new);
	    return (Node *) -1;

	  case TOKEN_END:
	    (void) ckfree((void *) new);
	    return NULL;

	  case TOKEN_EOF:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    (void) ckfree((void *) new);
	    return (Node *) -1;

	  case TOKEN_WORD:
	    node = dispatch_node(interp, channel, argv, token, new, names, inlines, textures);
	    if (!save) Vrml_free_token(token);
	    if (node == (Node *) -1) {
		(void) ckfree((void *) new);
		return node;
	    }
	    else if (node != NULL) {
		return node;
	    }
	    save = NULL;
	    break;

	  case TOKEN_CLOSE_CURLY:
	    (void) ckfree((void *) new);
	    if (recursive) return NULL;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": expecting a node", (char *) NULL);
	    (void) ckfree((void *) new);
	    return (Node *) -1;
	}
    }
}


/********************************************************************************
 *
 * Vrml_free_node
 *
 ********************************************************************************/

void
Vrml_free_node(node)
     Node *node;
{
    DecodeTable *tab;

    if (node->label) (void) ckfree((void *) node->label);

    if (!node->origin) {
	if (node->name) Vrml_free_token(node->name);
	if (node->code != -1) {
	    for (tab = decodeTable; tab->type; tab++) {
		if (node->code == tab->code) {
		    if (tab->free) (tab->free)(node);
		    break;
		}
	    }
	}
    }
    (void) ckfree((void *) node);
}


/********************************************************************************
 *
 * copy_node
 *
 * copy subtree rooted at node.
 *
 ********************************************************************************/

static Node *
copy_node(node)
     Node *node;
{
    Node *new, *p, *sibling, *child;

    if ((new = (Node *) ckalloc(sizeof(Node))) == NULL) {
	return NULL;
    }

    memcpy((void *) new, (void *) node, sizeof(Node));

    new->infos = NULL;
    new->scripts = NULL;
    new->lights = NULL;
    new->cameras = NULL;
    new->objects = NULL;
    new->anchors = NULL;
    new->lods = NULL;
    new->map = NULL;

    new->parent = NULL;
    new->child = NULL;
    new->sibling = NULL;
    new->next = NULL;
    new->toplevel = NULL;
    new->origin = node;

    sibling = NULL;
    for (p = node->child; p; p = p->sibling) {
	child = copy_node(p);
	if (child) {
	    if (sibling) {
		sibling->sibling = child;
	    }
	    else {
		new->child = child;
	    }
	    sibling = child;
	}
	else {
	    break;
	}
    }
    return new;
}


/********************************************************************************
 *
 * render_node
 *
 ********************************************************************************/

static int
render_node(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    DecodeTable *tab = decodeTable;

    if ((node->label = (char *) ckalloc(strlen(state->prefix) + 20)) == NULL) {
	Tcl_SetResult(interp, "out of memory", TCL_STATIC);
	return 2;
    }
    sprintf(node->label, "%s%d", state->prefix, state->id++);

    if (node->code != NODE_USE) {
	for (; tab->code; tab++) {
	    if (tab->code == node->code) {

#ifdef DEBUG
		if (tab->type) printf("rendering node: %s\n", tab->type);
#endif

		if (tab->render) {
		    return (tab->render)(interp, node, state);
		}
		break;
	    }
	}
    }
    return 0;
}


/********************************************************************************
 *
 * flatten_node
 *
 ********************************************************************************/

static Node *
flatten_node(node, names)
     Node *node;
     Node **names;
{
    int i, which;
    Node *p, *q;

    q = node;

    if ((node->code != NODE_Switch) || (node->node.which.whichChild == -3)) {
	for (p = node->child; p; p = p->sibling) {
	    q->next = p;
	    q = flatten_node(p, names);
	}
    }
    else if (node->code == NODE_Switch) {
	if ((which = node->node.which.whichChild) > 0) {
	    for (i = 0, p = node->child; i < which; i++) {
		p = p->sibling;
	    }
	    node->next = p;
	    q = flatten_node(p, names);
	}
    }

    /* handle USE nodes here */
    if (node->code == NODE_USE) {
	if ((p = node->namelist)) {
	    node->next = copy_node(p);
	    q = flatten_node(node->next, names);
	}
    }

    /* add an End node if Group node */

    switch(node->code) {
      case NODE_LOD:
      case NODE_Group:
      case NODE_Separator:
      case NODE_Switch:
      case NODE_TransformSeparator:
      case NODE_WWWAnchor:
	q->next = (Node *) ckalloc(sizeof(Node));
	q = q->next;
	q->code = node->code + 100;
	q->name = NULL;
	q->label = NULL;
	q->sibling = NULL;
	q->child = NULL;
	q->toplevel = NULL;
	q->namelist = NULL;

	q->infos = NULL;
	q->scripts = NULL;
	q->lights = NULL;
	q->cameras = NULL;
	q->objects = NULL;
	q->anchors = NULL;
	q->lods = NULL;
	q->map = NULL;
    }
    q->next = NULL;
    return q;
}


/********************************************************************************
 *
 * reset_states
 *
 *
 ********************************************************************************/

static void
reset_states(interp, state)
     Tcl_Interp *interp;
     State *state;
{
    int i, j;
    Node *p, *q, *child, *sibling;
    SurfaceProperties *r, *s;
    TransformProperties *t, *u;
    CurrentModel *m, *n;
    
    _inptr = _buffer + BUFSIZE;
    _end = _buffer + BUFSIZE;

    if (state->list != NULL) {
	for (p = state->list; p; p = q) {

	    /* must handle Switch node differently here */
	    if ((p->code == NODE_Switch) && (p->node.which.whichChild != -3)) {
		for (i = 0, child = p->child; child; i++, child = sibling) {
		    sibling = child->sibling;
		    if (i != (const) p->node.which.whichChild) {
			Vrml_free_node(child);
		    }
		}
	    }
	    q = p->next;
	    p->child = NULL;
	    Vrml_free_node(p);
	}
	state->list = NULL;

#ifdef DEBUG
	printf("token count = %d\n", tokenCount);
#endif

    }
    if (state->prefix) {
	(void) ckfree((void *) state->prefix);
	state->prefix = NULL;
    }

#ifdef DEBUG
    tokenCount = 0;
#endif

    state->id = 0;
    state->curr = NULL;
    state->names = NULL;
    state->cell = NULL;

    state->infos = NULL;
    state->scripts = NULL;
    state->objects = NULL;
    state->lights = NULL;
    state->cameras = NULL;
    state->anchors = NULL;
    state->lods = NULL;

    /* empty model stack */
    for (m = state->model; m; m = n) {
	n = m->next;
	(void) ckfree((void *) m);
    }
    state->model = NULL;

    /* empty transform stack */
    for (t = state->transform; t->next; t = u) {
	u = t->next;
	(void) ckfree((void *) t);
    }
    state->transform = &rootTransform;

    /* initialize rootTransform */
    t = &rootTransform;
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    t->global[i][j] = t->local[i][j] = (i == j);
	}
    }
    t->next = NULL;

    /* empty surface stack */
    for (s = state->surface; s->next; s = r) {
	r = s->next;
	(void) ckfree((void *) s);
    }
    state->surface = &rootSurface;

    /* initialize rootSurface */
    s = &rootSurface;
    if (s->coord3) (void) ckfree((void *) s->coord3);
    if (s->ambientColor) (void) ckfree((void *) s->ambientColor);
    if (s->diffuseColor) (void) ckfree((void *) s->diffuseColor);
    if (s->specularColor) (void) ckfree((void *) s->specularColor);
    if (s->emissiveColor) (void) ckfree((void *) s->emissiveColor);
    if (s->shininess) (void) ckfree((void *) s->shininess);
    if (s->transparency) (void) ckfree((void *) s->transparency);
    if (s->normal) (void) ckfree((void *) s->normal);
    if (s->texcoord2) (void) ckfree((void *) s->texcoord2);

    s->coord3 = NULL;
    s->coord3len = 0;

    s->ambientColor = (SFColor *) ckalloc(sizeof(SFColor));
    s->ambientlen = 1;
    s->ambientColor->rgb[0] = s->ambientColor->rgb[1] = s->ambientColor->rgb[2] = 0.2;

    s->diffuseColor = (SFColor *) ckalloc(sizeof(SFColor));
    s->diffuselen = 1;
    s->diffuseColor->rgb[0] = s->diffuseColor->rgb[1] = s->diffuseColor->rgb[2] = 0.8;

    s->specularColor = (SFColor *) ckalloc(sizeof(SFColor));
    s->specularlen = 1;
    s->specularColor->rgb[0] = s->specularColor->rgb[1] = s->specularColor->rgb[2] = 0.0;

    s->emissiveColor = (SFColor *) ckalloc(sizeof(SFColor));
    s->emissivelen = 1;
    s->emissiveColor->rgb[0] = s->emissiveColor->rgb[1] = s->emissiveColor->rgb[2] = 0.0;

    s->shininess = (float *) ckalloc(sizeof(float));
    s->shininesslen = 1;
    s->shininess[0] = 0.2;

    s->transparency = (float *) ckalloc(sizeof(float));
    s->transparencylen = 1;
    s->transparency[0] = 0.0;

    s->materialBinding = ENUM_DEFAULT;

    s->normal = NULL;
    s->normallen = 0;

    s->normalBinding = ENUM_DEFAULT;

    s->image = NULL;
    s->wrapS = s->wrapT = ENUM_REPEAT;
    s->texture2 = NULL;

    s->translation.v[0] = s->translation.v[1] = 0;
    s->rotation = 0;
    s->scaleFactor.v[0] = s->scaleFactor.v[1] = 1;
    s->center.v[0] = s->center.v[1] = 0;

    s->texcoord2 = NULL;
    s->texcoord2len = 0;

    s->next = NULL;

}

/********************************************************************************
 *
 * Vrml_read_vrml
 *
 *
 * switch ((int) clientData)
 *
 *   case ACTION_SCAN: vrml_scan fileId
 *		       break;
 *
 *   case ACTION_PARSE: vrml_parse cell prefix
 *			break;
 *
 *   case ACTION_CONT: vrml_continue fileId
 *		       break;
 *
 *   case ACTION_SKIP: vrml_skip
 *		       break;
 *
 *   case ACTION_END: vrml_end
 *		      break;
 *
 ********************************************************************************/

int
Vrml_read_vrml(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    int mode, action = (int) clientData;
    Tcl_Channel channel;
    Node *node, *names, *toplevel, *inlines, *textures;
    Node *p, *q, *tail;
    CurrentModel *m;
    SFVec3f pos, fwd, up;

#ifndef USE_TCL75
    FILE *file;
#endif

    Model *rootModel;
    char *options[] = {
	"-visible",
	"off",
    };

    switch(action) {

      case ACTION_SCAN:
	reset_states(interp, &rootState);
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " fileId\"", (char *) NULL);
	    return TCL_ERROR;
	}
	break;

      case ACTION_CONT:
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " fileId\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (rootState.cell == NULL) {
	    Tcl_AppendResult(interp, argv[0], ": cannot continue", (char *) NULL);
	    return TCL_ERROR;
	}
	break;

      case ACTION_PARSE:
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " cell prefix\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (rootState.list == NULL) {
	    Tcl_AppendResult(interp, argv[0], ": nothing to parse", (char *) NULL);
	    return TCL_ERROR;
	}
	if ((rootState.cell = SmFindCell(interp, argv[1])) == NULL) {
	    Tcl_AppendResult(interp, argv[0], ": cell \"", argv[1], "\" does not exist.", (char *) NULL);
	    return TCL_ERROR;
	}
	if ((rootState.prefix = (char *) ckalloc(strlen(argv[2]) + 1)) == NULL) {
	    Tcl_AppendResult(interp, argv[0], ": out of memory", (char *) NULL);
	    return TCL_ERROR;
	}
	strcpy(rootState.prefix, argv[2]);
	break;

      case ACTION_SKIP:
	if (argc != 1) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (rootState.cell == NULL) {
	    Tcl_AppendResult(interp, argv[0], ": cannot skip", (char *) NULL);
	    return TCL_ERROR;
	}
	break;

      case ACTION_END:
	if (argc != 1) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	reset_states(interp, &rootState);
	break;
    }

    if ((action == ACTION_SCAN) || (action == ACTION_CONT)) {

#ifdef USE_TCL75
	if ((channel = Tcl_GetChannel(interp, argv[1], &mode)) == NULL) {
	    return TCL_ERROR;
	}
	if (!(mode & TCL_READABLE)) {
	    Tcl_AppendResult(interp, "channel ", argv[1], " is not readable", (char *) NULL);
	    return TCL_ERROR;
	}
#else
	if (Tcl_GetOpenFile(interp, argv[1], 0, 1, &file) != TCL_OK) {
	    return TCL_ERROR;
	}
	channel = fileno(file);
#endif

	if ((rootState.curr) && (rootState.curr->code == NODE_Texture2)) {
	    Node *node = rootState.curr;

	    node->node.texture2.fileImage.width = 0;
	    node->node.texture2.fileImage.height = 0;
	    node->node.texture2.fileImage.comp = 0;
	    node->node.texture2.fileImage.data = NULL;
	    if (node->node.texture2.loaded == 1) {
		node->node.texture2.loaded = 2;
		if (Vrml_read_SFImage(interp, channel, argv[0], &node->node.texture2.fileImage) != TCL_OK) {
		    node->node.texture2.loaded = 3;
		    Tcl_AppendResult(interp, "error reading Texture2 image from fileId", (char *) NULL);
		    return TCL_ERROR;
		}
	    }
	}
	else {
	    if (check_version(channel) == -1) {
		Tcl_AppendResult(interp, argv[1], ": file is not a VRML V1.0 document", (char *) NULL);
		return TCL_ERROR;
	    }

	    names = toplevel = inlines = textures = p = NULL;
	    node = Vrml_read_node(interp, channel, argv[0], &names, &inlines, &textures, 0, NULL);
	    while ((node != NULL) && (node != (Node *) -1)) {

		if (p) {
		    p->toplevel = node;
		}
		else {
		    toplevel = node;
		}
		p = node;
		node = Vrml_read_node(interp, channel, argv[0], &names, &inlines, &textures, 0, NULL);

		while (Tcl_DoOneEvent(TCL_IDLE_EVENTS | TCL_WINDOW_EVENTS | TCL_DONT_WAIT));
	    }

	    if (node == (Node *) -1) {
		for (p = toplevel; p; p = q) {
		    q = p->toplevel;
		    Vrml_free_node(p);
		}

#ifdef DEBUG
		printf("token count = %d\n", tokenCount);
#endif

		reset_states(interp, &rootState);
		return TCL_ERROR;
	    }

	    /* flatten scene graph and insert it into current list */
	    if (toplevel) {
		q = rootState.curr;;
		if (!rootState.list) {
		    rootState.list = toplevel;
		}
		for (p = toplevel; p; p = p->toplevel) {
		    tail = flatten_node(p, &names);
		    if (q) {
			tail->next = q->next;
			q->next = p;
		    }
		    else {
			rootState.curr = p;
		    }
		    q = tail;

		    while (Tcl_DoOneEvent(TCL_IDLE_EVENTS | TCL_WINDOW_EVENTS | TCL_DONT_WAIT));
		}
	    }
	    if (action == ACTION_SCAN) {
		rootState.names = names;
	    }
	}
    }

    if (action == ACTION_END) {
	reset_states(interp, &rootState);
    }
    else if (action == ACTION_SCAN) {
	Vrml_collect_URLs(interp, inlines, textures);
    }
    else {

	if (action == ACTION_PARSE) {

	    /* create root model */
	    if ((m = (CurrentModel *) ckalloc(sizeof(CurrentModel))) == NULL) {
		Tcl_AppendResult(interp, argv[0], ": out of memory", (char *) NULL);
		reset_states(interp, &rootState);
		return TCL_ERROR;
	    }
	    m->node = NULL;
	    m->next = NULL;
	    rootState.model = m;
	    pos.v[0] = pos.v[1] = pos.v[2] = 0;
	    fwd.v[0] = fwd.v[1] = 0; fwd.v[2] = 1;
	    up.v[0] = up.v[2] = 0; up.v[1] = 1;

	    /* make object invisible until it is fully parsed */
	    if (Vrml_new_object(interp, &rootState, NULL, "polygon", &pos, &fwd, &up, 2, options) != TCL_OK) {
		reset_states(interp, &rootState);
		(void) ckfree((void *) m);
		return TCL_ERROR;
	    }
	}

	/* parse node list */

	for (p = rootState.curr; p; p = p->next) {

	    switch(render_node(interp, p, &rootState)) {
	      case 1:
		rootState.curr = p;
		return TCL_OK;

	      case 2:
		reset_states(interp, &rootState);
		return TCL_ERROR;
	    }
	    while (Tcl_DoOneEvent(TCL_IDLE_EVENTS | TCL_WINDOW_EVENTS | TCL_DONT_WAIT));
	}

	/* collect results */
	Vrml_collect_results(interp, &rootState);

	/* make root model visible now */
	if ((rootModel = SmFindModel(rootState.cell, rootState.prefix))) {
	    rootModel->visible = 1;
	    SmRedrawCell(rootState.cell);
	}

	reset_states(interp, &rootState);
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_Init
 *
 *
 ********************************************************************************/

int
Vrml_Init(interp)
     Tcl_Interp *interp;
{
    TransformProperties *x;
    SurfaceProperties *s;

    s = &rootSurface;
    x = &rootTransform;

    rootState.list = NULL;
    rootState.prefix = NULL;
    rootState.model = NULL;
    rootState.transform = x;
    rootState.surface = s;

    /* initialize rootSurface */
    s->coord3 = NULL;
    s->ambientColor = NULL;
    s->diffuseColor = NULL;
    s->specularColor = NULL;
    s->emissiveColor = NULL;
    s->shininess = NULL;
    s->transparency = NULL;
    s->normal = NULL;
    s->texture2 = NULL;
    s->texcoord2 = NULL;

    s->vertexOrdering = ENUM_UNKNOWN_ORDERING;
    s->shapeType = ENUM_UNKNOWN_SHAPE_TYPE;
    s->faceType = ENUM_CONVEX;

    reset_states(interp, &rootState);

    /* create VRML commands */
    Tcl_CreateCommand(interp, "vrml_scan", Vrml_read_vrml, (ClientData) ACTION_SCAN,
		      (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "vrml_parse", Vrml_read_vrml, (ClientData) ACTION_PARSE,
		      (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "vrml_continue", Vrml_read_vrml, (ClientData) ACTION_CONT,
		      (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "vrml_skip", Vrml_read_vrml, (ClientData) ACTION_SKIP,
		      (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "vrml_end", Vrml_read_vrml, (ClientData) ACTION_END,
		      (Tcl_CmdDeleteProc *) NULL);

#ifdef USE_TCL75
    return Tcl_PkgProvide(interp, "vrml", SM_VERSION);
#else
    return TCL_OK;
#endif
}
