/********************************************************************************
 *
 * point.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_PointLight
 *
 * Read in a PointLight node.
 *
 * fields:	on		TRUE	SFBool
 *		intensity	1	SFFloat
 *		color		1 1 1	SFColor
 *		location	0 0 -1	SFVec3f
 *
 *******************************************************************************/

Node *
Vrml_read_PointLight(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodePointLight *light;

    light = &node->node.pointLight;
    light->on = 1;
    light->intensity = 1;
    light->color.rgb[0] = light->color.rgb[1] = light->color.rgb[2] = 1;
    light->location.v[0] = light->location.v[1] = 0;
    light->location.v[2] = 1;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading PointLight", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading PointLight", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad PointLight format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {

	switch(Vrml_get_token(channel, &field)) {

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading PointLight", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading PointLight", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad PointLight format", (char *) NULL);
	    return (Node *) -1;
	}

	if (!strcmp(field, "on")) {
	    if (Vrml_read_SFBool(interp, channel, argv, &light->on) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "intensity")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &light->intensity, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (light->intensity < 0) light->intensity = 0;
	    if (light->intensity > 1) light->intensity = 1;
	}
	else if (!strcmp(field, "location")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &light->location, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "color")) {
	    if (Vrml_read_SFColor(interp, channel, argv, &light->color, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad PointLight field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }

  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_PointLight
 *
 * Render a PointLight node.
 *
 * fields:	on		TRUE	SFBool
 *		intensity	1	SFFloat
 *		color		1 1 1	SFColor
 *		location	0 0 -1	SFVec3f
 *
 *******************************************************************************/

int
Vrml_render_PointLight(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    NodePointLight *light = &node->node.pointLight;

    if (light->on) {
	node->lights = state->lights;
	state->lights = node;

	if ((state->model == NULL) ||
	    (state->model->next == NULL) ||
	    (state->model->next->next == NULL)) {

	    /* stationary light source */
	    light->model = NULL;
	}
	else {

	    /* tracking light source */
	    light->model = state->model->node;
	}
	Vrml_coord3_transform(state, &light->location, &light->wlocation);
    }
    return 0;
}
