/********************************************************************************
 *
 * pointset.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_PointSet
 *
 * Read in a PointSet node.
 *
 * fields:	startIndex	0	SFLong
 *		numPoints	-1	SFLong
 *
 *******************************************************************************/

Node *
Vrml_read_PointSet(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodePointSet *set;

    set = &node->node.pointSet;
    set->startIndex = 0;
    set->numPoints = -1;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading PointSet", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading PointSet", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad PointSet format", (char *) NULL);
	return (Node *) -1;
    }

    /* parse all fields until close curly bracket */
    while (1) {

	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading PointSet", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading PointSet", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad PointSet format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "startIndex")) {
	    if (Vrml_read_SFLong(interp, channel, argv, &set->startIndex, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (set->startIndex < 0) {
		Tcl_AppendResult(interp, argv, ": negative PointSet startIndex", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "numPoints")) {
	    if (Vrml_read_SFLong(interp, channel, argv, &set->numPoints, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (set->numPoints < -1) {
		Tcl_AppendResult(interp, argv, ": negative PointSet numPoints", (char *) NULL);
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad PointSet field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_PointSet
 *
 * Render a PointSet node.
 *
 * fields:	startIndex	0	SFLong
 *		numPoints	-1	SFLong
 *
 *******************************************************************************/

int
Vrml_render_PointSet(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, comp, indices[6];
    int result;
    long start, num;
    char ambientColor[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuseColor[TCL_DOUBLE_SPACE * 3 + 10];
    char specularColor[TCL_DOUBLE_SPACE * 3 + 10];
    char emissiveColor[TCL_DOUBLE_SPACE * 3 + 10];
    char shininess[TCL_DOUBLE_SPACE * 3 + 10];

    char *vertices;
    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];

    char *wrapS, *wrapT;

    char *p;
    char *materials;
    char *textures;
    char *normals;

    char vector[TCL_DOUBLE_SPACE * 3 + 10];

    char doubleW[TCL_DOUBLE_SPACE];
    char doubleX[TCL_DOUBLE_SPACE];
    char doubleY[TCL_DOUBLE_SPACE];
    char doubleZ[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-vertices",
	NULL,
	"-pointsize",
	"1.0",
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-materials",
	NULL,
	"-textures",
	NULL,
	"-normals",
	NULL,
    };
    NodePointSet *set = &node->node.pointSet;

    float s0, s1, s2;
    SFVec3f *vin;
    SFVec3f pos, fwd, up, scale;
    SFVec2f texcoord2;
    SurfaceProperties *s = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];

    argv[7] = ambient;
    argv[9] = diffuse;
    argv[11] = specular;
    argv[13] = emissive;

    /* build up point coordinates */
    vertices = NULL;
    if ((start = set->startIndex) < s->coord3len) {
	if ((num = set->numPoints) == -1) {
	    num = s->coord3len - start;
	}
	if ((num > 0) && ((vertices = (char *) ckalloc(num * (TCL_DOUBLE_SPACE * 3 + 10))) != NULL)) {

	    *vertices = '\0';
	    for (i = 0, vin = s->coord3 + start; i < num; i++, vin++) {

		Tcl_PrintDouble(interp, (double) vin->v[0] * s0, doubleX);
		Tcl_PrintDouble(interp, (double) vin->v[1] * s1, doubleY);
		Tcl_PrintDouble(interp, (double) vin->v[2] * s2, doubleZ);

		sprintf(vertices + strlen(vertices), "{%s %s %s} ", doubleX, doubleY, doubleZ);
	    }
	}
    }
    argv[1] = vertices ? vertices : "";

    /* add ambient */
    if (s->ambientColor) {
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[2], doubleZ);

	sprintf(ambient, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (s->diffuseColor) {
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[2], doubleZ);

	sprintf(diffuse, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (s->specularColor) {
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[2], doubleZ);
	Tcl_PrintDouble(interp, (double) s->shininess[0], doubleW);

	sprintf(specular, "%s %s %s %s", doubleX, doubleY, doubleZ, doubleW);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (s->emissiveColor) {
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[2], doubleZ);

	sprintf(emissive, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* add materials

       default material binding = OVERALL,
     */

    materials = NULL;
    switch(s->materialBinding) {

      case ENUM_DEFAULT:
      case ENUM_PER_PART:
      case ENUM_PER_PART_INDEXED:
      case ENUM_PER_FACE:
      case ENUM_PER_FACE_INDEXED:
      case ENUM_PER_VERTEX:
      case ENUM_PER_VERTEX_INDEXED:
	Vrml_initialize_material_indices(state, indices, (int) start);
	if ((num > 0) && ((materials = (char *) ckalloc(num * 100)) != NULL)) {
	    p = materials;
	    for (i = 0; i < num; i++) {
		Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		sprintf(p, "{%s %s %s %s %s}",
			ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		p += strlen(p);
	    }
	    *p = '\0';
	}
	break;
    }
    argv[15] = materials ? materials : "";

    /* add textures

       {id textureID comp function s-mode t-mode mag-filter min-filter bordercolor texcoords}

     */

    textures = NULL;
    if (s->texture2) {
	comp = s->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;

	wrapS = (s->wrapS == ENUM_REPEAT) ? "repeat" : "clamp";
	wrapT = (s->wrapT == ENUM_REPEAT) ? "repeat" : "clamp";

	if ((s->texcoord2len > 0) && (num > 0) && ((textures = (char *) ckalloc(num * 30 + 100)) != NULL)) {
	    sprintf(textures, "%s %d modulate %s %s nearest nearest {0 0 0} {",
		    s->texture2->label, comp, wrapS, wrapT);
	    p = textures + strlen(textures);
	    for (i = 0, j = 0; i < num; i++, j++) {
		if (j == s->texcoord2len) j = 0;
		Vrml_texcoord2_transform(state, s->texcoord2 + j, &texcoord2);

		Tcl_PrintDouble(interp, (double) texcoord2.v[0], doubleX);
		Tcl_PrintDouble(interp, (double) texcoord2.v[1], doubleY);

		sprintf(p, "{%s %s} ", doubleX, doubleY);
		p += strlen(p);
	    }
	    *p++ = '}';
	    *p = '\0';
	}
    }
    argv[17] = textures ? textures : "";

    /* add normals */
    normals = NULL;
    switch(s->normalBinding) {

      case ENUM_DEFAULT:
	break;

      default:
	Vrml_initialize_normal_indices(state, indices, start);
	if ((num > 0) && ((normals = (char *) ckalloc(num * 100)) != NULL)) {
	    p = normals;
	    for (i = 0; i < num; i++) {
		Vrml_get_normal(interp, state, indices, &scale, vector);
		sprintf(p, "{%s} ", vector);
		p += strlen(p);
	    }
	    *p = '\0';
	}
	break;
    }
    argv[19] = normals ? normals : "";

    /* create point model */
    result = Vrml_new_object(interp, state, node, "point", &pos, &fwd, &up, 20, argv);

    if (vertices) (void) ckfree((void *) vertices);
    if (materials) (void) ckfree((void *) materials);
    if (textures) (void) ckfree((void *) textures);
    if (normals) (void) ckfree((void *) normals);

    return (result == TCL_OK) ? 0 : 2;
}
