#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_push_transform
 *
 ********************************************************************************/

int
Vrml_push_transform(interp, state)
     Tcl_Interp *interp;
     State *state;
{
    int i, j;
    float d[4];
    float (*global)[4], (*local)[4];
    TransformProperties *xform;

    local = state->transform->local;
    global = state->transform->global;

    if ((xform = (TransformProperties *) ckalloc(sizeof(TransformProperties))) == NULL) {
	Tcl_AppendResult(interp, "out of memory", (char *) NULL);
	return TCL_ERROR;
    }

    for (i = 0; i < 3; i++) {
	d[i] = (float) sqrt((double) local[0][i] * local[0][i] +
			    local[1][i] * local[1][i] +
			    local[2][i] * local[2][i]);
    }
    d[3] = 1;
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    xform->global[i][j] = global[i][j];
	    xform->local[i][j] = (i == j) ? d[i] : 0;
	}
    }
    xform->next = state->transform;
    state->transform = xform;

    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_push_surface
 *
 ********************************************************************************/

int
Vrml_push_surface(interp, state)
     Tcl_Interp *interp;
     State *state;
{
    SurfaceProperties *surface;

    if ((surface = (SurfaceProperties *) ckalloc(sizeof(SurfaceProperties))) == NULL) {
	Tcl_AppendResult(interp, "out of memory", (char *) NULL);
	return TCL_ERROR;
    }
    memcpy((char *) surface, (char *) state->surface, sizeof(SurfaceProperties));
    surface->next = state->surface;
    state->surface = surface;
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_push_model
 *
 ********************************************************************************/

int
Vrml_push_model(interp, state, node)
     Tcl_Interp *interp;
     State *state;
     Node *node;
{
    CurrentModel *m;

    if ((m = (CurrentModel *) ckalloc(sizeof(CurrentModel))) == NULL) {
	Tcl_AppendResult(interp, "out of memory", (char *) NULL);
	return TCL_ERROR;
    }
    m->node = node;
    m->next = state->model;
    state->model = m;
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_pop_transform
 *
 ********************************************************************************/

void
Vrml_pop_transform(state)
     State *state;
{
    TransformProperties *xform;

    if ((xform = state->transform)) {
	state->transform = xform->next;
	(void) ckfree((void *) xform);
    }
}


/********************************************************************************
 *
 * Vrml_pop_surface
 *
 ********************************************************************************/

void
Vrml_pop_surface(state)
     State *state;
{
    SurfaceProperties *surface;

    if ((surface = state->surface)) {
	state->surface = surface->next;
	(void) ckfree((void *) surface);
    }
}


/********************************************************************************
 *
 * Vrml_pop_model
 *
 ********************************************************************************/

void
Vrml_pop_model(state)
     State *state;
{
    CurrentModel *m;

    if ((m = state->model)) {
	state->model = m->next;
	(void) ckfree((void *) m);
    }
}


/********************************************************************************
 *
 * Vrml_apply_translation
 *
 * Apply a translation into the current transform matrix.
 *
 *
 *******************************************************************************/

void
Vrml_apply_translation(state, translation)
     State *state;
     SFVec3f *translation;
{
    TransformProperties *p = state->transform;
    float x, y, z;

    x = translation->v[0];
    y = translation->v[1];
    z = translation->v[2];

    p->global[0][3] += p->global[0][0] * x + p->global[0][1] * y + p->global[0][2] * z;
    p->global[1][3] += p->global[1][0] * x + p->global[1][1] * y + p->global[1][2] * z;
    p->global[2][3] += p->global[2][0] * x + p->global[2][1] * y + p->global[2][2] * z;

    p->local[0][3] += p->local[0][0] * x + p->local[0][1] * y + p->local[0][2] * z;
    p->local[1][3] += p->local[1][0] * x + p->local[1][1] * y + p->local[1][2] * z;
    p->local[2][3] += p->local[2][0] * x + p->local[2][1] * y + p->local[2][2] * z;
}


/********************************************************************************
 *
 * Vrml_apply_scale
 *
 * Apply a scale into the current transform matrix.
 *
 *
 *******************************************************************************/

void
Vrml_apply_scale(state, scale)
     State *state;
     SFVec3f *scale;
{
    int i, j;
    TransformProperties *p = state->transform;

    for (i = 0; i < 3; i++) {
	for (j = 0; j < 3; j++) {
	    p->local[i][j] *= scale->v[j];
	    p->global[i][j] *= scale->v[j];
	}
    }
}


/********************************************************************************
 *
 * Vrml_apply_rotation
 *
 * Apply a rotation into the current transform matrix.
 *
 *
 *******************************************************************************/

void
Vrml_apply_rotation(state, rotation)
     State *state;
     float *rotation;
{
    int i, j, k;
    float cosine, sine, lsum, gsum;
    float m[4][4], r[4][4], s[4][4];
    float (*global)[4], (*local)[4];
    float ux, uy, uz;

    cosine = (float) cos((double) rotation[3]);
    sine = (float) sin((double) rotation[3]);

    ux = rotation[0];
    uy = rotation[1];
    uz = rotation[2];

    m[0][0] = ux * ux + cosine * (1 - ux * ux);
    m[0][1] = ux * uy * (1 - cosine) - uz * sine;
    m[0][2] = uz * ux * (1 - cosine) + uy * sine;
    m[0][3] = 0;

    m[1][0] = ux * uy * (1 - cosine) + uz * sine;
    m[1][1] = uy * uy + cosine * (1 - uy * uy);
    m[1][2] = uy * uz * (1 - cosine) - ux * sine;
    m[1][3] = 0;

    m[2][0] = uz * ux * (1 - cosine) - uy * sine;
    m[2][1] = uy * uz * (1 - cosine) + ux * sine;
    m[2][2] = uz * uz + cosine * (1 - uz * uz);
    m[2][3] = 0;

    m[3][0] = m[3][1] = m[3][2] = 0;
    m[3][3] = 1;

    global = state->transform->global;
    local = state->transform->local;

    /* multiply c into m */
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    gsum = lsum = 0;
	    for (k = 0; k < 4; k++) {
		gsum += global[i][k] * m[k][j];
		lsum += local[i][k] * m[k][j];
	    }
	    r[i][j] = gsum;
	    s[i][j] = lsum;
	}
    }

    /* copy r back to c */
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    global[i][j] = r[i][j];
	    local[i][j] = s[i][j];
	}
    }
}


/********************************************************************************
 *
 * Vrml_object_state
 *
 * Compute current object's position relative to its parent.
 *
 *
 *******************************************************************************/

void
Vrml_object_state(state, pos, fwd, up, scale)
     State *state;
     SFVec3f *pos;
     SFVec3f *fwd;
     SFVec3f *up;
     SFVec3f *scale;
{
    double d0, d1, d2;
    double f1, f2;
    float x, y, z;
    TransformProperties *m = state->transform;

    x = m->local[0][3];
    y = m->local[1][3];
    z = m->local[2][3];

    pos->v[0] = m->local[0][3];
    pos->v[1] = m->local[1][3];
    pos->v[2] = m->local[2][3];

    d0 = sqrt((double) m->local[0][0] * m->local[0][0] + m->local[1][0] * m->local[1][0] + m->local[2][0] * m->local[2][0]);
    d1 = sqrt((double) m->local[0][1] * m->local[0][1] + m->local[1][1] * m->local[1][1] + m->local[2][1] * m->local[2][1]);
    d2 = sqrt((double) m->local[0][2] * m->local[0][2] + m->local[1][2] * m->local[1][2] + m->local[2][2] * m->local[2][2]);

    if (d2 > 0) {
	f2 = 1.0 / d2;
	fwd->v[0] = m->local[0][2] * f2;
	fwd->v[1] = m->local[1][2] * f2;
	fwd->v[2] = m->local[2][2] * f2;
    }
    else {
	fwd->v[0] = fwd->v[1] = fwd->v[2] = 0;
    }

    if (d1 > 0) {
	f1 = 1.0 / d1;
	up->v[0] = m->local[0][1] * f1;
	up->v[1] = m->local[1][1] * f1;
	up->v[2] = m->local[2][1] * f1;
    }
    else {
	up->v[0] = up->v[1] = up->v[2] = 0;
    }

    scale->v[0] = d0;
    scale->v[1] = d1;
    scale->v[2] = d2;
}


/********************************************************************************
 *
 * Vrml_coord3_transform
 *
 * Convert a 3d point from local to global space.
 *
 *******************************************************************************/

void
Vrml_coord3_transform(state, in, out)
     State *state;
     SFVec3f *in;
     SFVec3f *out;
{
    int i, j;
    float sum;
    float (*m)[4] = state->transform->global;

    for (i = 0; i < 3; i++) {
	sum = 0;
	for (j = 0; j < 3; j++) {
	    sum += m[i][j] * in->v[j];
	}
	out->v[i] = sum + m[i][3];
    }
}


/********************************************************************************
 *
 * Vrml_vector_transform
 *
 * Convert a directional vector from local to global space.
 *
 *******************************************************************************/

void
Vrml_vector_transform(state, in, out)
     State *state;
     SFVec3f *in;
     SFVec3f *out;
{
    int i, j;
    float v[3];
    float d, sum;
    float (*m)[4] = state->transform->global;

    /* normalize vector */
    d = 1 / sqrt((double) (in->v[0] * in->v[0] + in->v[1] * in->v[1] + in->v[2] * in->v[2]));
    v[0] = in->v[0] * d;
    v[1] = in->v[1] * d;
    v[2] = in->v[2] * d;
    d = 0;
    for (i = 0; i < 3; i++) {
	sum = 0;
	for (j = 0; j < 3; j++) {
	    sum += m[i][j] * v[j];
	}
	out->v[i] = sum;
	d += sum * sum; 
    }
    if (d > 0) {
	d = 1.0 / sqrt((double) d);
	out->v[0] *= d;
	out->v[1] *= d;
	out->v[2] *= d;
    }
}


/********************************************************************************
 *
 * Vrml_texcoord2_transform
 *
 * Apply current transformation to a 2d texture coordinate.
 *
 *
 *******************************************************************************/

void
Vrml_texcoord2_transform(state, in, out)
     State *state;
     SFVec2f *in;
     SFVec2f *out;
{
    float s, t, u, v;
    double sine, cosine;
    SurfaceProperties *surf = state->surface;

    s = in->v[0];
    t = in->v[1];

    /* scale about an arbitrary center point */
    s = (s - surf->center.v[0]) * surf->scaleFactor.v[0];
    t = (t - surf->center.v[1]) * surf->scaleFactor.v[1];

    /* rotate about the center point */
    sine = sin((double) surf->rotation);
    cosine = cos((double) surf->rotation);

    u = s * cosine - t * sine;
    v = s * sine + t * cosine;

    /* translation */
    out->v[0] = (u + surf->center.v[0]) + surf->translation.v[0];
    out->v[1] = (v + surf->center.v[1]) + surf->translation.v[1];
}


/********************************************************************************
 *
 * Vrml_rotate_vector
 *
 * rotate a vector about an arbitrary axis.
 *
 *******************************************************************************/

void
Vrml_rotate_vector(float rotation[], float x, float y, float z, float *vout)
{
    float sine, cosine;
    float rx, ry, rz;

    /* rotate up about orientation vector through angle orientation[3] */
    cosine = (float) cos((double) rotation[3]);
    sine = (float) sin((double) rotation[3]);

    rx = rotation[0];
    ry = rotation[1];
    rz = rotation[2];

    vout[0] = (rx * rx + cosine * (1 - rx * rx)) * x +
	      (rx * ry * (1 - cosine) - rz * sine) * y +
	      (rz * rx * (1 - cosine) + ry * sine) * z;
    vout[1] = (rx * ry * (1 - cosine) + rz * sine) * x +
	      (ry * ry + cosine * (1 - ry * ry)) * y +
	      (ry * rz * (1 - cosine) - rx * sine) * z;
    vout[2] = (rz * rx * (1 - cosine) - ry * sine) * x +
	      (ry * rz * (1 - cosine) + rx * sine) * y +
	      (rz * rz + cosine * (1 - rz * rz)) * z;
}


/********************************************************************************
 *
 * Vrml_output_SFVec3f
 *
 * Output an SFVec3f.
 *
 *******************************************************************************/

void
Vrml_output_SFVec3f(interp, buffer, vec)
     Tcl_Interp *interp;
     char *buffer;
     SFVec3f *vec;
{
    char *p;

    Tcl_PrintDouble(interp, (double) vec->v[0], buffer);
    p = buffer + strlen(buffer);
    *p++ = ' ';

    Tcl_PrintDouble(interp, (double) vec->v[1], p);
    p = buffer + strlen(buffer);
    *p++ = ' ';

    Tcl_PrintDouble(interp, (double) vec->v[2], p);
}


/********************************************************************************
 *
 * Vrml_get_material
 *
 * Get the next set of materials.
 *
 *******************************************************************************/

void
Vrml_get_materials(interp, state, indices, ambient, diffuse, specular, emissive, shininess)
     Tcl_Interp *interp;
     State *state;
     int *indices;
     char *ambient;
     char *diffuse;
     char *specular;
     char *emissive;
     char *shininess;
{
    int am = indices[0];
    int di = indices[1];
    int sp = indices[2];
    int em = indices[3];
    int sh = indices[4];

    char r[TCL_DOUBLE_SPACE];
    char g[TCL_DOUBLE_SPACE];
    char b[TCL_DOUBLE_SPACE];

    SurfaceProperties *s = state->surface;
    
    /* ambientColor */
    if (s->ambientlen > 0) {
	if (am >= s->ambientlen) {
	    am = s->ambientlen - 1;
	}
	Tcl_PrintDouble(interp, (double) s->ambientColor[am].rgb[0], r);
	Tcl_PrintDouble(interp, (double) s->ambientColor[am].rgb[1], g);
	Tcl_PrintDouble(interp, (double) s->ambientColor[am].rgb[2], b);
	sprintf(ambient, "{%s %s %s}", r, g, b);
    }
    else {
	sprintf(ambient, "{0.0 0.0 0.0}");
    }
    
    /* diffuseColor */
    if (s->diffuselen > 0) {
	if (di >= s->diffuselen) {
	    di = s->diffuselen - 1;
	}
	Tcl_PrintDouble(interp, (double) s->diffuseColor[di].rgb[0], r);
	Tcl_PrintDouble(interp, (double) s->diffuseColor[di].rgb[1], g);
	Tcl_PrintDouble(interp, (double) s->diffuseColor[di].rgb[2], b);
	sprintf(diffuse, "{%s %s %s}", r, g, b);
    }
    else {
	sprintf(diffuse, "{0.0 0.0 0.0}");
    }

    /* specularColor */
    if (s->specularlen > 0) {
	if (sp >= s->specularlen) {
	    sp = s->specularlen - 1;
	}
	Tcl_PrintDouble(interp, (double) s->specularColor[sp].rgb[0], r);
	Tcl_PrintDouble(interp, (double) s->specularColor[sp].rgb[1], g);
	Tcl_PrintDouble(interp, (double) s->specularColor[sp].rgb[2], b);
	sprintf(specular, "{%s %s %s}", r, g, b);
    }
    else {
	sprintf(specular, "{0.0 0.0 0.0}");
    }

    /* emissiveColor */
    if (s->emissivelen > 0) {
	if (em >= s->emissivelen) {
	    em = s->emissivelen - 1;
	}
	Tcl_PrintDouble(interp, (double) s->emissiveColor[em].rgb[0], r);
	Tcl_PrintDouble(interp, (double) s->emissiveColor[em].rgb[1], g);
	Tcl_PrintDouble(interp, (double) s->emissiveColor[em].rgb[2], b);
	sprintf(emissive, "{%s %s %s}", r, g, b);
    }
    else {
	sprintf(emissive, "{0.0 0.0 0.0}");
    }
    
    if (s->shininesslen > 0) {
	if (sh >= s->shininesslen) {
	    sh = s->shininesslen - 1;
	}
	Tcl_PrintDouble(interp, (double) s->shininess[sh], shininess);
    }
    else {
	sprintf(shininess, "0.0");
    }
    
    /* update indices */
    indices[0]++;
    indices[1]++;
    indices[2]++;
    indices[3]++;
    indices[4]++;
    indices[5]++;
    if ((indices[0] >= s->ambientlen) &&
	(indices[1] >= s->diffuselen) &&
	(indices[2] >= s->specularlen) &&
	(indices[3] >= s->emissivelen) &&
	(indices[4] >= s->shininesslen) &&
	(indices[5] >= s->transparencylen)) {
	indices[0] = indices[1] = indices[2] = indices[3] = indices[4] = indices[5] = 0;
    }
}


/********************************************************************************
 *
 * Vrml_get_material_values
 *
 * Get the next set of materials.
 *
 *******************************************************************************/

void
Vrml_get_material_values(interp, state, indices, mat)
     Tcl_Interp *interp;
     State *state;
     int *indices;
     Material *mat;
{
    int am = indices[0];
    int di = indices[1];
    int sp = indices[2];
    int em = indices[3];
    int sh = indices[4];

    SurfaceProperties *s = state->surface;
    
    /* ambientColor */
    if (s->ambientlen > 0) {
	if (am >= s->ambientlen) {
	    am = s->ambientlen - 1;
	}
	mat->ambientRGBA[0] = s->ambientColor[am].rgb[0];
	mat->ambientRGBA[1] = s->ambientColor[am].rgb[1];
	mat->ambientRGBA[2] = s->ambientColor[am].rgb[2];
	mat->ambientRGBA[3] = 0;
    }
    else {
	mat->ambientRGBA[0] = 0;
	mat->ambientRGBA[1] = 0;
	mat->ambientRGBA[2] = 0;
	mat->ambientRGBA[3] = 0;
    }
    
    /* diffuseColor */
    if (s->diffuselen > 0) {
	if (di >= s->diffuselen) {
	    di = s->diffuselen - 1;
	}
	mat->diffuseRGBA[0] = s->diffuseColor[di].rgb[0];
	mat->diffuseRGBA[1] = s->diffuseColor[di].rgb[1];
	mat->diffuseRGBA[2] = s->diffuseColor[di].rgb[2];
	mat->diffuseRGBA[3] = 0;
    }
    else {
	mat->diffuseRGBA[0] = 0;
	mat->diffuseRGBA[1] = 0;
	mat->diffuseRGBA[2] = 0;
	mat->diffuseRGBA[3] = 0;
    }

    /* specularColor */
    if (s->specularlen > 0) {
	if (sp >= s->specularlen) {
	    sp = s->specularlen - 1;
	}
	mat->specularRGBA[0] = s->specularColor[sp].rgb[0];
	mat->specularRGBA[1] = s->specularColor[sp].rgb[1];
	mat->specularRGBA[2] = s->specularColor[sp].rgb[2];
	mat->specularRGBA[3] = 0;
    }
    else {
	mat->specularRGBA[0] = 0;
	mat->specularRGBA[1] = 0;
	mat->specularRGBA[2] = 0;
	mat->specularRGBA[3] = 0;
    }

    /* emissiveColor */
    if (s->emissivelen > 0) {
	if (em >= s->emissivelen) {
	    em = s->emissivelen - 1;
	}
	mat->emissiveRGBA[0] = s->emissiveColor[em].rgb[0];
	mat->emissiveRGBA[1] = s->emissiveColor[em].rgb[1];
	mat->emissiveRGBA[2] = s->emissiveColor[em].rgb[2];
	mat->emissiveRGBA[3] = 0;
    }
    else {
	mat->emissiveRGBA[0] = 0;
	mat->emissiveRGBA[1] = 0;
	mat->emissiveRGBA[2] = 0;
	mat->emissiveRGBA[3] = 0;
    }
    
    if (s->shininesslen > 0) {
	if (sh >= s->shininesslen) {
	    sh = s->shininesslen - 1;
	}
	mat->shininess = s->shininess[sh];
    }
    else {
	mat->shininess = 0;
    }
    
    /* update indices */
    indices[0]++;
    indices[1]++;
    indices[2]++;
    indices[3]++;
    indices[4]++;
    indices[5]++;
    if ((indices[0] >= s->ambientlen) &&
	(indices[1] >= s->diffuselen) &&
	(indices[2] >= s->specularlen) &&
	(indices[3] >= s->emissivelen) &&
	(indices[4] >= s->shininesslen) &&
	(indices[5] >= s->transparencylen)) {
	indices[0] = indices[1] = indices[2] = indices[3] = indices[4] = indices[5] = 0;
    }
}


/********************************************************************************
 *
 * Vrml_get_normal
 *
 * Get the next normal vector
 *
 *******************************************************************************/

void
Vrml_get_normal(interp, state, indices, scale, normal)
     Tcl_Interp *interp;
     State *state;
     int *indices;
     SFVec3f *scale;
     char *normal;
{
    SFVec3f *vin, vout;
    SurfaceProperties *s = state->surface;

    /* normal vector */
    if (s->normallen > 0) {
	if (indices[0] >= s->normallen) {
	    sprintf(normal, "0.0 0.0 0.0");
	}
	else {
	    vin = s->normal + indices[0]++;
	    vout.v[0] = vin->v[0] * scale->v[0];
	    vout.v[1] = vin->v[1] * scale->v[1];
	    vout.v[2] = vin->v[2] * scale->v[2];
	    Vrml_output_SFVec3f(interp, normal, &vout);
	}
    }
    else {
	sprintf(normal, "0.0 0.0 0.0");
    }
}


/********************************************************************************
 *
 * Vrml_initialize_material_indices
 *
 * Initialize material indices.
 *
 *******************************************************************************/

void
Vrml_initialize_material_indices(state, indices, start)
     State *state;
     int indices[];
     int start;
{
    int r, period;
    SurfaceProperties *s = state->surface;

    period = s->ambientlen;
    if (period < s->diffuselen) period = s->diffuselen;
    if (period < s->specularlen) period = s->specularlen;
    if (period < s->emissivelen) period = s->emissivelen;
    if (period < s->shininesslen) period = s->shininesslen;
    if (period < s->transparencylen) period = s->transparencylen;

    r = start % period;
    if (r >= s->ambientlen) {
	indices[0] = s->ambientlen - 1;
    }
    else {
	indices[0] = r;
    }
    if (r >= s->diffuselen) {
	indices[1] = s->diffuselen - 1;
    }
    else {
	indices[1] = r;
    }
    if (r >= s->specularlen) {
	indices[2] = s->specularlen - 1;
    }
    else {
	indices[2] = r;
    }
    if (r >= s->emissivelen) {
	indices[3] = s->emissivelen - 1;
    }
    else {
	indices[3] = r;
    }
    if (r >= s->shininesslen) {
	indices[4] = s->shininesslen - 1;
    }
    else {
	indices[4] = r;
    }
    if (r >= s->transparencylen) {
	indices[5] = s->transparencylen - 1;
    }
    else {
	indices[5] = r;
    }
}


/********************************************************************************
 *
 * Vrml_initialize_normal_indices
 *
 * Initialize normal indices.
 *
 *******************************************************************************/

void
Vrml_initialize_normal_indices(state, indices, start)
     State *state;
     int *indices;
     int start;
{
    if (state->surface->normallen > 0)
      *indices = start % state->surface->normallen;
}


/********************************************************************************
 *
 * Vrml_new_object
 *
 * cmd create polygon name -parent node -position pos
 *
 ********************************************************************************/

int
Vrml_new_object(interp, state, node, type, pos, fwd, up, argc, argv)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char *type;
     SFVec3f *pos;
     SFVec3f *fwd;
     SFVec3f *up;
     int argc;
     char **argv;
{
    int result;
    char p[100];
    char f[100];
    char u[100];
    int i;
    char **targv;
    Model *model;

    if ((targv = (char **) ckalloc(sizeof(char *) * (12 + argc))) == NULL) {
	Tcl_SetResult(interp, "out of memory", TCL_STATIC);
	return TCL_ERROR;
    }
    for (i = 0; i < argc; i++) {
	targv[12 + i] = argv[i];
    }
    targv[0] = "cmd";
    targv[1] = "create";
    targv[2] = type;
    targv[3] = NULL;
    targv[4] = "-parent";
    targv[5] = NULL;
    targv[6] = "-position";
    targv[7] = NULL;
    targv[8] = "-fwd";
    targv[9] = NULL;
    targv[10] = "-up";
    targv[11] = NULL;

    if (node) {
	targv[3] = node->label;

	if (state->model->next == NULL) {
	    /* current model is the root model */
	    targv[5] = state->prefix;
	}
	else {
	    targv[5] = state->model->node->label;
	}
    }
    else {
	targv[3] = state->prefix;
	targv[5] = "";
    }
    sprintf(p, "%f %f %f", pos->v[0], pos->v[1], pos->v[2]);
    sprintf(f, "%f %f %f", fwd->v[0], fwd->v[1], fwd->v[2]);
    sprintf(u, "%f %f %f", up->v[0], up->v[1], up->v[2]);
    targv[7] = p;
    targv[9] = f;
    targv[11] = u;

    if ((result = SmHandleNewModel(state->cell, 12 + argc, targv, &model)) == TCL_OK) {
	Tcl_ResetResult(interp);
    }
    return result;
}
