/********************************************************************************
 *
 * results.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


typedef struct NameList {
    int			count;
    char		*name;
    struct Node		*node;
    struct NameList	*next;
} NameList;


static char	*empty = "";
static NameList	*namelist;


/********************************************************************************
 *
 * collect_models
 *
 ********************************************************************************/

static int
collect_models(interp, state, node, argv, argc)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char ***argv;
     int argc;
{
    int id;
    char *list;
    NameList *p;

    /* count the number of models */
    if (node->objects) {
	id = collect_models(interp, state, node->objects, argv, argc + 1);
    }
    else {
	id = 1;
	*argv = (char **) ckalloc(sizeof(char *) * (argc + 1));
    }

    /* add model label to result */
    (*argv)[id] = node->label;

    /* insert node into namelist if necessary */
    if (node->name) {
	for (p = namelist; p; p = p->next) {
	    if (!strcmp(p->name, node->name)) {
		node->map = p->node;
		p->node = node;
		p->count++;
		break;
	    }
	}
	if (!p) {
	    p = (NameList *) ckalloc(sizeof(NameList));
	    p->count = 1;
	    p->name = node->name;
	    p->node = node;
	    p->next = namelist;
	    namelist = p;
	    state->nameCount++;
	}
    }

    /* generate list of model names */
    if (argc == 1) {
	(*argv)[0] = state->prefix;
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_infos
 *
 ********************************************************************************/

static int
collect_infos(interp, node, argv, argc)
     Tcl_Interp *interp;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id;
    char *list, *elements[2];
    NodeInfo *info = &node->node.info;

    /* count the number of infos */
    if (node->infos) {
	id = collect_infos(interp, node->infos, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* {name {str ...}} */
    elements[0] = (node->name) ? node->name : empty;
    elements[1] = Tcl_Merge(info->stringlen, info->string);
    (*argv)[id] = Tcl_Merge(2, elements);
    (void) ckfree((void *) elements[1]);

    /* generate list of infos */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    if ((*argv)[i] != empty) (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_scripts
 *
 ********************************************************************************/

static int
collect_scripts(interp, node, argv, argc)
     Tcl_Interp *interp;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id;
    char *list, *elements[2];
    NodeScript *script = &node->node.script;

    /* count the number of scripts */
    if (node->scripts) {
	id = collect_scripts(interp, node->scripts, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* {language script} */
    elements[0] = (script->language) ? script->language : empty;
    elements[1] = (script->script) ? script->script : empty;
    (*argv)[id] = Tcl_Merge(2, elements);

    /* generate list of scripts */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    if ((*argv)[i] != empty) (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_map
 *
 ********************************************************************************/

void
collect_map(interp, state, namelist)
     Tcl_Interp *interp;
     State *state;
     NameList *namelist;
{
    int i, m, len;
    char **argv, **maps;
    Node *node;
    NameList *p, *q;

    len = strlen(state->prefix) + 20;
    maps = (char **) ckalloc(sizeof(char *) * state->nameCount);
    for (m = 0, p = namelist; p; p = q, m++) {
	q = p->next;
	argv = (char **) ckalloc(sizeof(char *) * (p->count + 1));
	argv[0] = p->name;
	i = 0;

	for (i = 1, node = p->node; node; i++, node = node->map) {
	    argv[i] = node->label;
	}
	maps[m] = Tcl_Merge(p->count + 1, argv);
	(void) ckfree((void *) argv);
    }

    Tcl_AppendElement(interp, Tcl_Merge(state->nameCount, maps));

    for (m = 0; m < state->nameCount; m++) {
	(void) ckfree((void *) maps[m]);
    }

    /* reset names */
    for (p = namelist; p; p = q) {
	q = p->next;
	(void) ckfree((void *) p);
    }
}


/********************************************************************************
 *
 * collect_anchors
 *
 ********************************************************************************/

static int
collect_anchors(interp, state, node, argv, argc)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id;
    char *list, *elements[4];
    NodeWWWAnchor *anchor = &node->node.wwwAnchor;

    /* count the number of anchors */
    if (node->anchors) {
	id = collect_anchors(interp, state, node->anchors, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* {label name description map} */

    elements[0] = node->label;
    elements[1] = (anchor->name) ? anchor->name : empty;
    elements[2] = (anchor->description) ? anchor->description : empty;
    elements[3] = (anchor->map == ENUM_NONE) ? "none" : "point";

    (*argv)[id] = Tcl_Merge(4, elements);

    /* generate list of anchors */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_lods
 *
 ********************************************************************************/

static int
collect_lods(interp, state, node, argv, argc)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id, count;
    char *list, *elements[4];
    char *center;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE], c2[TCL_DOUBLE_SPACE];
    char *range, *p;
    char **childargv;
    Node *child;
    NodeLOD *lod = &node->node.LOD;

    /* count the number of lods */
    if (node->lods) {
	id = collect_lods(interp, state, node->lods, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* output center coordinates */
    Tcl_PrintDouble(interp, (double) lod->wcenter.v[0], c0);
    Tcl_PrintDouble(interp, (double) lod->wcenter.v[1], c1);
    Tcl_PrintDouble(interp, (double) lod->wcenter.v[2], c2);
    elements[0] = c0;
    elements[1]=  c1;
    elements[2] = c2;
    center = Tcl_Merge(3, elements);

    /* output range values */
    if (lod->rangelen > 0) {
	range = (char *) ckalloc((TCL_DOUBLE_SPACE + 2) * lod->rangelen);
	Tcl_PrintDouble(interp, (double) lod->range[0], range);
	p = range + strlen(range);
	for (i = 1; i < lod->rangelen; i++) {
	    *p++ = ' ';
	    Tcl_PrintDouble(interp, (double) lod->range[i], p);
	    p = range + strlen(range);
	}
    }
    else {
	range = NULL;
    }

    /* output children labels */
    count = 0;
    for (child = node->child; child; child = child->sibling) {
	count++;
    }
    if (count > 0) {
	childargv = (char **) ckalloc(sizeof(char *) * count);
	for (i = 0, child = node->child; child; i++, child = child->sibling) {
	    childargv[i] = child->label;
	}
    }
    else {
	childargv = NULL;
    }

    /* {parent-label center {range ...} {child-label ...}} */
    elements[0] = (node->parent) ? node->parent->label : state->prefix;
    elements[1] = center;
    elements[2] = (range) ? range : empty;
    elements[3] = Tcl_Merge(count, childargv);

    (*argv)[id] = Tcl_Merge(4, elements);

    if (center) (void) ckfree((void *) center);
    if (range != empty) (void) ckfree((void *) range);
    if (childargv) (void) ckfree((void  *) childargv);
    if (elements[3]) (void) ckfree((void *) elements[3]);

    /* generate list of lods */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_cameras
 *
 ********************************************************************************/

static int
collect_cameras(interp, state, node, argv, argc)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id;
    char *list, *elements[11];
    char vup[TCL_DOUBLE_SPACE * 3 + 10];
    char vpn[TCL_DOUBLE_SPACE * 3 + 10];
    char vrp[TCL_DOUBLE_SPACE * 3 + 10];
    char prp[TCL_DOUBLE_SPACE * 3 + 10];
    char near[TCL_DOUBLE_SPACE];
    char far[TCL_DOUBLE_SPACE];
    char width[TCL_DOUBLE_SPACE];
    char height[TCL_DOUBLE_SPACE];
    char focalDistance[TCL_DOUBLE_SPACE];
    double h;
    SFVec3f focal;

    NodeOrthographicCamera *ortho = &node->node.orthographicCamera;
    NodePerspectiveCamera *persp = &node->node.perspectiveCamera;

    /* count the number of cameras */
    if (node->cameras) {
	id = collect_cameras(interp, state, node->cameras, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* {name type vrp vpn vup prp near far width height} */

    focal.v[0] = focal.v[1] = 0;
    if (node->code == NODE_OrthographicCamera) {
	focal.v[2] = ortho->focalLength;
	Vrml_output_SFVec3f(interp, vrp, &ortho->vrp);
	Vrml_output_SFVec3f(interp, vpn, &ortho->vpn);
	Vrml_output_SFVec3f(interp, vup, &ortho->vup);
	Tcl_PrintDouble(interp, (double) ortho->nearDistance, near);
	Tcl_PrintDouble(interp, (double) ortho->farDistance, far);
	Tcl_PrintDouble(interp, (double) ortho->height * ortho->aspectRatio, width);
	Tcl_PrintDouble(interp, (double) ortho->height, height);
	Tcl_PrintDouble(interp, (double) ortho->focalDistance, focalDistance);
    }
    else {
	focal.v[2] = persp->focalLength;
	Vrml_output_SFVec3f(interp, vrp, &persp->vrp);
	Vrml_output_SFVec3f(interp, vpn, &persp->vpn);
	Vrml_output_SFVec3f(interp, vup, &persp->vup);
	Tcl_PrintDouble(interp, (double) persp->nearDistance, near);
	Tcl_PrintDouble(interp, (double) persp->farDistance, far);
	h = 2 * persp->focalLength * tan((double) persp->heightAngle * 0.5);
	Tcl_PrintDouble(interp, (double) h * persp->aspectRatio, width);
	Tcl_PrintDouble(interp, (double) h, height);
	Tcl_PrintDouble(interp, (double) persp->focalDistance, focalDistance);
    }
    Vrml_output_SFVec3f(interp, prp, &focal);

    elements[0] = (node->name) ? node->name : empty;
    elements[1] = (node->code == NODE_OrthographicCamera) ? "parallel" : "perspective";
    elements[2] = vrp;
    elements[3] = vpn;
    elements[4] = vup;
    elements[5] = prp;
    elements[6] = near;
    elements[7] = far;
    elements[8] = width;
    elements[9] = height;
    elements[10] = focalDistance;

    (*argv)[id] = Tcl_Merge(11, elements);

    /* generate list of cameras */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_lights
 *
 ********************************************************************************/

static int
collect_lights(interp, state, node, argv, argc)
     Tcl_Interp *interp;
     State *state;
     Node *node;
     char ***argv;
     int argc;
{
    int i, id;
    char *list, *elements[10];
    char color[TCL_DOUBLE_SPACE * 3 + 10];
    char r[TCL_DOUBLE_SPACE];
    char g[TCL_DOUBLE_SPACE];
    char b[TCL_DOUBLE_SPACE];
    char direction[TCL_DOUBLE_SPACE * 3 + 10];
    char location[TCL_DOUBLE_SPACE * 4 + 10];
    char dropOffRate[TCL_DOUBLE_SPACE];
    char cutOffAngle[TCL_DOUBLE_SPACE];
    float intensity;
    static char *ambient = "0.0 0.0 0.0";

    NodeDirectionalLight *directional = &node->node.directionalLight;
    NodePointLight *point = &node->node.pointLight;
    NodeSpotLight *spot = &node->node.spotLight;


    /* count the number of lights */
    if (node->lights) {
	id = collect_lights(interp, state, node->lights, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /*
		directional	{type ambient diffuse specular direction}
		point		{type ambient diffuse specular location}
		spot		{type ambient diffuse specular location direction exponent cutoff}
     */

    if (node->code == NODE_DirectionalLight) {

	intensity = directional->intensity;

	Tcl_PrintDouble(interp, (double) directional->color.rgb[0] * intensity, r);
	Tcl_PrintDouble(interp, (double) directional->color.rgb[1] * intensity, g);
	Tcl_PrintDouble(interp, (double) directional->color.rgb[2] * intensity, b);
	sprintf(color, "%s %s %s", r, g, b);


	if (directional->model) {
	    elements[9] = directional->model->label;
	    Vrml_output_SFVec3f(interp, direction, &directional->direction);
	}
	else {
	    elements[9] = "";
	    Vrml_output_SFVec3f(interp, direction, &directional->wdirection);
	}
	(void) strcat(direction, " 0.0");
	elements[0] = "directional";
	elements[1] = ambient;
	elements[2] = color;
	elements[3] = color;
	elements[4] = direction;
	elements[5] = "0 0 -1";
	elements[6] = "0";
	elements[7] = "180";
	elements[8] = "1 0 0";
    }
    else if (node->code == NODE_PointLight) {
	intensity = point->intensity;

	Tcl_PrintDouble(interp, (double) point->color.rgb[0] * intensity, r);
	Tcl_PrintDouble(interp, (double) point->color.rgb[1] * intensity, g);
	Tcl_PrintDouble(interp, (double) point->color.rgb[2] * intensity, b);
	sprintf(color, "%s %s %s", r, g, b);

	if (point->model) {
	    elements[9] = point->model->label;
	    Vrml_output_SFVec3f(interp, location, &point->location);
	}
	else {
	    elements[9] = "";
	    Vrml_output_SFVec3f(interp, location, &point->wlocation);
	}

	(void) strcat(location, " 1.0");
	elements[0] = "point";
	elements[1] = ambient;
	elements[2] = color;
	elements[3] = color;
	elements[4] = location;
	elements[5] = "0 0 -1";
	elements[6] = "0";
	elements[7] = "180";
	elements[8] = "1 0 0";
    }
    else {
	intensity = spot->intensity;

	Tcl_PrintDouble(interp, (double) spot->color.rgb[0] * intensity, r);
	Tcl_PrintDouble(interp, (double) spot->color.rgb[1] * intensity, g);
	Tcl_PrintDouble(interp, (double) spot->color.rgb[2] * intensity, b);
	sprintf(color, "%s %s %s", r, g, b);

	if (spot->model) {
	    elements[9] = spot->model->label;
	    Vrml_output_SFVec3f(interp, direction, &spot->direction);
	    Vrml_output_SFVec3f(interp, location, &spot->location);
	}
	else {
	    elements[9] = "";
	    Vrml_output_SFVec3f(interp, direction, &spot->wdirection);
	    Vrml_output_SFVec3f(interp, location, &spot->wlocation);
	}

	(void) strcat(location, " 1.0");
	Tcl_PrintDouble(interp, (double) spot->dropOffRate, dropOffRate);
	Tcl_PrintDouble(interp, (double) spot->cutOffAngle * 180.0 / 3.14159265, cutOffAngle);
	elements[0] = "spot";
	elements[1] = ambient;
	elements[2] = color;
	elements[3] = color;
	elements[4] = location;
	elements[5] = direction;
	elements[6] = dropOffRate;
	elements[7] = cutOffAngle;
	elements[8] = "1 0 0";
    }

    (*argv)[id] = Tcl_Merge(10, elements);

    /* generate list of lights */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	for (i = 0; i <= id; i++) {
	    (void) ckfree((void *) (*argv)[i]);
	}
	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * Vrml_collect_results
 *
 ********************************************************************************/

void
Vrml_collect_results(interp, state)
     Tcl_Interp *interp;
     State *state;
{
    char **argv;

    /* results that need to be collected:

       - model labels:	{label ...}
       - info strings:	{{name {string ...}} ...}
       - name maps:	{{name {label ...}} ...}
       - anchors:	{{label name description map} ...}
       - lods:		{{parent-label center {range ...} {child-label ...}} ...}
       - cameras	{{name type vrp vpn vup prp near far width height} ...}
       - lights:
			directional	{type color intensity direction}
			point		{type color intensity location}
			spot		{type color intensity location direction dropOffRate CutOffAngle}
       - scripts strings {{language script} ...}

       */

    Tcl_ResetResult(interp);

    /* reset names */
    namelist = NULL;
    state->nameCount = 0;

    if (state->objects) {
	(void) collect_models(interp, state, state->objects, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "{}");
    }

    if (state->infos) {
	(void) collect_infos(interp, state->infos, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (namelist) {
	collect_map(interp, state, namelist);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (state->anchors) {
	(void) collect_anchors(interp, state, state->anchors, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (state->lods) {
	(void) collect_lods(interp, state, state->lods, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (state->cameras) {
	(void) collect_cameras(interp, state, state->cameras, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (state->lights) {
	(void) collect_lights(interp, state, state->lights, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (state->scripts) {
	(void) collect_scripts(interp, state->scripts, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }
}


/********************************************************************************
 *
 * collect_inlines
 *
 ********************************************************************************/

static int
collect_inlines(interp, node, argv, argc)
     Tcl_Interp *interp;
     Node *node;
     char ***argv;
     int argc;
{
    int id;
    char *list;

    /* count the number of infos */
    if (node->inlinelist) {
	id = collect_inlines(interp, node->inlinelist, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* URL */
    (*argv)[id] = node->node.wwwInline.name;
    
    /* generate list of node names */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * collect_textures
 *
 ********************************************************************************/

static int
collect_textures(interp, node, argv, argc)
     Tcl_Interp *interp;
     Node *node;
     char ***argv;
     int argc;
{
    int id;
    char *list;
    NodeTexture2 *texture = &node->node.texture2;

    /* count the number of infos */
    if (node->texturelist) {
	id = collect_textures(interp, node->texturelist, argv, argc + 1);
    }
    else {
	id = 0;
	*argv = (char **) ckalloc(sizeof(char *) * argc);
    }

    /* URL */
    (*argv)[id] = texture->filename;
    
    /* generate list of node names */
    if (argc == 1) {
	list = Tcl_Merge(id + 1, *argv);
	Tcl_AppendElement(interp, list);

	(void) ckfree((void *) list);
	(void) ckfree((void *) *argv);
    }
    return (id + 1);
}


/********************************************************************************
 *
 * Vrml_collect_URLs
 *
 ********************************************************************************/

void
Vrml_collect_URLs(interp, inlines, textures)
     Tcl_Interp *interp;
     Node *inlines;
     Node *textures;
{
    char **argv;

    if (inlines) {
	(void) collect_inlines(interp, inlines, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }

    if (textures) {
	(void) collect_textures(interp, textures, &argv, 1);
    }
    else {
	Tcl_AppendElement(interp, "");
    }
}
