/* 
 * smcell.c --
 *
 *	This file implements cells.
 *
 */

#include "copyright.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <tk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "smInt.h"


/*
 * Prototypes for procedures defined externally:
 */

int		SmNormalizeVector3D _ANSI_ARGS_ ((DOUBLE vin[], DOUBLE vout[], double epsilon));

void		SmCrossVector3D _ANSI_ARGS_ ((DOUBLE a[], DOUBLE b[], DOUBLE c[]));

void		SmDisplayViewport _ANSI_ARGS_ ((Tk_Canvas canvas,
						Tk_Item *itemPtr, Display *dpy, Drawable dst,
						int x, int y, int width, int height));

void            SmViewportUpdate _ANSI_ARGS_ ((ClientData clientData));

int		SmViewportCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp,
					    int argc, char *argv[]));

int		SmTextureCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp,
					   int argc, char *argv[]));

int		SmInfoCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp,
					int argc, char *argv[]));

void		SmSetClientData _ANSI_ARGS_ ((ClientData clientData));

Texture *	SmGetTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, char *label));

void		SmFreeTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, Texture *texture));

int		SmHandleNewModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));


/*
 * Prototypes for procedures defined in this file that are exported:
 */

int		Tksm_Init _ANSI_ARGS_ ((Tcl_Interp *interp));

void		SmInsertPortIntoCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

void		SmRemovePortFromCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

void		SmRemoveModelFromParent _ANSI_ARGS_ ((Model *model));

Cell		*SmFindCell _ANSI_ARGS_ ((Tcl_Interp *interp, char *label));

Model		*SmFindModel _ANSI_ARGS_ ((Cell *cell, char *label));

void		SmRedrawCell _ANSI_ARGS_ ((Cell *cell));

int		SmComputeVertexNormals _ANSI_ARGS_ ((Model *model, int reset));

CellHeader	*SmGetCellHeader _ANSI_ARGS_ ((Tcl_Interp *interp));

int		SmParseColorCoefficients _ANSI_ARGS_ ((Tcl_Interp *interp, char *value,
						       double *r, double *g, double *b));

int		SmGetBoundingBox _ANSI_ARGS_ ((Model *model, DOUBLE *x0, DOUBLE *y0, DOUBLE *z0,
					       DOUBLE *x1, DOUBLE *y1, DOUBLE *z1, int recursive));

int		SmComputeBoundingBox _ANSI_ARGS_ ((Model *model));


/*
 * Prototypes for procedures defined in this file that are static:
 */

static int		HandleCommand _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp, int argc, char **argv));

static int		HandleCellCommand _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp, int argc, char **argv));

static void		DestroyCell _ANSI_ARGS_ ((ClientData clientData));

static void		DestroyCells _ANSI_ARGS_ ((ClientData clientData));

static int		NewPolyModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		NewLineModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		NewPointModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		NewCylinderModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		NewDiskModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		NewSphereModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));

static int		ConfigureModel _ANSI_ARGS_ ((Cell *cell, Model *model, Tk_ConfigSpec specs[],
						     int argc, char **argv, int flags));

static void		DeleteModel _ANSI_ARGS_ ((Model *model));

static void		FreeModel _ANSI_ARGS_ ((Model *model));

static int		ComputeModelTransform _ANSI_ARGS_ ((Model *model, double epsilon));

static int		ParseVertexNormals _ANSI_ARGS_ ((Tcl_Interp *interp, Model *model, char *value));

static int		ParseVertexMaterials _ANSI_ARGS_ ((Tcl_Interp *interp, Model *model, char *value));

static void		InsertCell _ANSI_ARGS_ ((Tcl_Interp *interp, Cell *cell));

static void		ComputeSurfaceNormal _ANSI_ARGS_ ((Vertex *v, Surface *s, DOUBLE n[3], double epsilon));

static int		GetChildren _ANSI_ARGS_ ((Model *model));

static int		ProcessMaterials _ANSI_ARGS_ ((Model *model));

static int		ProcessTextures _ANSI_ARGS_ ((Model *model));

static int		ParseTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header,
						   Model *model, char *entry));

static int		ProcessWidths _ANSI_ARGS_ ((Model *model));

static int		ParseWidth _ANSI_ARGS_ ((Tcl_Interp *interp, Model *model, char *entry));

static int		ProcessStipples _ANSI_ARGS_ ((Model *model));

static int		ParseStipple _ANSI_ARGS_ ((Tcl_Interp *interp, Model *model, char *entry));

static int		ProcessSizes _ANSI_ARGS_ ((Model *model));

static void		FastNormalizeVector3D _ANSI_ARGS_ ((DOUBLE vin[], DOUBLE vout[]));

static int		ComputeSurfaceBoundingBox _ANSI_ARGS_ ((Surface *s, Vertex *v, DOUBLE *x0, DOUBLE *y0, DOUBLE *z0,
								DOUBLE *x1, DOUBLE *y1, DOUBLE *z1));

static int		SetModelVertices _ANSI_ARGS_ ((Tcl_Interp *interp, Model *model, char *value, int mode));

static int		ColorValueByName _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp,
						       int argc, char *argv[]));


/*
 * Custom options for cells
 */


/*
 * Information used for parsing cell configuration specs:
 */

static Tk_ConfigSpec cellConfigSpecs[] = {

    {TK_CONFIG_DOUBLE, "-epsilon", "epsilon", "Epsilon",
       "1e-5", Tk_Offset(Cell, epsilon), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};


/*
 * Information used for parsing model configuration specs:
 */

extern Tk_ConfigSpec	modelPolygonConfigSpecs[];
extern Tk_ConfigSpec	modelPointConfigSpecs[];
extern Tk_ConfigSpec	modelLineConfigSpecs[];
extern Tk_ConfigSpec	modelCylinderConfigSpecs[];
extern Tk_ConfigSpec	modelDiskConfigSpecs[];
extern Tk_ConfigSpec	modelSphereConfigSpecs[];

extern Tk_ItemType	tkViewportType;



/*
 *--------------------------------------------------------------
 *
 * Tksm_Init:
 *
 * initialize the Cell package.
 *
 *--------------------------------------------------------------
 */

int
Tksm_Init(Tcl_Interp *interp)
{
    CellHeader *header;

    if ((header = (CellHeader *) ckalloc(sizeof(CellHeader)))) {
	header->cells = NULL;
	header->textures = NULL;
	header->id = 0;
	Tcl_InitHashTable(&header->hash, TCL_STRING_KEYS);
	Tcl_InitHashTable(&header->glxGC, TCL_ONE_WORD_KEYS);
	Tcl_CreateCommand(interp, "cell", HandleCommand, (ClientData) header, (Tcl_CmdDeleteProc *) DestroyCells);
	Tcl_CreateCommand(interp, "viewport", SmViewportCmd, (ClientData) Tk_MainWindow(interp), (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "texture", SmTextureCmd, (ClientData) header, (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "sminfo", SmInfoCmd, (ClientData) Tk_Display(Tk_MainWindow(interp)), (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "color_value_by_name", ColorValueByName, (ClientData) Tk_MainWindow(interp), (Tcl_CmdDeleteProc *) NULL);

	Tk_CreateItemType(&tkViewportType);

#ifdef USE_TCL75
	return Tcl_PkgProvide(interp, "tksm", SM_VERSION);
#else
	return TCL_OK;
#endif
    }
    else {
	Tcl_AppendResult(interp, "Cell package initialization failed: ckalloc returns NULL", (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmFindCell:
 *
 * locate a cell by its label.
 *
 *--------------------------------------------------------------
 */

Cell *
SmFindCell(interp, label)
     Tcl_Interp *interp;
     char *label;
{
    CellHeader *header;
    Tcl_HashEntry *e;

    if ((header = SmGetCellHeader(interp)) == NULL) return NULL;
    if ((e = Tcl_FindHashEntry(&header->hash, label)) == NULL) return NULL;
    return (Cell *) Tcl_GetHashValue(e);
}

/*
 *--------------------------------------------------------------
 *
 * SmInsertPortIntoCell:
 *
 * insert a viewport into the cell viewport list.
 *
 *--------------------------------------------------------------
 */

void
SmInsertPortIntoCell(cell, port)
     Cell *cell;
     PortItem *port;
{
    port->next = cell->ports;
    cell->ports = port;
}

/*
 *--------------------------------------------------------------
 *
 * SmRemovePortFromCell:
 *
 * remove a viewport from the cell viewport list.
 *
 *--------------------------------------------------------------
 */

void
SmRemovePortFromCell(cell, port)
     Cell *cell;
     PortItem *port;
{
    PortItem *p, *q;

    for (p = cell->ports, q = NULL; p; q = p, p = p->next) {
	if (p == port) {
	    if (q) {
		q->next = p->next;
	    }
	    else {
		cell->ports = p->next;
	    }
	    return;
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmGetCellHeader:
 *
 * return cell header pointer.
 *
 *--------------------------------------------------------------
 */

CellHeader *
SmGetCellHeader(interp)
     Tcl_Interp *interp;
{
    Tcl_CmdInfo info;

    return Tcl_GetCommandInfo(interp, "cell", &info) ? (CellHeader *) info.clientData : NULL;
}

/*
 *--------------------------------------------------------------
 *
 * InsertCell:
 *
 * insert a new cell into list of cells.
 *
 *--------------------------------------------------------------
 */

static void
InsertCell(interp, cell)
     Tcl_Interp *interp;
     Cell *cell;
{
    Tcl_CmdInfo info;
    CellHeader *header;

    if (!(Tcl_GetCommandInfo(interp, "cell", &info))) {
	return;
    }

    header = (CellHeader *) info.clientData;
    if (header->cells) {
	cell->next = header->cells;
    }
    else {
	cell->next = NULL;
    }
    header->cells = cell;
}

/*
 *--------------------------------------------------------------
 *
 * FreeModel:
 *
 * release memory occupied by a model.
 *
 *--------------------------------------------------------------
 */

static void
FreeModel(model)
     Model *model;
{
    int i;
    Surface *s;
    Model *ch, *next;
    PortItem *p;
    LightSource *l;

    /* remove tracking light sources associated with this model */
    for (p = model->cell->ports; p; p = p->next) {
	for (i = 0, l = p->lights; i < 8; i++, l++) {
	    if ((l->on) && (model == l->model)) {
		l->on = 0;
	    }
	}
    }

    SmRemoveModelFromParent(model);

    if (model->label) (void) ckfree((void *) model->label);
    if (model->v) (void) ckfree((void *) model->v);
    if (model->e) Tcl_DeleteHashEntry(model->e);
    if (model->color != NULL) Tk_FreeColor(model->color);
    if (model->bfcolor != NULL) Tk_FreeColor(model->bfcolor);
    if (model->quadric != NULL) gluDeleteQuadric(model->quadric);

    if (model->normals) (void) ckfree((void *) model->normals);
    if (model->materials) (void) ckfree((void *) model->materials);
    if (model->textures) (void) ckfree((void *) model->textures);
    if (model->widths) (void) ckfree((void *) model->widths);
    if (model->stipples) (void) ckfree((void *) model->stipples);
    if (model->sizes) (void) ckfree((void *) model->sizes);
    if (model->pointSizes) (void) ckfree((void *) model->pointSizes);

    if (model->ns > 0) {
	for (s = model->s, i = 0; i < (const) model->ns; i++) {
	    if (s[i].index) (void) ckfree((void *) s[i].index);
	    if (s[i].lvn) (void) ckfree((void *) s[i].lvn);
	    if (s[i].texcoords) (void) ckfree((void *) s[i].texcoords);
	    if (s[i].materials) (void) ckfree((void *) s[i].materials);
	    if (s[i].texture) {
		Tcl_Interp *interp = model->cell->interp;
		SmFreeTexture(interp, SmGetCellHeader(interp), s[i].texture);
	    }
	}
	(void) ckfree((void *) model->s);
    }

    for (ch = model->child; ch; ch = next) {
	next = ch->sibling;
	DeleteModel(ch);
    }
    (void) ckfree((void *) model);
}

/*
 *--------------------------------------------------------------
 *
 * DeleteModel:
 *
 * delete a model.
 *
 *--------------------------------------------------------------
 */

static void
DeleteModel(model)
     Model *model;
{
    Model *p, *q;

    for (p = model->cell->models, q = NULL; p; q = p, p = p->next) {
	if (p == model) {
	    if (q) {
		q->next = p->next;
	    }
	    else {
		model->cell->models = p->next;
	    }
	    FreeModel(model);
	    return;
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmFindModel:
 *
 * locate a model within a cell by its label.
 *
 *--------------------------------------------------------------
 */

Model *
SmFindModel(cell, label)
     Cell *cell;
     char *label;
{
    Tcl_HashEntry *e;

    if ((e = Tcl_FindHashEntry(&cell->hash, label)) == NULL) {
	return NULL;
    }
    return (Model *) Tcl_GetHashValue(e);
}

/*
 *--------------------------------------------------------------
 *
 * SmRemoveModelFromParent:
 *
 * dissociate a model from its parent model.
 *
 *--------------------------------------------------------------
 */

void
SmRemoveModelFromParent(model)
     Model *model;
{
    Model *p, *sib;

    if ((p = model->parent)) {
	if (p->child == model) {
	    p->child = model->sibling;
	}
	else {
	    for (sib = p->child; sib; sib = sib->sibling) {
		if (sib->sibling == model) {
		    sib->sibling = model->sibling;
		    return;
		}
	    }
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmRedrawCell:
 *
 * redraw all viewports associated with the cell.
 *
 *--------------------------------------------------------------
 */

void
SmRedrawCell(cell)
     Cell *cell;
{
    PortItem *p;

    for (p = cell->ports; p; p = p->next) {
	p->redraw = 1;
	if (p->canvas) {
	    Tk_CanvasEventuallyRedraw(p->canvas, ((Tk_Item *) p)->x1, ((Tk_Item *) p)->y1, ((Tk_Item *) p)->x2, ((Tk_Item *) p)->y2);
	}
	else if (p->async && !p->update) {
	    p->update = 1;
	    Tk_DoWhenIdle(SmViewportUpdate, (ClientData) p);
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * HandleCommand:
 *
 * handle cell commands:
 *
 *	cell create <label>
 *	cell list
 *	cell types
 *
 *--------------------------------------------------------------
 */

static int
HandleCommand(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    char c;
    int new, length;
    Cell *cell;
    CellHeader *header;
    Tcl_HashEntry *e;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " command ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    c = *argv[1];
    length = strlen(argv[1]);
    header = SmGetCellHeader(interp);

    if ((c == 'c') && (strncmp(argv[1], "create", length) == 0) && (length >= 2)) {

	/* command 'create' */
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " create label\"", (char *) NULL);
	    return TCL_ERROR;
	}

	e = Tcl_CreateHashEntry(&header->hash, argv[2], &new);

	if (!new) {
	    Tcl_AppendResult(interp, "a cell with label \"", argv[2], "\" already exists.", (char *) NULL);
	    return TCL_ERROR;
	}

	if ((cell = (Cell *) ckalloc(sizeof(Cell))) == NULL) {
	    Tcl_DeleteHashEntry(e);
	    Tcl_AppendResult(interp, "failed to allocate memory for new cell.", (char *) NULL);
	    return TCL_ERROR;
	}

	if (Tk_ConfigureWidget(interp, Tk_MainWindow(interp), cellConfigSpecs, argc - 3, argv + 3,
			       (char *) cell, 0) != TCL_OK) {
	    Tcl_DeleteHashEntry(e);
	    (void) ckfree((void *) cell);
	    return TCL_ERROR;
	}

	if ((cell->label = (char *) ckalloc(strlen(argv[2]) + 10)) == NULL) {
	    Tcl_DeleteHashEntry(e);
	    (void) ckfree((void *) cell);
	    Tcl_AppendResult(interp, "failed to allocate memory for new cell (label).", (char *) NULL);
	    return TCL_ERROR;
	}

	(void) strcpy(cell->label, argv[2]);
	cell->models = NULL;
	cell->ports = NULL;
	cell->interp = interp;
	InsertCell(interp, cell);

	Tcl_InitHashTable(&cell->hash, TCL_STRING_KEYS);
	Tcl_SetHashValue(e, (ClientData) cell);
	Tcl_CreateCommand(interp, argv[2], HandleCellCommand, (ClientData) cell, (Tcl_CmdDeleteProc *) DestroyCell);
    }
    else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0) && (length >= 2)) {

	/* command 'list' */
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " list\"", (char *) NULL);
	    return TCL_ERROR;
	}

	for (cell = header->cells; cell; cell = cell->next) {
	    Tcl_AppendElement(interp, cell->label);
	}
    }
    else if ((c == 't') && (strncmp(argv[1], "types", length) == 0) && (length >= 2)) {

	/* command 'types' */

	sprintf(interp->result, "polygon cylinder disk sphere");
    }
    else {
	Tcl_AppendResult(interp, "bad command \"", argv[1], "\": must be create, list or types.",
			 (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * HandleCellCommand:
 *
 * handle commands associated with a cell:
 *
 *	<cmd> bbox <model>
 *	<cmd> cget option
 *	<cmd> children <model>
 *	<cmd> clip <origx> <origy> <origz> <dimx> <dimy> <dimz>
 *	<cmd> configure [options...]
 *	<cmd> convert <model> <vertex>
 *	<cmd> create <type> <model> [options...]
 *	<cmd> delete <model>
 *	<cmd> destroy
 *	<cmd> exists <model>
 *	<cmd> extent <model>
 *	<cmd> list
 *	<cmd> modelconfigure <model> [options...]
 *	<cmd> modelcget <model> option
 *	<cmd> type <model>
 *	<cmd> setv <model> mode <list of {index vertex}>
 *
 *--------------------------------------------------------------
 */

static int
HandleCellCommand(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    int nitems, length;
    char c;
    char **items;
    Model *model;
    Tk_Window tkwin;
    CellHeader *header;
    Tk_ConfigSpec *specs;
    Tcl_HashEntry *e;

    Cell *cell = (Cell *) clientData;

    tkwin = Tk_MainWindow(interp);
    header = SmGetCellHeader(interp);

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " command ?args?\"", (char *) NULL);
	return TCL_ERROR;
    }

    c = *argv[1];
    length = strlen(argv[1]);

    if ((c == 'b') && (strncmp(argv[1], "bbox", length) == 0) && (length >= 2)) {

	/* command 'bbox' */

	DOUBLE x0, y0, z0, x1, y1, z1;
	char bufferx0[TCL_DOUBLE_SPACE];
	char buffery0[TCL_DOUBLE_SPACE];
	char bufferz0[TCL_DOUBLE_SPACE];
	char bufferx1[TCL_DOUBLE_SPACE];
	char buffery1[TCL_DOUBLE_SPACE];
	char bufferz1[TCL_DOUBLE_SPACE];

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " bbox model\"",
			   (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (!SmGetBoundingBox(model, &x0, &y0, &z0, &x1, &y1, &z1, 0)) {

	    /* model is not empty */

	    Tcl_PrintDouble(interp, (double) x0, bufferx0);
	    Tcl_PrintDouble(interp, (double) y0, buffery0);
	    Tcl_PrintDouble(interp, (double) z0, bufferz0);
	    Tcl_PrintDouble(interp, (double) x1, bufferx1);
	    Tcl_PrintDouble(interp, (double) y1, buffery1);
	    Tcl_PrintDouble(interp, (double) z1, bufferz1);

	    Tcl_AppendElement(interp, bufferx0);
	    Tcl_AppendElement(interp, buffery0);
	    Tcl_AppendElement(interp, bufferz0);
	    Tcl_AppendElement(interp, bufferx1);
	    Tcl_AppendElement(interp, buffery1);
	    Tcl_AppendElement(interp, bufferz1);
	}
    }
    else if ((c == 'e') && (strncmp(argv[1], "exists", length) == 0) && (length >= 2)) {

	/* command 'exists' */

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " exists model\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    *interp->result = '0';
	}
	else {
	    *interp->result = '1';
	}
	*(interp->result + 1) = '\0';
	return TCL_OK;
    }
    else if ((c == 'c') && (strncmp(argv[1], "clip", length) == 0) && (length >= 2)) {

	/* command 'clip' */

	double dimx, dimy, dimz;
	int visible, redraw;
	int roomx, roomy, roomz;
	int origx, origy, origz;

	if (argc != 8) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " clip room-x room-y room-z dimension-x dimension-y dimension-z",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if ((Tcl_GetInt(interp, argv[2], &origx) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[3], &origy) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &origz) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[5], &dimx) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[6], &dimy) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[7], &dimz) != TCL_OK)) {
	    return TCL_ERROR;
	}

	redraw = 0;
	for (model = cell->models; model; model = model->next) {
	    if (model->parent) continue;
	    if (model->invalid) continue;

	    /* compute room indices */

	    if (dimx == 0) {
		roomx = 0;
	    }
	    else {
		roomx = (int) floor(model->pos[0] / dimx + 0.5);
	    }

	    if (dimy == 0) {
		roomy = 0;
	    }
	    else {
		roomy = (int) floor(model->pos[1] / dimy + 0.5);
	    }

	    if (dimz == 0) {
		roomz = 0;
	    }
	    else {
		roomz = (int) floor(model->pos[2] / dimz + 0.5);
	    }

	    visible = model->visible;
	    if ((model->visible = ((abs(roomx - origx) < 2) && (abs(roomy - origy) < 2) && (abs(roomz - origz) < 2)))) {
		Tcl_AppendElement(interp, model->label);
	    }
	    if (visible != model->visible) {
		redraw = 1;
	    }
	}
	if (redraw) {
	    SmRedrawCell(cell);
	}
	return TCL_OK;
    }
    else if ((c == 'c') && (strncmp(argv[1], "convert", length) == 0) && (length >= 2)) {

	/* command 'convert' */

	double x, y, z;
	DOUBLE *modelT;
	char bufferx[TCL_DOUBLE_SPACE];
	char buffery[TCL_DOUBLE_SPACE];
	char bufferz[TCL_DOUBLE_SPACE];

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " convert model vertex\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if (Tcl_SplitList(interp, argv[3], &nitems, &items) != TCL_OK) {
	    return TCL_ERROR;
	}

	if (nitems != 3) {
	    Tcl_AppendResult(interp, "wrong # elements in vertex \"", argv[3], "\": should be {x y z}", (char *) NULL);
	    (void) ckfree((void *) items);
	    return TCL_ERROR;
	}

	if ((Tcl_GetDouble(interp, items[0], &x) != TCL_OK) ||
	    (Tcl_GetDouble(interp, items[1], &y) != TCL_OK) ||
	    (Tcl_GetDouble(interp, items[2], &z) != TCL_OK)) {
	    (void) ckfree((void *) items);
	    return TCL_ERROR;
	}

	modelT = model->modelT;
	Tcl_PrintDouble(interp, x * modelT[0] + y * modelT[4] + z * modelT[8], bufferx);
	Tcl_PrintDouble(interp, x * modelT[1] + y * modelT[5] + z * modelT[9], buffery);
	Tcl_PrintDouble(interp, x * modelT[2] + y * modelT[6] + z * modelT[10], bufferz);

	Tcl_AppendElement(interp, bufferx);
	Tcl_AppendElement(interp, buffery);
	Tcl_AppendElement(interp, bufferz);
	return TCL_OK;
    }
    else if ((c == 'e') && (strncmp(argv[1], "extent", length) == 0) && (length >= 2)) {

	/* command 'extent' */

	DOUBLE x0, y0, z0, x1, y1, z1;
	char bufferx0[TCL_DOUBLE_SPACE];
	char buffery0[TCL_DOUBLE_SPACE];
	char bufferz0[TCL_DOUBLE_SPACE];
	char bufferx1[TCL_DOUBLE_SPACE];
	char buffery1[TCL_DOUBLE_SPACE];
	char bufferz1[TCL_DOUBLE_SPACE];

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " extent model\"",
			   (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if (!SmGetBoundingBox(model, &x0, &y0, &z0, &x1, &y1, &z1, 1)) {
	    Tcl_PrintDouble(interp, (double) x0, bufferx0);
	    Tcl_PrintDouble(interp, (double) y0, buffery0);
	    Tcl_PrintDouble(interp, (double) z0, bufferz0);
	    Tcl_PrintDouble(interp, (double) x1, bufferx1);
	    Tcl_PrintDouble(interp, (double) y1, buffery1);
	    Tcl_PrintDouble(interp, (double) z1, bufferz1);

	    Tcl_AppendElement(interp, bufferx0);
	    Tcl_AppendElement(interp, buffery0);
	    Tcl_AppendElement(interp, bufferz0);
	    Tcl_AppendElement(interp, bufferx1);
	    Tcl_AppendElement(interp, buffery1);
	    Tcl_AppendElement(interp, bufferz1);
	}
    }
    else if ((c == 'm') && (strncmp(argv[1], "modelconfigure", length) == 0) && (length >= 2)) {

	int visible;

	/* command 'modelconfigure' */
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " modelconfigure model ?options?\"",
			   (char *) NULL);
	    return TCL_ERROR;
	}
	    
	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (model->type == SM_TYPE_CYLINDER) {
	    specs = modelCylinderConfigSpecs;
	}
	else if (model->type == SM_TYPE_DISK) {
	    specs = modelDiskConfigSpecs;
	}
	else if (model->type == SM_TYPE_SPHERE) {
	    specs = modelSphereConfigSpecs;
	}
	else if (model->type == SM_TYPE_POLYGON) {
	    specs = modelPolygonConfigSpecs;
	}
	else if (model->type == SM_TYPE_LINE) {
	    specs = modelLineConfigSpecs;
	}
	else if (model->type == SM_TYPE_POINT) {
	    specs = modelPointConfigSpecs;
	}

	switch(argc) {
	  case 3:
	    return Tk_ConfigureInfo(interp, tkwin, specs, (char *) model, NULL, TK_CONFIG_ARGV_ONLY);
	
	  case 4:
	    return Tk_ConfigureInfo(interp, tkwin, specs, (char *) model, argv[3], TK_CONFIG_ARGV_ONLY);

	  default:
	    visible = model->visible;
	    if (ConfigureModel(cell, model, specs, argc-3, argv+3, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (visible || model->visible) {
		SmRedrawCell(cell);
	    }
	}
    }
    else if ((c == 'm') && (strncmp(argv[1], "modelcget", length) == 0) && (length >= 2)) {

	/* command 'modelcget' */
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " modelcget model option\"", (char *) NULL);
	    return TCL_ERROR;
	}
	    
	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (model->type == SM_TYPE_CYLINDER) {
	    specs = modelCylinderConfigSpecs;
	}
	else if (model->type == SM_TYPE_DISK) {
	    specs = modelDiskConfigSpecs;
	}
	else if (model->type == SM_TYPE_SPHERE) {
	    specs = modelSphereConfigSpecs;
	}
	else if (model->type == SM_TYPE_POLYGON) {
	    specs = modelPolygonConfigSpecs;
	}
	else if (model->type == SM_TYPE_LINE) {
	    specs = modelLineConfigSpecs;
	}
	else if (model->type == SM_TYPE_POINT) {
	    specs = modelPointConfigSpecs;
	}

	return Tk_ConfigureValue(interp, tkwin, specs, (char *) model, argv[3], 0);
    }
    else if ((c == 'c') && (strncmp(argv[1], "children", length) == 0) && (length >= 2)) {

	/* command 'children' */
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " children model\"", (char *) NULL);
	    return TCL_ERROR;
	}
	    
	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	return GetChildren(model);
    }
    else if ((c == 'c') && (strncmp(argv[1], "create", length) == 0) && (length >= 2)) {
	
	/* command 'create' */
	if  (argc < 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " create type model ?options?\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (SmHandleNewModel(cell, argc, argv, &model) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (model->visible) {
	    SmRedrawCell(cell);
	}
    }
    else if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0) && (length >= 2)) {

	/* command 'cget' */
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " cget option\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Tk_ConfigureInfo(interp, Tk_MainWindow(interp), cellConfigSpecs, (char *) cell,
				argv[3], TK_CONFIG_ARGV_ONLY);
    }
    else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0) && (length >= 2)) {

	Tk_Window tkwin = Tk_MainWindow(interp);

	/* command 'configure' */
	if (argc < 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " configure ?options?\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	switch(argc) {

	  case 2:
	    return Tk_ConfigureInfo(interp, tkwin, cellConfigSpecs, (char *) cell, NULL, TK_CONFIG_ARGV_ONLY);

	  case 3:
	    return Tk_ConfigureInfo(interp, tkwin, cellConfigSpecs, (char *) cell, argv[3], TK_CONFIG_ARGV_ONLY);

	  default:
	    if (Tk_ConfigureWidget(interp, tkwin, cellConfigSpecs, argc - 2, argv + 2, (char *) cell,
				   TK_CONFIG_ARGV_ONLY) != TCL_OK) {
		return TCL_ERROR;
	    }

	    for (model = cell->models; model; model = model->next) {
		model->invalid = (ComputeModelTransform(model, cell->epsilon) == TCL_ERROR);
	    }
	    SmRedrawCell(cell);
	    return TCL_OK;
	}
    }
    else if ((c == 'd') && (strncmp(argv[1], "delete", length) == 0) && (length >= 2)) {

	int visible;
	Model *p, *q;

	/* command 'delete' */
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " delete model\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (p = cell->models, q = NULL; p; q = p, p = p->next) {
	    if (!strcmp(p->label, argv[2])) {
		visible = p->visible;
		DeleteModel(p);
		if (visible) {
		    SmRedrawCell(cell);
		}
		return TCL_OK;
	    }
	}
	Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".", (char *) NULL);
	return TCL_ERROR;
    }
    else if ((c == 's') && (strncmp(argv[1], "setv", length) == 0) && (length >= 2)) {

	int mode;

	/* command 'setv' */
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " setv model mode vertices\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	if (!strcmp(argv[3], "offset")) {
	    mode = 0;
	}
	else if (!strcmp(argv[3], "absolute")) {
	    mode = 1;
	}
	else {
	    Tcl_AppendResult(interp, "bad mode \"", argv[3], "\": must be absolute or offset",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (model->visible) {
	    SmRedrawCell(model->cell);
	}
	return SetModelVertices(interp, model, argv[4], mode);
    }
    else if ((c == 't') && (strncmp(argv[1], "type", length) == 0) && (length >= 2)) {

	/* command 'type' */
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " type model\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if ((model = SmFindModel(cell, argv[2])) == NULL) {
	    Tcl_AppendResult(interp, "model \"", argv[2], "\" not found in cell \"", cell->label, "\".",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	switch(model->type) {
	    case SM_TYPE_POINT:
		sprintf(interp->result, "point");
		break;

	    case SM_TYPE_LINE:
		sprintf(interp->result, "line");
		break;

	    case SM_TYPE_POLYGON:
		sprintf(interp->result, "polygon");
		break;

	    case SM_TYPE_SPHERE:
		sprintf(interp->result, "sphere");
		break;

	    case SM_TYPE_DISK:
		sprintf(interp->result, "disk");
		break;

	    case SM_TYPE_PARTDISK:
		sprintf(interp->result, "partialdisk");
		break;

	    case SM_TYPE_CYLINDER:
		sprintf(interp->result, "cylinder");
		break;

	    default:
		sprintf(interp->result, "error: unknown mode type");
		break;
	}
    }
    else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0) && (length >= 2)) {

	/* command 'list' */
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " list\"", (char *) NULL);
	    return TCL_ERROR;
	}

	for (model = cell->models; model; model = model->next) {
	    Tcl_AppendElement(interp, model->label);
	}
    }
    else if ((c == 'd') && (strncmp(argv[1], "destroy", length) == 0) && (length >= 2)) {

	/* command 'destroy' */
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " destroy\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if ((e = Tcl_FindHashEntry(&header->hash, cell->label)) == NULL) {
	    Tcl_AppendResult(interp, "cell \"", cell->label, "\" does not exist.", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_DeleteCommand(interp, cell->label);
    }
    else {
	Tcl_AppendResult(interp, "bad cell command \"", argv[1],
			 "\": must be bbox, cget, children, clip, configure, convert, create, delete, destroy, exists, extent, list, modelcget, modelconfigure, setv, or type.", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DestroyCell:
 *
 * handle the destruction of cells.
 *
 *--------------------------------------------------------------
 */

static void
DestroyCell(clientData)
     ClientData clientData;
{
    Cell *current, *last;
    CellHeader *header;
    Tcl_HashEntry *e;
    Model *model;
    PortItem *p, *q;

    Cell *cell = (Cell *) clientData;

    header = SmGetCellHeader(cell->interp);
    for (model = cell->models; model;) {
	if (model->parent) {
	    model = model->next;
	}
	else {
	    DeleteModel(model);
	    model = cell->models;
	}
    }

    for (p = cell->ports; p; p = q) {
	q = p->next;
	p->cell = NULL;
	p->next = NULL;
	p->redraw = 1;
	if (p->canvas) {
	    Tk_CanvasEventuallyRedraw(p->canvas, ((Tk_Item *) p)->x1, ((Tk_Item *) p)->y1,
				      ((Tk_Item *) p)->x2, ((Tk_Item *) p)->y2);
	}
	else if (p->tkwin) {
	    p->redraw = 1;
	    if (p->async && !p->update) {
		p->update = 1;
		Tk_DoWhenIdle(SmViewportUpdate, (ClientData) p);
	    }
	}
    }

    for (current = header->cells, last = NULL; current; last = current, current = current->next) {
	if (current == cell) {
	    if (last) {
		last->next = current->next;
	    }
	    else {
		header->cells = current->next;
	    }
	    break;
	}
    }
    e = Tcl_FindHashEntry(&header->hash, cell->label);
    Tcl_DeleteHashEntry(e);
    Tcl_DeleteHashTable(&cell->hash);
    (void) ckfree((void *) cell->label);

    (void) ckfree((void *) cell);
}

/*
 *--------------------------------------------------------------
 *
 * DestroyCells:
 *
 * destroy all cells..
 *
 *--------------------------------------------------------------
 */

static void
DestroyCells(clientData)
     ClientData clientData;
{
    Tcl_HashEntry *e;
    Tcl_HashSearch search;
    CellHeader *header = (CellHeader *) clientData;

    /* destroy all cell instances */
    while (header->cells) {
	DestroyCell((ClientData) header->cells);
    }

    /* free up memory used by glxGC */
    for (e = Tcl_FirstHashEntry(&header->glxGC, &search); e; e = Tcl_NextHashEntry(&search)) {
	(void) ckfree((void *) Tcl_GetHashValue(e));
    }
}

/*
 *--------------------------------------------------------------
 *
 * GetChildren:
 *
 * return the list of model's children.
 *
 *--------------------------------------------------------------
 */

static int
GetChildren(model)
     Model *model;
{
    Model *child;
    Tcl_Interp *interp = model->cell->interp;

    for (child = model->child; child; child = child->sibling) {
	Tcl_AppendElement(interp, child->label);
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SmHandleNewModel:
 *
 * handle the creation of new models.
 *
 *	syntax of the input is: <cmd> new <type> <label> ?options?
 *	valid model types are:
 *		'polygon'
 *		'line'
 *		'point'
 *		'cylinder'
 *		'disk'
 *		'sphere'
 *
 *--------------------------------------------------------------
 */

int
SmHandleNewModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int length;
    char c;
    Tcl_Interp *interp = cell->interp;

    c = argv[2][0];
    length = strlen(argv[2]);

    if ((c == 'p') && (strncmp(argv[2], "polygon", length) == 0) && (length >= 3)) {
	return NewPolyModel(cell, argc-3, argv+3, model);
    }
    if ((c == 'p') && (strncmp(argv[2], "point", length) == 0) && (length >= 3)) {
	return NewPointModel(cell, argc-3, argv+3, model);
    }
    if ((c == 'l') && (strncmp(argv[2], "line", length) == 0) && (length >= 2)) {
	return NewLineModel(cell, argc-3, argv+3, model);
    }
    else if ((c == 'c') && (strncmp(argv[2], "cylinder", length) == 0) && (length >= 2)) {
	return NewCylinderModel(cell, argc-3, argv+3, model);
    }
    else if ((c == 'd') && (strncmp(argv[2], "disk", length) == 0) && (length >= 2)) {
	return NewDiskModel(cell, argc-3, argv+3, model);
    }
    else if ((c == 's') && (strncmp(argv[2], "sphere", length) == 0) && (length >= 2)) {
	return NewSphereModel(cell, argc-3, argv+3, model);
    }
    else {
	Tcl_AppendResult(interp, "wrong model type \"", argv[2],
			 "\": must be cylinder, disk, line, point, polygon, or sphere.",
			 (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *--------------------------------------------------------------
 *
 * NewPolyModel:
 *
 * create a new model of type 'polygon'.
 *
 *--------------------------------------------------------------
 */

static int
NewPolyModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_POLYGON;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = m->ns = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_NORMALS | SM_MODEL_UPDATE_BBOX;

    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);

    /* parse options */
    if (ConfigureModel(cell, m, modelPolygonConfigSpecs, argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewLineModel:
 *
 * create a new model of type 'line'.
 *
 *--------------------------------------------------------------
 */

static int
NewLineModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_LINE;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = m->ns = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_NORMALS + SM_MODEL_UPDATE_WIDTHS +
      SM_MODEL_UPDATE_STIPPLES + SM_MODEL_UPDATE_BBOX;

    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);

    /* parse options */
    if (ConfigureModel(cell, m, modelLineConfigSpecs, argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewPointModel:
 *
 * create a new model of type 'point'.
 *
 *--------------------------------------------------------------
 */

static int
NewPointModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Surface *s;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_POINT;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_NORMALS + SM_MODEL_UPDATE_SIZES + SM_MODEL_UPDATE_BBOX;

    if ((s = (Surface *) ckalloc(sizeof(Surface))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }
    s->id = 0;
    s->vcnt = 0;
    s->model = m;

    s->index = NULL;
    s->materials = NULL;
    s->lvn = NULL;
    s->texcoords = NULL;
    s->texture = NULL;

    m->ns = 1;
    m->s = s;

    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);

    /* parse options */
    if (ConfigureModel(cell, m, modelPointConfigSpecs, argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewCylinderModel:
 *
 * create a new model of type 'cylinder'.
 *
 *--------------------------------------------------------------
 */

static int
NewCylinderModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_CYLINDER;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = m->ns = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_BBOX;


    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);
    m->quadric = gluNewQuadric();
    m->radius = 0;
    m->baseRadius = 0;
    m->topRadius = 0;
    m->innerRadius = 0;
    m->outerRadius = 0;
    m->startAngle = 0;
    m->sweepAngle = 0;
    m->height = 0;
    m->slices = 0;
    m->loops = 0;
    m->stacks = 0;

    /* parse options */
    if (ConfigureModel(cell, m, modelCylinderConfigSpecs,  argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewDiskModel:
 *
 * create a new model of type 'disk'.
 *
 *--------------------------------------------------------------
 */

static int
NewDiskModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_DISK;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = m->ns = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_BBOX;

    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);
    m->quadric = gluNewQuadric();
    m->radius = 0;
    m->baseRadius = 0;
    m->topRadius = 0;
    m->innerRadius = 0;
    m->outerRadius = 0;
    m->startAngle = 0;
    m->sweepAngle = 0;
    m->height = 0;
    m->slices = 0;
    m->loops = 0;
    m->stacks = 0;

    /* parse options */
    if (ConfigureModel(cell, m, modelDiskConfigSpecs, argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewSphereModel:
 *
 * create a new model of type 'sphere'.
 *
 *--------------------------------------------------------------
 */

static int
NewSphereModel(cell, argc, argv, model)
     Cell *cell;
     int argc;
     char **argv;
     Model **model;
{
    int new;
    Model *m;
    Tcl_HashEntry *me;
    Tcl_Interp *interp = cell->interp;

    /* check for duplicate label */
    me = Tcl_CreateHashEntry(&cell->hash, argv[0], &new);
    if (!new) {
	Tcl_AppendResult(interp, "another model already has label \"", argv[0], "\".", (char *) NULL);
	return TCL_ERROR;
    }

    /* allocate memory for new model */
    if ((m = (Model *) ckalloc(sizeof(Model))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	Tcl_DeleteHashEntry(me);
	return TCL_ERROR;
    }

    /* initialize model */
    m->type = SM_TYPE_SPHERE;
    m->next = NULL;
    m->cell = cell;
    m->child = NULL;
    m->sibling = NULL;
    m->parent = NULL;
    m->e = me;

    m->quadric = NULL;

    m->v = NULL;
    m->s = NULL;
    m->nv = m->ns = 0;

    m->color = NULL;
    m->bfcolor = NULL;

    m->normals = NULL;
    m->materials = NULL;
    m->textures = NULL;
    m->widths = NULL;
    m->stipples = NULL;
    m->sizes = NULL;
    m->pointSizes = NULL;

    m->invalid = 0;
    m->updateFlag = SM_MODEL_UPDATE_BBOX;

    if ((m->label = (char *) ckalloc(strlen(argv[0]) + 1)) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for new model.", (char *) NULL);
	FreeModel(m);
	return TCL_ERROR;
    }
    (void) strcpy(m->label, argv[0]);
    m->quadric = gluNewQuadric();
    m->radius = 0;
    m->baseRadius = 0;
    m->topRadius = 0;
    m->innerRadius = 0;
    m->outerRadius = 0;
    m->startAngle = 0;
    m->sweepAngle = 0;
    m->height = 0;
    m->slices = 0;
    m->loops = 0;
    m->stacks = 0;

    /* parse options */
    if (ConfigureModel(cell, m, modelSphereConfigSpecs, argc-1, argv+1, 0) != TCL_OK) {
	FreeModel(m);
	return TCL_ERROR;
    }

    /* insert model into linked list and hashtable */
    Tcl_SetHashValue(me, (ClientData) m);
    m->next = cell->models;
    cell->models = m;
    Tcl_AppendResult(interp, m->label, (char *) NULL);
    *model = m;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureModel:
 *
 * configure various aspects of a model.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureModel(cell, model, specs, argc, argv, flags)
     Cell *cell;
     Model *model;
     Tk_ConfigSpec specs[];
     int argc;
     char **argv;
     int flags;
{
    DOUBLE r, g, b;
    DOUBLE br, bg, bb;
    double radius, topRadius, baseRadius;
    int updateColor = 0;
    int updateBFColor = 0;
    Tk_Window tkwin = Tk_MainWindow(cell->interp);
    XColor *color = model->color;
    XColor *bfcolor = model->bfcolor;

    double lineWidth;
    int lineStipple, lineFactor;

    int pointSize;

    if (model->type == SM_TYPE_LINE) {
	lineWidth = model->lineWidth;
	lineStipple = model->lineStipple;
	lineFactor = model->lineFactor;
    }
    else if (model->type == SM_TYPE_POINT) {
	pointSize = model->pointSize;
    }
    else if (model->type == SM_TYPE_SPHERE) {
	radius = model->radius;
    }
    else if (model->type == SM_TYPE_DISK) {
	radius = model->outerRadius;
    }
    else if (model->type == SM_TYPE_CYLINDER) {
	topRadius = model->topRadius;
	baseRadius = model->baseRadius;
    }

    SmSetClientData((ClientData) SmGetCellHeader(cell->interp));
    if (Tk_ConfigureWidget(cell->interp, tkwin, specs, argc, argv, (char *) model, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* must update position first to make sure all surface vertex indices are valid */
    if (model->updateFlag & SM_MODEL_UPDATE_POSITION) {
	if ((model->invalid = (ComputeModelTransform(model, cell->epsilon) == TCL_ERROR))) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_POSITION;
    }

    if (model->type == SM_TYPE_LINE) {
	if (model->lineWidth <= 0) {
	    model->lineWidth = 1.0;
	}
	model->lineStipple &= 0xffff;
	if (!(model->updateFlag & SM_MODEL_UPDATE_WIDTHS) &&
	    (model->lineWidth != lineWidth)) {
	    model->updateFlag |= SM_MODEL_UPDATE_WIDTHS;
	}
	if (!(model->updateFlag & SM_MODEL_UPDATE_STIPPLES) &&
	    ((model->lineStipple != lineStipple) ||
	     (model->lineFactor != lineFactor))) {
	    model->updateFlag |= SM_MODEL_UPDATE_STIPPLES;
	}
    }
    else if (model->type == SM_TYPE_POINT) {
	if (model->pointSize <= 0) {
	    model->pointSize = 1.0;
	}
	if (!(model->updateFlag & SM_MODEL_UPDATE_SIZES) &&
	    (model->pointSize != pointSize)) {
	    model->updateFlag |= SM_MODEL_UPDATE_SIZES;
	}
	model->s->vcnt = model->nv;
    }
    else if (model->type == SM_TYPE_SPHERE) {
	if (radius != model->radius) {
	    model->updateFlag |= SM_MODEL_UPDATE_BBOX;
	}
    }
    else if (model->type == SM_TYPE_DISK) {
	if (radius != model->outerRadius) {
	    model->updateFlag |= SM_MODEL_UPDATE_BBOX;
	}
    }
    else if (model->type == SM_TYPE_CYLINDER) {
	if ((topRadius != model->topRadius) ||
	    (baseRadius != model->baseRadius)) {
	    model->updateFlag |= SM_MODEL_UPDATE_BBOX;
	}
    }

    if (model->updateFlag & SM_MODEL_UPDATE_NORMALS) {
	if (SmComputeVertexNormals(model, 1) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_NORMALS;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_MATERIALS) {
	if (ProcessMaterials(model) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_MATERIALS;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_TEXTURES) {
	if (ProcessTextures(model) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_TEXTURES;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_WIDTHS) {
	if (ProcessWidths(model) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_WIDTHS;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_STIPPLES) {
	if (ProcessStipples(model) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_STIPPLES;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_SIZES) {
	if (ProcessSizes(model) != TCL_OK) {
	    return TCL_ERROR;
	}
	model->updateFlag &= ~SM_MODEL_UPDATE_SIZES;
    }

    if (model->updateFlag & SM_MODEL_UPDATE_BBOX) {
	(void) SmComputeBoundingBox(model);
	model->updateFlag &= ~SM_MODEL_UPDATE_BBOX;
    }

    if (model->color != color) {
	color = model->color;
	updateColor = 1;
    }
    r = (DOUBLE) color->red / (DOUBLE) 65535.0;
    g = (DOUBLE) color->green / (DOUBLE) 65535.0;
    b = (DOUBLE) color->blue / (DOUBLE) 65535.0;

    if (updateColor || (model->updateFlag & SM_MODEL_UPDATE_AMBIENT)) {
	model->ambientRGBA[0] = model->ambient[0] * r;
	model->ambientRGBA[1] = model->ambient[1] * g;
	model->ambientRGBA[2] = model->ambient[2] * b;
	model->ambientRGBA[3] = 1.0;
    }

    if (updateColor || (model->updateFlag & SM_MODEL_UPDATE_DIFFUSE)) {
	model->diffuseRGBA[0] = model->diffuse[0] * r;
	model->diffuseRGBA[1] = model->diffuse[1] * g;
	model->diffuseRGBA[2] = model->diffuse[2] * b;
	model->diffuseRGBA[3] = 1.0;
    }

    if (updateColor || (model->updateFlag & SM_MODEL_UPDATE_EMISSIVE)) {
	model->emissiveRGBA[0] = model->emissive[0] * r;
	model->emissiveRGBA[1] = model->emissive[1] * g;
	model->emissiveRGBA[2] = model->emissive[2] * b;
	model->emissiveRGBA[3] = 1.0;
    }

    if (model->type >= SM_TYPE_POLYGON) {
	if (model->bfcolor != bfcolor) {
	    bfcolor = model->bfcolor;
	    updateBFColor = 1;
	}
	br = (DOUBLE) bfcolor->red / (DOUBLE) 65535.0;
	bg = (DOUBLE) bfcolor->green / (DOUBLE) 65535.0;
	bb = (DOUBLE) bfcolor->blue / (DOUBLE) 65535.0;

	if (updateBFColor || (model->updateFlag & SM_MODEL_UPDATE_BF_AMBIENT)) {
	    model->bfambientRGBA[0] = model->bfambient[0] * br;
	    model->bfambientRGBA[1] = model->bfambient[1] * bg;
	    model->bfambientRGBA[2] = model->bfambient[2] * bb;
	    model->bfambientRGBA[3] = 1.0;
	}

	if (updateBFColor || (model->updateFlag & SM_MODEL_UPDATE_BF_DIFFUSE)) {
	    model->bfdiffuseRGBA[0] = model->bfdiffuse[0] * br;
	    model->bfdiffuseRGBA[1] = model->bfdiffuse[1] * bg;
	    model->bfdiffuseRGBA[2] = model->bfdiffuse[2] * bb;
	    model->bfdiffuseRGBA[3] = 1.0;
	}

	if (updateBFColor || (model->updateFlag & SM_MODEL_UPDATE_BF_EMISSIVE)) {
	    model->bfemissiveRGBA[0] = model->bfemissive[0] * br;
	    model->bfemissiveRGBA[1] = model->bfemissive[1] * bg;
	    model->bfemissiveRGBA[2] = model->bfemissive[2] * bb;
	    model->bfemissiveRGBA[3] = 1.0;
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ComputeModelTransform:
 *
 * compute the model transform matrix for a model.
 *
 *--------------------------------------------------------------
 */

static int
ComputeModelTransform(model, epsilon)
     Model *model;
     double epsilon;
{
    int i, j;
    Surface *s;
    Tcl_Interp *interp = model->cell->interp;
    DOUBLE x[3], y[3], z[3];
    char errBuf[64];

    /* check fwd and up vectors */
    if (((fabs((double) model->fwd[0]) < epsilon) && (fabs((double) model->fwd[1]) < epsilon) && (fabs((double) model->fwd[2]) < epsilon)) ||
	SmNormalizeVector3D(model->fwd, z, epsilon)) {
	Tcl_AppendResult(interp, "model \"", model->label, "\" has null forward vector", (char *) NULL);
	return TCL_ERROR;
    }

    if ((fabs((double) model->up[0]) < epsilon) && (fabs((double) model->up[1]) < epsilon) && (fabs((double) model->up[2]) < epsilon)) {
	Tcl_AppendResult(interp, "model \"", model->label, "\" has null up vector", (char *) NULL);
	return TCL_ERROR;
    }

    SmCrossVector3D(model->up, z, x);
    if (((fabs((double) x[0]) < epsilon) && (fabs((double) x[1]) < epsilon) && (fabs((double) x[2]) < epsilon)) || SmNormalizeVector3D(x, x, epsilon)) {
	Tcl_AppendResult(interp, "model \"", model->label, "\" has invalid forward and up vectors", (char *) NULL);
	return TCL_ERROR;
    }
    SmCrossVector3D(z, x, y);

    /* make sure vertex indices are within bounds */
    if (model->type != SM_TYPE_POINT) {
	for (i = 0, s = model->s; i < (const) model->ns; s++, i++) {
	    for (j = 0; j < s->vcnt; j++) {
		if (s->index[j] < 0 || s->index[j] >= model->nv) {
		    sprintf(errBuf, "%d", s->index[j]);
		    Tcl_AppendResult(interp, "model \"", model->label,
				     "\" surface/line vertex index out of bounds \"",
				     errBuf, "\"", (char *) NULL);
		    return TCL_ERROR;
		}
	    }
	}
    }

    /* set up model transform matrix */
    model->modelT[0] = x[0];
    model->modelT[1] = x[1];
    model->modelT[2] = x[2];
    model->modelT[3] = 0;

    model->modelT[4] = y[0];
    model->modelT[5] = y[1];
    model->modelT[6] = y[2];
    model->modelT[7] = 0;

    model->modelT[8] = z[0];
    model->modelT[9] = z[1];
    model->modelT[10] = z[2];
    model->modelT[11] = 0;

    model->modelT[12] = model->pos[0];
    model->modelT[13] = model->pos[1];
    model->modelT[14] = model->pos[2];
    model->modelT[15] = 1;

    /* passed all tests */
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ComputeSurfaceNormal:
 *
 * compute the surface normal vector from vertex positions.
 *
 *--------------------------------------------------------------
 */

static void
ComputeSurfaceNormal(v, s, n, epsilon)
     Vertex *v;
     Surface *s;
     DOUBLE n[3];
     double epsilon;
{
    Vertex *v0, *v1, *v2;
    DOUBLE a[3], b[3];

    v0 = &v[s->index[0]];
    v1 = &v[s->index[1]];
    v2 = &v[s->index[2]];

    a[0] = v0->lv[0] - v1->lv[0];
    a[1] = v0->lv[1] - v1->lv[1];
    a[2] = v0->lv[2] - v1->lv[2];

    b[0] = v1->lv[0] - v2->lv[0];
    b[1] = v1->lv[1] - v2->lv[1];
    b[2] = v1->lv[2] - v2->lv[2];
	
    SmCrossVector3D(a, b, n);
    if (SmNormalizeVector3D(n, n, epsilon)) {
	n[0] = n[1] = n[2] = 0;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmComputeVertexNormals:
 *
 * compute vertex normals.
 *
 *--------------------------------------------------------------
 */

int
SmComputeVertexNormals(model, reset)
     Model *model;
     int reset;
{
    int i, j;
    int argc;
    char **argv;
    Vertex *v;
    Surface *s;
    Cell *cell = model->cell;
    Tcl_Interp *interp = cell->interp;

    /* reset vertex normals */
    if (reset) {
	for (s = model->s, i = 0; i < (const) model->ns; s++, i++) {
	    s->normal = SM_VERTEX_NORMAL_DEFAULT;
	    if (s->lvn) {
		(void) ckfree((void *) s->lvn);
		s->lvn = NULL;
	    }
	}

	/* parse user-specified normals */
	if (model->normals) {

	    if (model->type == SM_TYPE_POINT) {
		if (ParseVertexNormals(interp, model, model->normals) != TCL_OK) {
		    (void) ckfree((void *) argv);
		    (void) ckfree((void *) model->normals);
		    model->normals = NULL;
		    return TCL_ERROR;
		}
	    }
	    else {
		if (Tcl_SplitList(interp, model->normals, &argc, &argv) != TCL_OK) {
		    return TCL_ERROR;
		}
		for (i = 0; i < (const) argc; i++) {
		    if (ParseVertexNormals(interp, model, argv[i]) != TCL_OK) {
			(void) ckfree((void *) argv);
			(void) ckfree((void *) model->normals);
			model->normals = NULL;
			return TCL_ERROR;
		    }
		}
		(void) ckfree((void *) argv);
		(void) ckfree((void *) model->normals);
		model->normals = NULL;
	    }
	}
    }

    if (model->type == SM_TYPE_POLYGON) {

	/* reset vertex normal to zero */
	for (v = model->v, i = 0; i < (const) model->nv; v++, i++) {
	    v->lvn.v[0] = v->lvn.v[1] = v->lvn.v[2] = 0;
	}

	/* sum surface normals */
	for (s = model->s, i = 0; i < (const) model->ns; s++, i++) {
	    ComputeSurfaceNormal(model->v, s, s->ln, cell->epsilon);
	    for (j = 0; j < (const) s->vcnt; j++) {
		v = &model->v[s->index[j]];
		v->lvn.v[0] += s->ln[0];
		v->lvn.v[1] += s->ln[1];
		v->lvn.v[2] += s->ln[2];
	    }
	}

	/* normalize vertex normals */
	for (v = model->v, i = 0; i < (const) model->nv; v++, i++) {
	    if ((v->lvn.v[0] != 0) || (v->lvn.v[1] != 0) || (v->lvn.v[2] != 0))
	        FastNormalizeVector3D(v->lvn.v, v->lvn.v);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseVertexNormals:
 *
 * parse vertex normals for a single surface.
 *
 *--------------------------------------------------------------
 */

static int
ParseVertexNormals(interp, model, value)
     Tcl_Interp *interp;
     Model *model;
     char *value;
{
    int i, argc, nc, index;
    int point;
    char **argv, **coords;
    Normal3D *n;
    Surface *s;
    double v[3];
    Cell *cell = model->cell;

    point = (model->type == SM_TYPE_POINT);
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    if (argc < 1 - point) {
	Tcl_AppendResult(interp, "bad vertex normal format: ", value, (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (!point) {
	if (Tcl_GetInt(interp, argv[0], &index) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (index < 1 || index > model->ns) {
	    Tcl_AppendResult(interp, "bad surface index: ", argv[0], (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	s = &model->s[index - 1];
    }
    else {
	s = model->s;
    }

    /* number of normals must equal number of vertices */
    if ((argc != 1 - point) && (argc != s->vcnt + 1 - point)) {
	Tcl_AppendResult(interp, "bad format: number of vertex normals must equal either 0 or number of vertices.",
			 (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (s->lvn) {
	(void) ckfree((void *) s->lvn);
	s->lvn = NULL;
    }

    if (argc > 1 - point) {

	/* allocate memory for vertex normal vectors */
	if ((s->lvn = (Normal3D *) ckalloc(s->vcnt * sizeof(Normal3D))) == NULL) {
	    Tcl_AppendResult(interp, "out of memory: unable to allocate memory for vertex normals.",
			     (char *) NULL);
	    goto err;
	}

	for (i = 1 - point, n = s->lvn; i < (const) argc; i++, n++) {
	    if (Tcl_SplitList(interp, argv[i], &nc, &coords) != TCL_OK) {
		goto err;
	    }
	    if (nc != 3) {
		Tcl_AppendResult(interp, "bad vector format: 3 components required.", (char *) NULL);
		(void) ckfree((void *) coords);
		goto err;
	    }
	    if (Tcl_GetDouble(interp, coords[0], &v[0]) != TCL_OK ||
		Tcl_GetDouble(interp, coords[1], &v[1]) != TCL_OK ||
		Tcl_GetDouble(interp, coords[2], &v[2]) != TCL_OK) {
		(void) ckfree((void *) coords);
		goto err;
	    }
	    n->v[0] = v[0];
	    n->v[1] = v[1];
	    n->v[2] = v[2];
	    
	    if (SmNormalizeVector3D(n->v, n->v, cell->epsilon)) {
		Tcl_AppendResult(interp, "illegal vertex normal: ", argv[i], (char *) NULL);
		(void) ckfree((void *) coords);
		goto err;
	    }
	    (void) ckfree((void *) coords);
	}
    }
    s->normal = (argc == 1 - point) ? SM_VERTEX_NORMAL_DEFAULT : SM_VERTEX_NORMAL_USER;
    (void) ckfree((void *) argv);
    return TCL_OK;

  err:
    (void) ckfree((void *) argv);
    if (s->lvn) {
	(void) ckfree((void *) s->lvn);
	s->lvn = NULL;
    }
    s->normal = SM_VERTEX_NORMAL_DEFAULT;
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ProcessMaterials:
 *
 * process vertex material option.
 *
 *--------------------------------------------------------------
 */

static int
ProcessMaterials(model)
     Model *model;
{
    int i, argc;
    char **argv;
    Surface *s;
    Tcl_Interp *interp = model->cell->interp;

    /* reset vertex materials */
    for (s = model->s, i = 0; i < (const) model->ns; s++, i++) {
	if (s->materials) {
	    (void) ckfree((void *) s->materials);
	    s->materials = NULL;
	}
    }
    if (model->materials == NULL) return TCL_OK;

    if (model->type == SM_TYPE_POINT) {
	if (ParseVertexMaterials(interp, model, model->materials) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) model->materials);
	    model->materials = NULL;
	    return TCL_ERROR;
	}
    }
    else {
	if (Tcl_SplitList(interp, model->materials, &argc, &argv) != TCL_OK) {
	    (void) ckfree((void *) model->materials);
	    model->materials = NULL;
	    return TCL_ERROR;
	}

	for (i = 0; i < (const) argc; i++) {
	    if (ParseVertexMaterials(interp, model, argv[i]) != TCL_OK) {
		(void) ckfree((void *) argv);
		(void) ckfree((void *) model->materials);
		model->materials = NULL;
		return TCL_ERROR;
	    }
	}
	(void) ckfree((void *) argv);
	(void) ckfree((void *) model->materials);
	model->materials = NULL;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseVertexMaterials:
 *
 * parse vertex materials for a single surface.
 *
 *--------------------------------------------------------------
 */

static int
ParseVertexMaterials(interp, model, value)
     Tcl_Interp *interp;
     Model *model;
     char *value;
{
    int i, argc, n, index;
    int point;
    char **argv, **m;
    double r, g, b;
    Surface *s;
    Material *mat;

    point = (model->type == SM_TYPE_POINT);
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc < 1 - point) {
	(void) ckfree((void *) argv);
	Tcl_AppendResult(interp, "bad vertex material format:", value, (char *) NULL);
	return TCL_ERROR;
    }

    if (!point) {
	if (Tcl_GetInt(interp, argv[0], &index) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (index < 1 || index > model->ns) {
	    Tcl_AppendResult(interp, "bad materials surface index: ", argv[0], (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	s = &model->s[index - 1];
    }
    else {
	s = model->s;
    }

    /* number of material values must equal number of vertices */
    if ((argc != 1 - point) && (argc != s->vcnt + 1 - point)) {
	Tcl_AppendResult(interp, "bad format: number of vertex materials must equal either 0 or number of vertices.",
			 (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (s->materials) {
	(void) ckfree((void *) s->materials);
	s->materials = NULL;
    }

    if (argc > 1 - point) {

	/* allocate memory for vertex material records */
	if ((s->materials = (Material *) ckalloc(s->vcnt * sizeof(Material))) == NULL) {
	    Tcl_AppendResult(interp, "out of memory: unable to allocate memory for vertex materials.",
			     (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	for (i = 1 - point, mat = s->materials; i < (const) argc; i++, mat++) {

	    if (Tcl_SplitList(interp, argv[i], &n, &m) != TCL_OK) {
		goto err;
	    }
	    if (n != 5) {
		if (point) {
		    Tcl_AppendResult(interp,
				     "bad material format: should be {{ambientColor diffuseColor specularColor emissiveColor shininess} ...}",
				     (char *) NULL);
		}
		else {
		    Tcl_AppendResult(interp,
				     "bad material format: should be {index {ambientColor diffuseColor specularColor emissiveColor shininess} ...}",
				     (char *) NULL);
		}
		(void) ckfree((void *) m);
		goto err;
	    }

	    if (SmParseColorCoefficients(interp, m[0], &r, &g, &b) != TCL_OK) {
		(void) ckfree((void *) m);
		goto err;
	    }
	    mat->ambientRGBA[0] = (float) r;
	    mat->ambientRGBA[1] = (float) g;
	    mat->ambientRGBA[2] = (float) b;
	    mat->ambientRGBA[3] = 1.0;

	    if (SmParseColorCoefficients(interp, m[1], &r, &g, &b) != TCL_OK) {
		(void) ckfree((void *) m);
		goto err;
	    }
	    mat->diffuseRGBA[0] = (float) r;
	    mat->diffuseRGBA[1] = (float) g;
	    mat->diffuseRGBA[2] = (float) b;
	    mat->diffuseRGBA[3] = 1.0;

	    if (SmParseColorCoefficients(interp, m[2], &r, &g, &b) != TCL_OK) {
		(void) ckfree((void *) m);
		goto err;
	    }
	    mat->specularRGBA[0] = (float) r;
	    mat->specularRGBA[1] = (float) g;
	    mat->specularRGBA[2] = (float) b;
	    mat->specularRGBA[3] = 1.0;

	    if (SmParseColorCoefficients(interp, m[3], &r, &g, &b) != TCL_OK) {
		(void) ckfree((void *) m);
		goto err;
	    }
	    mat->emissiveRGBA[0] = (float) r;
	    mat->emissiveRGBA[1] = (float) g;
	    mat->emissiveRGBA[2] = (float) b;
	    mat->emissiveRGBA[3] = 1.0;

	    if (Tcl_GetDouble(interp, m[4], &mat->shininess) != TCL_OK) {
		(void) ckfree((void *) m);
		goto err;
	    }
	    (void) ckfree((void *) m);
	}
    }
    (void) ckfree((void *) argv);
    return TCL_OK;

  err:
    (void) ckfree((void *) argv);
    if (s->materials) {
	(void) ckfree((void *) s->materials);
	s->materials = NULL;
    }
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ProcessTextures:
 *
 * process the textures option.
 *
 * each texture entry (except for point models) has the syntax:
 *
 * {{index texture comp func s-mode t-mode mag-filter min-filter bd-color texcoords} ...}
 *
 * texture for a point model has the syntax:
 *
 * {texture comp func s-mode t-mode mag-filter min-filter bd-color texcoords}
 *
 *--------------------------------------------------------------
 */

static int
ProcessTextures(model)
     Model *model;
{
    int i, result, argc;
    char **argv;
    Surface *s;
    Tcl_Interp *interp = model->cell->interp;
    CellHeader *header = SmGetCellHeader(interp);

    /* reset texture maps */
    for (s = model->s, i = 0; i < (const) model->ns; s++, i++) {
	if (s->texture) {
	    SmFreeTexture(interp, header, s->texture);
	    s->texture = NULL;
	}
	if (s->texcoords) {
	    (void) ckfree((void *) s->texcoords);
	    s->texcoords = NULL;
	}
    }

    if (model->textures) {

	if (model->type == SM_TYPE_POINT) {
	    result = ParseTexture(interp, header, model, model->textures);
	    (void) ckfree((void *) model->textures);
	    model->textures = NULL;
	    return result;
	}
	else {
	    if (Tcl_SplitList(interp, model->textures, &argc, &argv) != TCL_OK) {
		(void) ckfree((void *) model->textures);
		model->textures = NULL;
		return TCL_ERROR;
	    }

	    for (i = 0; i < (const) argc; i++) {
		if (ParseTexture(interp, header, model, argv[i]) != TCL_OK) {
		    (void) ckfree((void *) argv);
		    (void) ckfree((void *) model->textures);
		    model->textures = NULL;
		    return TCL_ERROR;
		}
	    }
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) model->textures);
	    model->textures = NULL;
	}
    }
    return TCL_OK;
}
		

/*
 *--------------------------------------------------------------
 *
 * ParseTexture:
 *
 * parse a single texture entry.
 *
 * a texture entry (except for point models) has the syntax:
 *
 * {{index texture comp func s-mode t-mode mag min bdcolor texcoords} ...}
 *
 * texture for a point model has the syntax:
 *
 * {texture comp func s-mode t-mode mag-filter min-filter bd-color texcoords}
 *
 *--------------------------------------------------------------
 */

static int
ParseTexture(interp, header, model, entry)
     Tcl_Interp *interp;
     CellHeader *header;
     Model *model;
     char *entry;
{
    int i, argc, funcargc, cargc;
    int point;
    char **argv, **funcargv, **cargv;
    Surface *s;
    TexCoords *texcoords;

    int index;
    int comp;
    double r, g, b;
    double x, y;

    point = (model->type == SM_TYPE_POINT);
    if (Tcl_SplitList(interp, entry, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    /* parse index */
    if (!point)  {
	if (Tcl_GetInt(interp, argv[0], &index) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	if ((index < 1) || (index > model->ns)) {
	    Tcl_AppendResult(interp, "index out of range: \"", argv[0], "\"", (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	s = model->s + (index - 1);
    }
    else {
	s = model->s;
    }

    if ((argc != 1 - point) && (argc != 10 - point)) {
	if (point) {
	    Tcl_AppendResult(interp,
			     "wrong # args: texture map entry should be {texture component func s-mode t-mode mag min bdcolor texcoords}",
			     (char *) NULL);
	}
	else {
	    Tcl_AppendResult(interp,
			     "wrong # args: texture map entry should be {{index texture component func s-mode t-mode mag min bdcolor texcoords} ,,,}",
			     (char *) NULL);
	}
	return TCL_ERROR;
    }

    if (s->texture) {
	SmFreeTexture(interp, header, s->texture);
	s->texture = NULL;
    }

    if (argc > 1 - point) {

	/* parse texture */
	if ((s->texture = SmGetTexture(interp, header, argv[1 - point])) == NULL) {
	    Tcl_AppendResult(interp, "texture not found: \"", argv[1 - point], "\"", (char *) NULL);
	    goto err;
	}

	/* parse component */
	if (Tcl_GetInt(interp, argv[2 - point], &comp) != TCL_OK) {
	    goto err;
	}

	if ((comp < 0) || (comp > 4)) {
	    Tcl_AppendResult(interp, "component out of range \"", argv[2 - point],
			     "\": must be between 1 and 4", (char *) NULL);
	    goto err;
	}
	if ((comp != 1) && (comp != 3)) {
	    Tcl_AppendResult(interp, "component selection not supported \"", argv[2 - point],
			     "\": component selection must be 1 (R) or 3 (RGB)", (char *) NULL);
	    goto err;
	}
	s->component = comp;

	/* parse function */
	if (Tcl_SplitList(interp, argv[3 - point], &funcargc, &funcargv) != TCL_OK) {
	    goto err;
	}

	if ((funcargc != 1) && (funcargc != 4)) {
	    Tcl_AppendResult(interp, "wrong # args \"", argv[3 - point],
			     "\": must be one of decal, modulate, and {blend r g b}", (char *) NULL);
	    (void) ckfree((void *) funcargv);
	    goto err;
	}
	
	if (funcargc == 1) {
	    if (strcmp(funcargv[0], "decal") == 0) {
		s->func = GL_DECAL;
	    }
	    else if (strcmp(funcargv[0], "modulate") == 0) {
		s->func = GL_MODULATE;
	    }
	    else {
		Tcl_AppendResult(interp, "bad function \"", funcargv[0],
				 "\": should be one of {blend r g b}, decal, or modulate", (char *) NULL);
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    (void) ckfree((void *) funcargv);
	}
	else {
	    if (strcmp(funcargv[0], "blend") != 0) {
		Tcl_AppendResult(interp, "bad function \"", funcargv[0],
				 "\": should be one of {blend r g b}, decal, or modulate", (char *) NULL);
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    if ((Tcl_GetDouble(interp, funcargv[1], &r) != TCL_OK) ||
		(Tcl_GetDouble(interp, funcargv[2], &g) != TCL_OK) ||
		(Tcl_GetDouble(interp, funcargv[3], &b) != TCL_OK)) {
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    if ((r < (double) 0) || (r > (double) 1) ||
		(g < (double) 0) || (g > (double) 1) ||
		(b < (double) 0) || (b > (double) 1)) {
		Tcl_AppendResult(interp, "rgb component values must be between 0 and 1",
				 (char *) NULL);
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    (void) ckfree((void *) funcargv);
	    s->texColor[0] = (GLfloat) r;
	    s->texColor[1] = (GLfloat) g;
	    s->texColor[2] = (GLfloat) b;
	    s->texColor[3] = (GLfloat) 1;
	}
	/* make sure the component selection is valid for the function specified */
	switch(s->func) {
	    
	  case GL_DECAL:	/* for decal, component must be 3 */
	    if (comp != 3) {
		Tcl_AppendResult(interp, "component selection of \"", argv[2 - point],
				 "\" is not supported for function \"decal\"", (char *) NULL);
		goto err;
	    }
	    break;

	  case GL_BLEND:	/* for blend, component must be 1 */
	    if (comp != 1) {
		Tcl_AppendResult(interp, "component selection of \"", argv[2 - point],
				 "\" is not supported for function \"blend\"", (char *) NULL);
		goto err;
	    }
	    break;
      
	  default:
	    break;
	}

	/* parse s-mode */
	if (strcmp(argv[4 - point], "clamp") == 0) {
	    s->smode = GL_CLAMP;
	}
	else if (strcmp(argv[4 - point], "repeat") == 0) {
	    s->smode = GL_REPEAT;
	}
	else {
	    Tcl_AppendResult(interp, "bad wrap mode \"", argv[4 - point],
			     "\": should be either clamp or repeat", (char *) NULL);
	    goto err;
	}
	
	/* parse t-mode */
	if (strcmp(argv[5 - point], "clamp") == 0) {
	    s->tmode = GL_CLAMP;
	}
	else if (strcmp(argv[5 - point], "repeat") == 0) {
	    s->tmode = GL_REPEAT;
	}
	else {
	    Tcl_AppendResult(interp, "bad wrap mode \"", argv[5 - point],
			     "\": should be either clamp or repeat", (char *) NULL);
	    goto err;
	}
	
	/* parse mag-filter */
	if (strcmp(argv[6 - point], "nearest") == 0) {
	    s->magFilter = GL_NEAREST;
	}
	else if (strcmp(argv[6 - point], "linear") == 0) {
	    s->magFilter = GL_LINEAR;
	}
	else {
	    Tcl_AppendResult(interp, "bad magnification filter mode \"", argv[6 - point],
			     "\": should be either nearest or linear", (char *) NULL);
	    goto err;
	}
	
	/* parse min-filter */
	if (strcmp(argv[7 - point], "nearest") == 0) {
	    s->minFilter = GL_NEAREST;
	}
	else if (strcmp(argv[7 - point], "linear") == 0) {
	    s->minFilter = GL_LINEAR;
	}
	else {
	    Tcl_AppendResult(interp, "bad minification filter mode \"", argv[7 - point],
			     "\": should be either nearest or linear", (char *) NULL);
	    goto err;
	}

	/* parse border color*/
	if (Tcl_SplitList(interp, argv[8 - point], &funcargc, &funcargv) != TCL_OK) {
	    goto err;
	}
	if (funcargc != 3) {
	    Tcl_AppendResult(interp, "wrong # args \"", argv[8 - point],
			     "\": expected (r g b) values", (char *) NULL);
	    (void) ckfree((void *) funcargv);
	    goto err;
	}
	if ((Tcl_GetDouble(interp, funcargv[0], &r) != TCL_OK) ||
	    (Tcl_GetDouble(interp, funcargv[1], &g) != TCL_OK) ||
	    (Tcl_GetDouble(interp, funcargv[2], &b) != TCL_OK)) {
	    (void) ckfree((void *) funcargv);
	    goto err;
	}
	if ((r < (double) 0) || (r > (double) 1) ||
	    (g < (double) 0) || (g > (double) 1) ||
	    (b < (double) 0) || (b > (double) 1)) {
	    Tcl_AppendResult(interp, "rgb component values must be between 0 and 1",
			     (char *) NULL);
	    (void) ckfree((void *) funcargv);
	    goto err;
	}
	(void) ckfree((void *) funcargv);
	s->bdColor[0] = (GLfloat) r;
	s->bdColor[1] = (GLfloat) g;
	s->bdColor[2] = (GLfloat) b;
	s->bdColor[3] = (GLfloat) 1;
	
	/* parse texture coordinates */
	if (Tcl_SplitList(interp, argv[9 - point], &funcargc, &funcargv) != TCL_OK) {
	    goto err;
	}

	/* number of normals must equal number of vertices */
	if (funcargc != s->vcnt) {
	    Tcl_AppendResult(interp, "bad format: number of texture coordinates must equal number of vertices.",
			     (char *) NULL);
	    (void) ckfree((void *) funcargv);
	    goto err;
	}
	
	/* allocate memory for texture coordinates */
	if (s->texcoords) {
	    (void) ckfree((void *) s->texcoords);
	}
	if ((s->texcoords = (TexCoords *) ckalloc(funcargc * sizeof(TexCoords))) == NULL) {
	    Tcl_AppendResult(interp, "out of memory: unable to allocate memory for texture coordinates.",
			     (char *) NULL);
	    (void) ckfree((void *) funcargv);
	    goto err;
	}

	for (i = 0, texcoords = s->texcoords; i < (const) funcargc; i++, texcoords++) {
	    if (Tcl_SplitList(interp, funcargv[i], &cargc, &cargv) != TCL_OK) {
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    if (cargc != 2) {
		Tcl_AppendResult(interp, "bad texture coordinates: 2 components (s,t) required.", (char *) NULL);
		(void) ckfree((void *) cargv);
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    if ((Tcl_GetDouble(interp, cargv[0], &x) != TCL_OK) ||
		(Tcl_GetDouble(interp, cargv[1], &y) != TCL_OK)) {
		(void) ckfree((void *) cargv);
		(void) ckfree((void *) funcargv);
		goto err;
	    }
	    texcoords->v[0] = (DOUBLE) x;
	    texcoords->v[1] = (DOUBLE) y;
	    (void) ckfree((void *) cargv);
	}
	(void) ckfree((void *) funcargv);
    }
    (void) ckfree((void *) argv);
    return TCL_OK;

  err:
    (void) ckfree((void *) argv);
    if (s->texture) {
	SmFreeTexture(interp, header, s->texture);
	s->texture = NULL;
    }
    if (s->texcoords) {
	(void) ckfree((void *) s->texcoords);
	s->texcoords = NULL;
    }
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ProcessWidths:
 *
 * process line segments width option.
 *
 * {{index width} ...}
 *
 *--------------------------------------------------------------
 */

static int
ProcessWidths(model)
     Model *model;
{
    int i, argc;
    char **argv;
    Surface *line;
    Tcl_Interp *interp = model->cell->interp;

    /* reset widths of line segments */
    for (line = model->s, i = 0; i < (const) model->ns; line++, i++) {
	line->lineWidth = model->lineWidth;
	line->defaultWidth = 1;
    }
    if (model->widths == NULL) return TCL_OK;

    if (Tcl_SplitList(interp, model->widths, &argc, &argv) != TCL_OK) {
	(void) ckfree((void *) model->widths);
	model->widths = NULL;
	return TCL_ERROR;
    }

    for (i = 0; i < (const) argc; i++) {
	if (ParseWidth(interp, model, argv[i]) != TCL_OK) {
	    (void) ckfree((void *) model->widths);
	    model->widths = NULL;
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
    }
    (void) ckfree((void *) model->widths);
    model->widths = NULL;
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseWidth:
 *
 * parse line width for a single segment.
 *
 * {{index width} ...}
 *
 *--------------------------------------------------------------
 */

static int
ParseWidth(interp, model, value)
     Tcl_Interp *interp;
     Model *model;
     char *value;
{
    int argc, index;
    char **argv;
    Surface *line;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((argc != 1) && (argc != 2)) {
	(void) ckfree((void *) argv);
	Tcl_AppendResult(interp, "bad line segment width format \"", value,
			 "\": should be {{index width} ...}", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[0], &index) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (index < 1 || index > model->ns) {
	Tcl_AppendResult(interp, "bad line segment index: ", argv[0], (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    line = &model->s[index - 1];

    if (argc == 1) {
	line->lineWidth = model->lineWidth;
    }
    else {
	if (Tcl_GetDouble(interp, argv[1], &line->lineWidth) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	if (line->lineWidth < 0) {
	    line->lineWidth = 1;
	}
	line->defaultWidth = 0;
    }
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ProcessStipples:
 *
 * process line segments stipple option.
 *
 * {{index stipple scaleFactor} ...}
 *
 *--------------------------------------------------------------
 */

static int
ProcessStipples(model)
     Model *model;
{
    int i, argc;
    char **argv;
    Surface *line;
    Tcl_Interp *interp = model->cell->interp;

    /* reset widths of line segments */
    for (line = model->s, i = 0; i < (const) model->ns; line++, i++) {
	line->defaultStipple = 1;
	line->lineStipple = model->lineStipple;
	line->lineFactor = model->lineFactor;
    }
    if (model->stipples == NULL) return TCL_OK;

    if (Tcl_SplitList(interp, model->stipples, &argc, &argv) != TCL_OK) {
	(void) ckfree((void *) model->stipples);
	model->stipples = NULL;
	return TCL_ERROR;
    }

    (void) ckfree((void *) model->stipples);
    model->stipples = NULL;

    for (i = 0; i < (const) argc; i++) {
	if (ParseStipple(interp, model, argv[i]) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
    }
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseStipple:
 *
 * parse line stipple for a single segment.
 *
 * {{index stipple scaleFactor} ...}
 *
 *--------------------------------------------------------------
 */

static int
ParseStipple(interp, model, value)
     Tcl_Interp *interp;
     Model *model;
     char *value;
{
    int argc, index;
    char **argv;
    Surface *line;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((argc != 1) && (argc != 3)) {
	(void) ckfree((void *) argv);
	Tcl_AppendResult(interp, "bad line segment stipple format \"", value,
			 "\": should be {{index stipple scaleFactor} ...}", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[0], &index) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (index < 1 || index > model->ns) {
	Tcl_AppendResult(interp, "bad line segment index: ", argv[0], (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    line = &model->s[index - 1];

    if (argc == 1) {
	line->lineStipple = model->lineStipple;
	line->lineFactor = model->lineFactor;
    }
    else {
	if ((Tcl_GetInt(interp, argv[1], &line->lineStipple) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[2], &line->lineFactor)) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	line->lineStipple &= 0xffff;
	line->defaultStipple = 0;
    }
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ProcessSizes:
 *
 * process point size option.
 *
 * {size1 size2 ...}
 *
 *--------------------------------------------------------------
 */

static int
ProcessSizes(model)
     Model *model;
{
    int i, argc;
    char **argv;
    double size;
    Tcl_Interp *interp = model->cell->interp;

    /* reset point sizes */
    if (model->pointSizes) {
	(void) ckfree((void *) model->pointSizes);
	model->pointSizes = NULL;
    }
    if (model->sizes == NULL) return TCL_OK;

    if (Tcl_SplitList(interp, model->sizes, &argc, &argv) != TCL_OK) {
	(void) ckfree((void *) model->sizes);
	model->sizes = NULL;
	return TCL_ERROR;
    }

    if (argc != model->nv) {
	Tcl_AppendResult(interp, "bad format: number of sizes must equal number of vertices.",
			 (char *) NULL);
	(void) ckfree((void *) argv);
	(void) ckfree((void *) model->sizes);
	model->sizes = NULL;
	return TCL_ERROR;
    }

    /* allocate memory for point size records */
    if ((model->pointSizes = (DOUBLE *) ckalloc(model->nv * sizeof(DOUBLE))) == NULL) {
	Tcl_AppendResult(interp, "out of memory: unable to allocate memory for point sizes.",
			 (char *) NULL);
	(void) ckfree((void *) argv);
	(void) ckfree((void *) model->sizes);
	model->sizes = NULL;
	return TCL_ERROR;
    }

    for (i = 0; i < (const) argc; i++) {
	if (Tcl_GetDouble(interp, argv[i], &size) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) model->sizes);
	    model->sizes = NULL;
	    return TCL_ERROR;
	}
	model->pointSizes[i] = (DOUBLE) size;
    }
    (void) ckfree((void *) argv);
    (void) ckfree((void *) model->sizes);
    model->sizes = NULL;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SmParseColorCoefficients:
 *
 * parse color coefficients
 *
 * {r g b}
 *--------------------------------------------------------------
 */

int
SmParseColorCoefficients(interp, value, r, g, b)
     Tcl_Interp *interp;
     char *value;
     double *r, *g, *b;
{
    int argc;
    char **argv;
    Tk_Window tkwin = Tk_MainWindow(interp);
    XColor exactColor, screenColor;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    if ((argc != 3) && (argc != 1)) {
	Tcl_AppendResult(interp, "bad color coefficients \"", value, "\": should be {r g b} or color", (char *) NULL);
	return TCL_ERROR;
    }

    if (argc == 3) {
	if ((Tcl_GetDouble(interp, argv[0], r) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[1], g) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[2], b) != TCL_OK)) {
	    return TCL_ERROR;
	}
    }
    else {

	if (!XLookupColor(Tk_Display(tkwin), Tk_Colormap(tkwin), argv[0], &exactColor, &screenColor)) {
	    Tcl_AppendResult(interp, "bad color name \"", argv[0], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	else {
	    *r = (double) exactColor.red / 65535.0;
	    *g = (double) exactColor.green / 65535.0;
	    *b = (double) exactColor.blue / 65535.0;
	}
    }

    if ((*r < 0) || (*r > 1)) {
	Tcl_AppendResult(interp, "color coefficients must be between 0 and 1: \"", argv[0], "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if ((*g < 0) || (*g > 1)) {
	Tcl_AppendResult(interp, "color coefficients must be between 0 and 1: \"", argv[1], "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if ((*b < 0) || (*b > 1)) {
	Tcl_AppendResult(interp, "color coefficients must be between 0 and 1: \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FastNormalizeVector3D:
 *
 * normalize a 3D vector that is represented as three scalar
 * values.
 *
 *--------------------------------------------------------------
 */

static void
FastNormalizeVector3D(vin, vout)
     DOUBLE vin[];
     DOUBLE vout[];
{
    DOUBLE mag;

    mag = (DOUBLE) 1.0 / sqrt((double) (vin[0] * vin[0] + vin[1] * vin[1] + vin[
2] * vin[2]));
    vout[0] = vin[0] * mag;
    vout[1] = vin[1] * mag;
    vout[2] = vin[2] * mag;
}

/*
 *--------------------------------------------------------------
 *
 * SmComputeBoundingBox:
 *
 * compute the bounding box for a model
 *
 *--------------------------------------------------------------
 */

int
SmComputeBoundingBox(model)
     Model *model;
{
    int ns, nv, empty;
    Vertex *v;
    Surface *s;
    DOUBLE *u;
    DOUBLE maxx, maxy, maxz, minx, miny, minz;
    DOUBLE cmaxx, cmaxy, cmaxz, cminx, cminy, cminz;

    empty = 0;
    switch (model->type) {

      case SM_TYPE_CYLINDER:
	minz = 0;
	maxz = model->height;
	minx = miny = -((model->topRadius > model->baseRadius) ? model->topRadius : model->baseRadius);
	maxx = maxy = -minx;
	break;

      case SM_TYPE_DISK:
	minz = maxz = 0;
	minx = miny = -model->outerRadius;
	maxx = maxy = model->outerRadius;
	break;

      case SM_TYPE_SPHERE:
	minx = miny = minz = -model->radius;
	maxx = maxy = maxz = model->radius;
	break;

      case SM_TYPE_POLYGON:
      case SM_TYPE_LINE:
	if (model->ns > 0) {
	    empty = ComputeSurfaceBoundingBox(model->s, model->v, &minx, &miny, &minz, &maxx, &maxy, &maxz);

	    for (ns = 1, s = model->s+1; ns < (const) model->ns; ns++, s++) {
		if (!ComputeSurfaceBoundingBox(s, model->v, &cminx, &cminy, &cminz, &cmaxx, &cmaxy, &cmaxz)) {
		    if (empty) {
			minx = cminx;
			miny = cminy;
			minz = cminz;
			maxx = cmaxx;
			maxy = cmaxy;
			maxz = cmaxz;
		    }
		    else {
			if (cminx < minx) minx = cminx;
			if (cminy < miny) miny = cminy;
			if (cminz < minz) minz = cminz;
			if (cmaxx > maxx) maxx = cmaxx;
			if (cmaxy > maxy) maxy = cmaxy;
			if (cmaxz > maxz) maxz = cmaxz;
		    }
		    empty = 0;
		}
	    }
	}
	else {
	    empty = 1;
	}
	break;

      case SM_TYPE_POINT:
	if (model->nv > 0) {
	    u = model->v->lv;
	    minx = maxx = u[0];
	    miny = maxy = u[1];
	    minz = maxz = u[2];

	    for (nv = 1, v = model->v+1; nv < model->nv; nv++, v++) {
		u = v->lv;
		if (u[0] < minx) minx = u[0];
		if (u[0] > maxx) maxx = u[0];
		if (u[1] < miny) miny = u[1];
		if (u[1] > maxy) maxy = u[1];
		if (u[2] < minz) minz = u[2];
		if (u[2] > maxz) maxz = u[2];
	    }
	}
	else {
	    empty = 1;
	}
	break;
    }
    
    if (!(model->empty = empty)) {
	model->x0 = minx;
	model->y0 = miny;
	model->z0 = minz;
	model->x1 = maxx;
	model->y1 = maxy;
	model->z1 = maxz;
    }
    return empty;
}

/*
 *--------------------------------------------------------------
 *
 * SmGetBoundingBox:
 *
 * Find the bounding box for a model
 *
 *--------------------------------------------------------------
 */

int
SmGetBoundingBox(model, x0, y0, z0, x1, y1, z1, recursive)
     Model *model;
     DOUBLE *x0;
     DOUBLE *y0;
     DOUBLE *z0;
     DOUBLE *x1;
     DOUBLE *y1;
     DOUBLE *z1;
     int recursive;
{
    int i, empty;
    Model *child;
    DOUBLE x, y, z, gx, gy, gz;
    DOUBLE maxx, maxy, maxz, minx, miny, minz;
    DOUBLE cmaxx, cmaxy, cmaxz, cminx, cminy, cminz;
    DOUBLE gmaxx, gmaxy, gmaxz, gminx, gminy, gminz;

    if (!(empty = model->empty)) {
	minx = model->x0;
	miny = model->y0;
	minz = model->z0;
	maxx = model->x1;
	maxy = model->y1;
	maxz = model->z1;
    }

    if (recursive) {
	for (child = model->child; child; child = child->sibling) {
	    if (!SmGetBoundingBox(child, &gminx, &gminy, &gminz, &gmaxx, &gmaxy, &gmaxz, 1)) {

		/* convert child's bounding box from child's coordinate system to mine

		gminx = cminx * child->modelT[0] + cminy * child->modelT[4] + cminz * child->modelT[8] + child->modelT[12];
		gminy = cminx * child->modelT[1] + cminy * child->modelT[5] + cminz * child->modelT[9] + child->modelT[13];
		gminz = cminx * child->modelT[2] + cminy * child->modelT[6] + cminz * child->modelT[10] + child->modelT[14];
		gmaxx = cmaxx * child->modelT[0] + cmaxy * child->modelT[4] + cmaxz * child->modelT[8] + child->modelT[12];
		gmaxy = cmaxx * child->modelT[1] + cmaxy * child->modelT[5] + cmaxz * child->modelT[9] + child->modelT[13];
		gmaxz = cmaxx * child->modelT[2] + cmaxy * child->modelT[6] + cmaxz * child->modelT[10] + child->modelT[14];

		*/

		if (empty) {
		    minx = gminx;
		    miny = gminy;
		    minz = gminz;
		    maxx = gmaxx;
		    maxy = gmaxy;
		    maxz = gmaxz;
		}
		else {
		    if (gminx < minx) minx = gminx;
		    if (gminy < miny) miny = gminy;
		    if (gminz < minz) minz = gminz;
		    if (gmaxx > maxx) maxx = gmaxx;
		    if (gmaxy > maxy) maxy = gmaxy;
		    if (gmaxz > maxz) maxz = gmaxz;
		}
		empty = 0;
	    }
	}
    }
    if (!empty) {

	/* 0 0 0 */
	cmaxx = cminx = minx * model->modelT[0] + miny * model->modelT[4] + minz * model->modelT[8] + model->modelT[12];
	cmaxy = cminy = minx * model->modelT[1] + miny * model->modelT[5] + minz * model->modelT[9] + model->modelT[13];
	cmaxz = cminz = minx * model->modelT[2] + miny * model->modelT[6] + minz * model->modelT[10] + model->modelT[14];

	for (i = 1; i < 8; i++) {
	    if (i & 4) gx = maxx;
	    else gx = minx;

	    if (i & 2) gy = maxy;
	    else gy = miny;

	    if (i & 1) gz = maxz;
	    else gz = minz;

	    x = gx * model->modelT[0] + gy * model->modelT[4] + gz * model->modelT[8] + model->modelT[12];
	    y = gx * model->modelT[1] + gy * model->modelT[5] + gz * model->modelT[9] + model->modelT[13];
	    z = gx * model->modelT[2] + gy * model->modelT[6] + gz * model->modelT[10] + model->modelT[14];

	    if (x < cminx) cminx = x;
	    else if (x > cmaxx) cmaxx = x;
	    
	    if (y < cminy) cminy = y;
	    else if (y > cmaxy) cmaxy = y;

	    if (z < cminz) cminz = z;
	    else if (z > cmaxz) cmaxz = z;
	}

	*x0 = cminx;
	*y0 = cminy;
	*z0 = cminz;

	*x1 = cmaxx;
	*y1 = cmaxy;
	*z1 = cmaxz;
    }
    return empty;
}

/*
 *--------------------------------------------------------------
 *
 * ComputeSurfaceBoundingBox:
 *
 * compute the bounding box for a surface
 *
 *--------------------------------------------------------------
 */

static int
ComputeSurfaceBoundingBox(s, v, x0, y0, z0, x1, y1, z1)
     Surface *s;
     Vertex *v;
     DOUBLE *x0;
     DOUBLE *y0;
     DOUBLE *z0;
     DOUBLE *x1;
     DOUBLE *y1;
     DOUBLE *z1;
{
    int i;
    int *index;
    DOUBLE *u;
    DOUBLE minx, miny, minz;
    DOUBLE maxx, maxy, maxz;

    index = s->index;
    if (s->vcnt > 0) {
	minx = maxx = v[*index].lv[0];
	miny = maxy = v[*index].lv[1];
	minz = maxz = v[*index].lv[2];

	for (i = 1, index = s->index+1; i < s->vcnt; i++, index++) {
	    u = v[*index].lv;
	    if (u[0] < minx) minx = u[0];
	    if (u[0] > maxx) maxx = u[0];
	    if (u[1] < miny) miny = u[1];
	    if (u[1] > maxy) maxy = u[1];
	    if (u[2] < minz) minz = u[2];
	    if (u[2] > maxz) maxz = u[2];
	}
	*x0 = minx;
	*y0 = miny;
	*z0 = minz;
	*x1 = maxx;
	*y1 = maxy;
	*z1 = maxz;

	return 0;
    }
    else {
	return 1;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SetModelVertices
 *
 * set vertex coordinates
 *
 *--------------------------------------------------------------
 */

static int
SetModelVertices(interp, model, value, mode)
     Tcl_Interp *interp;
     Model *model;
     char *value;
     int mode;
{
    int i, argc, index, nitems, nv;
    char **items, **vec, **argv;
    double x, y, z;
    Vertex *v;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;

    }
    for (i = 0; i < argc; i++) {
	if (Tcl_SplitList(interp, argv[i], &nitems, &items) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (nitems != 2) {
	    Tcl_AppendResult(interp, "bad index-vertex pair \"", argv[i], "\"", (char *) NULL);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, items[0], &index) != TCL_OK) {
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if ((index < 1) || (index > model->nv)) {
	    Tcl_AppendResult(interp, "vertex index value out of bounds: \"", items[0], "\"", (char *) NULL);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (Tcl_SplitList(interp, items[1], &nv, &vec) != TCL_OK) {
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (nv != 3) {
	    Tcl_AppendResult(interp, "bad vertex coordinates: \"", items[1], "\"", (char *) NULL);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) vec);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if ((vec[0][0] && (Tcl_GetDouble(interp, vec[0], &x) != TCL_OK)) ||
	    (vec[1][0] && (Tcl_GetDouble(interp, vec[1], &y) != TCL_OK)) ||
	    (vec[2][0] && (Tcl_GetDouble(interp, vec[2], &z) != TCL_OK))) {
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) vec);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	v = model->v + (index-1);
	if (mode) {
	    /* absolute */
	    if (vec[0][0]) v->lv[0] = x;
	    if (vec[1][0]) v->lv[1] = y;
	    if (vec[2][0]) v->lv[2] = z;
	}
	else {
	    /* offset */
	    if (vec[0][0]) v->lv[0] += x;
	    if (vec[1][0]) v->lv[1] += y;
	    if (vec[2][0]) v->lv[2] += z;
	}
	(void) ckfree((void *) items);
	(void) ckfree((void *) vec);
    }
    (void) ckfree((void *) argv);
    (void) SmComputeVertexNormals(model, 0);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ColorValueByName:
 *
 * look up a color by name and return its rgb components
 *
 *--------------------------------------------------------------
 */

static int
ColorValueByName(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    XColor exactColor, screenColor;
    Tk_Window tkwin = (Tk_Window) clientData;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " color\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (!XLookupColor(Tk_Display(tkwin), Tk_Colormap(tkwin), argv[1], &exactColor, &screenColor)) {
	Tcl_AppendResult(interp, "bad color name \"", argv[1], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    else {
	sprintf(interp->result, "%u %u %u", exactColor.red, exactColor.green, exactColor.blue);
    }
    return TCL_OK;
}


