/* 
 * smoption.c --
 *
 *	This file implements model options.
 *
 */

#include "copyright.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <tk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "smInt.h"


/*
 * Prototypes for procedures defined externally:
 */

void		SmRemoveModelFromParent _ANSI_ARGS_ ((Model *model));

Model		*SmFindModel _ANSI_ARGS_ ((Cell *cell, char *label));

int		SmVectorParseProc _ANSI_ARGS_ ((ClientData clientData,
						Tcl_Interp *interp, Tk_Window tkwin,
						char *value, char *widgRec, int offset));

char		*SmVectorPrintProc _ANSI_ARGS_ ((ClientData clientData,
						 Tk_Window tkwin, char *widgRec, int offset,
						 Tcl_FreeProc **freeProcPtr));

CellHeader	*SmGetCellHeader _ANSI_ARGS_ ((Tcl_Interp *interp));

void		SmFreeTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, Texture *texture));


/*
 * Prototypes for procedures defined in this file that are exported:
 */


/*
 * Prototypes for procedures defined in this file that are static:
 */

static int	ModelParentParseProc _ANSI_ARGS_ ((ClientData clientData,
						   Tcl_Interp *interp, Tk_Window tkwin,
						   char *value, char *widgRec, int offset));

static char *	ModelParentPrintProc _ANSI_ARGS_ ((ClientData clientData,
						   Tk_Window tkwin, char *widgRec, int offset,
						   Tcl_FreeProc **freeProcPtr));

static int	ModelVerticesParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	ModelVerticesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	ModelSurfacesParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	ModelSurfacesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	ModelLinesParseProc _ANSI_ARGS_ ((ClientData clientData,
						  Tcl_Interp *interp, Tk_Window tkwin,
						  char *value, char *widgRec, int offset));

static char *	ModelLinesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						  Tk_Window tkwin, char *widgRec, int offset,
						  Tcl_FreeProc **freeProcPtr));

static int	ModelNormalsParseProc _ANSI_ARGS_ ((ClientData clientData,
						    Tcl_Interp *interp, Tk_Window tkwin,
						    char *value, char *widgRec, int offset));

static char *	ModelNormalsPrintProc _ANSI_ARGS_ ((ClientData clientData,
						    Tk_Window tkwin, char *widgRec, int offset,
						    Tcl_FreeProc **freeProcPtr));

static int	ModelStipplesParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	ModelStipplesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	ModelWidthsParseProc _ANSI_ARGS_ ((ClientData clientData,
						   Tcl_Interp *interp, Tk_Window tkwin,
						   char *value, char *widgRec, int offset));

static char *	ModelWidthsPrintProc _ANSI_ARGS_ ((ClientData clientData,
						   Tk_Window tkwin, char *widgRec, int offset,
						   Tcl_FreeProc **freeProcPtr));

static int	ModelSizesParseProc _ANSI_ARGS_ ((ClientData clientData,
						  Tcl_Interp *interp, Tk_Window tkwin,
						  char *value, char *widgRec, int offset));

static char *	ModelSizesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						  Tk_Window tkwin, char *widgRec, int offset,
						  Tcl_FreeProc **freeProcPtr));

static int	ModelMaterialsParseProc _ANSI_ARGS_ ((ClientData clientData,
						      Tcl_Interp *interp, Tk_Window tkwin,
						      char *value, char *widgRec, int offset));

static char *	ModelMaterialsPrintProc _ANSI_ARGS_ ((ClientData clientData,
						      Tk_Window tkwin, char *widgRec, int offset,
						      Tcl_FreeProc **freeProcPtr));

static int	ModelTexturesParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	ModelTexturesPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	ModelShadingParseProc _ANSI_ARGS_ ((ClientData clientData,
						    Tcl_Interp *interp, Tk_Window tkwin,
						    char *value, char *widgRec, int offset));

static char *	ModelShadingPrintProc _ANSI_ARGS_ ((ClientData clientData,
						    Tk_Window tkwin, char *widgRec, int offset,
						    Tcl_FreeProc **freeProcPtr));

static int	ModelCoefficientParseProc _ANSI_ARGS_ ((ClientData clientData,
							Tcl_Interp *interp, Tk_Window tkwin,
							char *value, char *widgRec, int offset));

static char *	ModelCoefficientPrintProc _ANSI_ARGS_ ((ClientData clientData,
							Tk_Window tkwin, char *widgRec, int offset,
							Tcl_FreeProc **freeProcPtr));

static int	ModelSpecularParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	ModelSpecularPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	PolygonModeParseProc _ANSI_ARGS_ ((ClientData clientData,
						   Tcl_Interp *interp, Tk_Window tkwin,
						   char *value, char *widgRec, int offset));

static char *	PolygonModePrintProc _ANSI_ARGS_ ((ClientData clientData,
						   Tk_Window tkwin, char *widgRec, int offset,
						   Tcl_FreeProc **freeProcPtr));

static int	PolygonCullFaceParseProc _ANSI_ARGS_ ((ClientData clientData,
						       Tcl_Interp *interp, Tk_Window tkwin,
						       char *value, char *widgRec, int offset));

static char *	PolygonCullFacePrintProc _ANSI_ARGS_ ((ClientData clientData,
						       Tk_Window tkwin, char *widgRec, int offset,
						       Tcl_FreeProc **freeProcPtr));

static int	QuadricDrawStyleParseProc _ANSI_ARGS_ ((ClientData clientData,
							Tcl_Interp *interp, Tk_Window tkwin,
							char *value, char *widgRec, int offset));

static char *	QuadricDrawStylePrintProc _ANSI_ARGS_ ((ClientData clientData,
							Tk_Window tkwin, char *widgRec, int offset,
							Tcl_FreeProc **freeProcPtr));

static int	QuadricNormalsParseProc _ANSI_ARGS_ ((ClientData clientData,
						      Tcl_Interp *interp, Tk_Window tkwin,
						      char *value, char *widgRec, int offset));

static char *	QuadricNormalsPrintProc _ANSI_ARGS_ ((ClientData clientData,
						      Tk_Window tkwin, char *widgRec, int offset,
						      Tcl_FreeProc **freeProcPtr));

static int	QuadricOrientationParseProc _ANSI_ARGS_ ((ClientData clientData,
							  Tcl_Interp *interp, Tk_Window tkwin,
							  char *value, char *widgRec, int offset));

static char *	QuadricOrientationPrintProc _ANSI_ARGS_ ((ClientData clientData,
							  Tk_Window tkwin, char *widgRec, int offset,
							  Tcl_FreeProc **freeProcPtr));


/*
 * Custom options for cell models
 */

static Tk_CustomOption sm_ModelOptionCULL = {
    PolygonCullFaceParseProc,
    PolygonCullFacePrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption sm_ModelOptionFRONT = {
    PolygonModeParseProc,
    PolygonModePrintProc,
    (ClientData) GL_FRONT,
};

static Tk_CustomOption sm_ModelOptionBACK = {
    PolygonModeParseProc,
    PolygonModePrintProc,
    (ClientData) GL_BACK,
};

static Tk_CustomOption sm_ModelOptionFWD = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_MODEL_FWD,
};

static Tk_CustomOption sm_ModelOptionUP = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_MODEL_UP,
};

static Tk_CustomOption sm_ModelOptionPOS = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_MODEL_POS,
};

static Tk_CustomOption sm_ModelOptionParent = {
    ModelParentParseProc,
    ModelParentPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionVertices = {
    ModelVerticesParseProc,
    ModelVerticesPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionSurfaces = {
    ModelSurfacesParseProc,
    ModelSurfacesPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionNormals = {
    ModelNormalsParseProc,
    ModelNormalsPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionStipples = {
    ModelStipplesParseProc,
    ModelStipplesPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionWidths = {
    ModelWidthsParseProc,
    ModelWidthsPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionSizes = {
    ModelSizesParseProc,
    ModelSizesPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionMaterials = {
    ModelMaterialsParseProc,
    ModelMaterialsPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionTextures = {
    ModelTexturesParseProc,
    ModelTexturesPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionShading = {
    ModelShadingParseProc,
    ModelShadingPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionAmbient = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_AMBIENT,
};

static Tk_CustomOption sm_ModelOptionDiffuse = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_DIFFUSE,
};

static Tk_CustomOption sm_ModelOptionEmissive = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_EMISSIVE,
};

static Tk_CustomOption sm_ModelOptionSpecular = {
    ModelSpecularParseProc,
    ModelSpecularPrintProc,
    (ClientData) SM_MODEL_SPECULAR,
};

static Tk_CustomOption sm_ModelOptionBFAmbient = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_BF_AMBIENT,
};

static Tk_CustomOption sm_ModelOptionBFDiffuse = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_BF_DIFFUSE,
};

static Tk_CustomOption sm_ModelOptionBFEmissive = {
    ModelCoefficientParseProc,
    ModelCoefficientPrintProc,
    (ClientData) SM_MODEL_BF_EMISSIVE,
};

static Tk_CustomOption sm_ModelOptionBFSpecular = {
    ModelSpecularParseProc,
    ModelSpecularPrintProc,
    (ClientData) SM_MODEL_BF_SPECULAR,
};

static Tk_CustomOption sm_QuadricOptionDrawStyle = {
    QuadricDrawStyleParseProc,
    QuadricDrawStylePrintProc,
    NULL,
};

static Tk_CustomOption sm_QuadricOptionNormals = {
    QuadricNormalsParseProc,
    QuadricNormalsPrintProc,
    NULL,
};

static Tk_CustomOption sm_QuadricOptionOrientation = {
    QuadricOrientationParseProc,
    QuadricOrientationPrintProc,
    NULL,
};

static Tk_CustomOption sm_ModelOptionLines = {
    ModelLinesParseProc,
    ModelLinesPrintProc,
    NULL,
};


/*
 * Information used for parsing polygon model configuration specs:
 */

Tk_ConfigSpec modelPolygonConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_BOOLEAN, "-average", (char *) NULL, (char *) NULL,
       "off", Tk_Offset(Model, useAverage), 0},

    {TK_CONFIG_CUSTOM, "-backmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionBACK},

    {TK_CONFIG_CUSTOM, "-bfambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionBFAmbient},

    {TK_CONFIG_COLOR, "-bfcolor", "surfaceColor", "SurfaceColor",
       "black", Tk_Offset(Model, bfcolor), },

    {TK_CONFIG_CUSTOM, "-bfdiffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionBFDiffuse},

    {TK_CONFIG_CUSTOM, "-bfemissive", "emissiveCoefficient", "EmissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionBFEmissive},

    {TK_CONFIG_CUSTOM, "-bfspecular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionBFSpecular},

    {TK_CONFIG_BOOLEAN, "-ccw", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, ccw), 0},

    {TK_CONFIG_COLOR, "-color", "surfaceColor", "SurfaceColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-cull", (char *) NULL, (char *) NULL,
       "back", 0, 0, &sm_ModelOptionCULL},

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "emissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-frontmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionFRONT},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_CUSTOM, "-materials", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionMaterials},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionNormals},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "smooth", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},

    {TK_CONFIG_CUSTOM, "-surfaces", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionSurfaces},

    {TK_CONFIG_CUSTOM, "-textures", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionTextures},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-usetexture", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, useTexture), 0},

    {TK_CONFIG_CUSTOM, "-vertices", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionVertices},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Information used for parsing line model configuration specs:
 */

Tk_ConfigSpec modelLineConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_COLOR, "-color", "lineColor", "LineColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "emissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_CUSTOM, "-lines", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionLines},

    {TK_CONFIG_INT, "-linestipple", "lineStipple", "LineStipple",
       "0xffff", Tk_Offset(Model, lineStipple), 0},

    {TK_CONFIG_INT, "-linestipplefactor", "lineStippleFactor", "LineStippleFactor",
       "0", Tk_Offset(Model, lineFactor), 0},

    {TK_CONFIG_DOUBLE, "-linewidth", "lineWidth", "LineWidth",
       "1.0", Tk_Offset(Model, lineWidth), 0},

    {TK_CONFIG_CUSTOM, "-materials", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionMaterials},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionNormals},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "flat", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},
 
    {TK_CONFIG_CUSTOM, "-stipples", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionStipples},

    {TK_CONFIG_CUSTOM, "-textures", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionTextures},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-usetexture", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, useTexture), 0},

    {TK_CONFIG_CUSTOM, "-vertices", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionVertices},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_CUSTOM, "-widths", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionWidths},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Information used for parsing point model configuration specs:
 */

Tk_ConfigSpec modelPointConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_COLOR, "-color", "lineColor", "LineColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "EmissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_DOUBLE, "-pointsize", "pointSize", "PointSize",
       "1.0", Tk_Offset(Model, pointSize), 0},

    {TK_CONFIG_CUSTOM, "-materials", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionMaterials},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionNormals},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "flat", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_CUSTOM, "-sizes", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionSizes},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},
 
    {TK_CONFIG_CUSTOM, "-textures", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionTextures},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-usetexture", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, useTexture), 0},

    {TK_CONFIG_CUSTOM, "-vertices", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionVertices},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Information used for parsing cylinder model configuration specs:
 */

Tk_ConfigSpec modelCylinderConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_CUSTOM, "-backmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionBACK},

    {TK_CONFIG_DOUBLE, "-baseradius", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(Model, baseRadius), 0},

    {TK_CONFIG_CUSTOM, "-bfambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionBFAmbient},

    {TK_CONFIG_COLOR, "-bfcolor", "surfaceColor", "SurfaceColor",
       "black", Tk_Offset(Model, bfcolor), },

    {TK_CONFIG_CUSTOM, "-bfdiffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionBFDiffuse},

    {TK_CONFIG_CUSTOM, "-bfemissive", "emissiveCoefficient", "EmissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionBFEmissive},

    {TK_CONFIG_CUSTOM, "-bfspecular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionBFSpecular},

    {TK_CONFIG_BOOLEAN, "-ccw", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, ccw), 0},

    {TK_CONFIG_COLOR, "-color", "surfaceColor", "SurfaceColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-cull", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_ModelOptionCULL},

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-drawstyle", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_QuadricOptionDrawStyle},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "emissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-frontmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionFRONT},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, height), 0},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_QuadricOptionNormals},

    {TK_CONFIG_CUSTOM, "-orientation", (char *) NULL, (char *) NULL,
       "outside", 0, 0, &sm_QuadricOptionOrientation},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "flat", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_INT, "-slices", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, slices), 0},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},

    {TK_CONFIG_INT, "-stacks", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(Model, stacks), 0},

    {TK_CONFIG_DOUBLE, "-topradius", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(Model, topRadius), 0},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Information used for parsing disk model configuration specs:
 */

Tk_ConfigSpec modelDiskConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_CUSTOM, "-backmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionBACK},

    {TK_CONFIG_CUSTOM, "-bfambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionBFAmbient},

    {TK_CONFIG_COLOR, "-bfcolor", "surfaceColor", "SurfaceColor",
       "black", Tk_Offset(Model, bfcolor), },

    {TK_CONFIG_CUSTOM, "-bfdiffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionBFDiffuse},

    {TK_CONFIG_CUSTOM, "-bfemissive", "emissiveCoefficient", "EmissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionBFEmissive},

    {TK_CONFIG_CUSTOM, "-bfspecular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionBFSpecular},

    {TK_CONFIG_BOOLEAN, "-ccw", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, ccw), 0},

    {TK_CONFIG_COLOR, "-color", "surfaceColor", "SurfaceColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-cull", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_ModelOptionCULL},

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-drawstyle", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_QuadricOptionDrawStyle},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "emissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-frontmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionFRONT},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_DOUBLE, "-innerradius", (char *) NULL, (char *) NULL,
       "5", Tk_Offset(Model, innerRadius), 0},

    {TK_CONFIG_INT, "-loops", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(Model, loops), 0},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_QuadricOptionNormals},

    {TK_CONFIG_CUSTOM, "-orientation", (char *) NULL, (char *) NULL,
       "outside", 0, 0, &sm_QuadricOptionOrientation},

    {TK_CONFIG_DOUBLE, "-outerradius", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, outerRadius), 0},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "flat", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_INT, "-slices", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, slices), 0},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Information used for parsing sphere model configuration specs:
 */

Tk_ConfigSpec modelSphereConfigSpecs[] = {

    {TK_CONFIG_CUSTOM, "-ambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionAmbient},

    {TK_CONFIG_CUSTOM, "-backmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionBACK},

    {TK_CONFIG_CUSTOM, "-bfambient", "ambientCoefficient", "AmbientCoefficient",
       "0.2 0.2 0.2", 0, 0, &sm_ModelOptionBFAmbient},

    {TK_CONFIG_COLOR, "-bfcolor", "surfaceColor", "SurfaceColor",
       "black", Tk_Offset(Model, bfcolor), },

    {TK_CONFIG_CUSTOM, "-bfdiffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionBFDiffuse},

    {TK_CONFIG_CUSTOM, "-bfemissive", "emissiveCoefficient", "EmissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionBFEmissive},

    {TK_CONFIG_CUSTOM, "-bfspecular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionBFSpecular},

    {TK_CONFIG_BOOLEAN, "-ccw", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, ccw), 0},

    {TK_CONFIG_COLOR, "-color", "surfaceColor", "SurfaceColor",
       "white", Tk_Offset(Model, color), },

    {TK_CONFIG_CUSTOM, "-cull", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_ModelOptionCULL},

    {TK_CONFIG_CUSTOM, "-diffuse", "diffuseCoefficient", "DiffuseCoefficient",
       "0.8 0.8 0.8", 0, 0, &sm_ModelOptionDiffuse},

    {TK_CONFIG_CUSTOM, "-drawstyle", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_QuadricOptionDrawStyle},

    {TK_CONFIG_CUSTOM, "-emissive", "emissiveCoefficient", "emissiveCoefficient",
       "0 0 0", 0, 0, &sm_ModelOptionEmissive},

    {TK_CONFIG_CUSTOM, "-frontmode", (char *) NULL, (char *) NULL,
       "fill", 0, 0, &sm_ModelOptionFRONT},

    {TK_CONFIG_CUSTOM, "-fwd", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &sm_ModelOptionFWD},

    {TK_CONFIG_CUSTOM, "-normals", (char *) NULL, (char *) NULL,
       "none", 0, 0, &sm_QuadricOptionNormals},

    {TK_CONFIG_CUSTOM, "-orientation", (char *) NULL, (char *) NULL,
       "outside", 0, 0, &sm_QuadricOptionOrientation},

    {TK_CONFIG_CUSTOM, "-parent", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &sm_ModelOptionParent},

    {TK_CONFIG_CUSTOM, "-position", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0", 0, 0, &sm_ModelOptionPOS},

    {TK_CONFIG_DOUBLE, "-radius", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, radius), 0},

    {TK_CONFIG_CUSTOM, "-shading", (char *) NULL, (char *) NULL,
       "flat", 0, 0, &sm_ModelOptionShading},

    {TK_CONFIG_INT, "-slices", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, slices), 0},

    {TK_CONFIG_CUSTOM, "-specular", "specularReflection", "SpecularReflection",
       "0.0 0.0 0.0 0.0", 0, 0, &sm_ModelOptionSpecular},

    {TK_CONFIG_INT, "-stacks", (char *) NULL, (char *) NULL,
       "10", Tk_Offset(Model, stacks), 0},

    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &sm_ModelOptionUP},

    {TK_CONFIG_BOOLEAN, "-visible", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(Model, visible), 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};



/*
 *--------------------------------------------------------------
 *
 * ModelParentParseProc:
 *
 * process the '-parent' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelParentParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *value;
    char *widgRec;
    int offset;
{
    Model *p, *parent, *model = (Model *) widgRec;

    if (*value) {
	if ((parent = SmFindModel(model->cell, value)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", value, "\" not found.", (char *) NULL);
	    return TCL_ERROR;
	}

	/* make sure we don't introduce a cycle in the model hierarchy */
	for (p = parent->parent; p; p = p->parent) {
	    if (p == model) {
		Tcl_AppendResult(interp, "cycle in model hierarchy: ", parent->label, " -> ", model->label, ".",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	}

	SmRemoveModelFromParent(model);
	model->parent = parent;
	model->sibling = parent->child;
	parent->child = model;
    } else {
	SmRemoveModelFromParent(model);
	model->parent = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_POSITION;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelParentPrintProc:
 *
 * produce a printable string for the '-parent' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelParentPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *p = ((Model *) widgRec)->parent;

    *freeProcPtr = NULL;

    if (p) {
	return p->label;
    } else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelVerticesParseProc:
 *
 * process the '-vertices' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelVerticesParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *value;
    char *widgRec;
    int offset;
{
    int i, argc, nc;
    char **argv, **comp;
    double lv0, lv1, lv2;
    Model *model = (Model *) widgRec;
    Vertex *v;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    if (argc == 0) {
	if (model->v) {
	    (void) ckfree((void *) model->v);
	}
	model->nv = 0;
	(void) ckfree((void *) argv);
	model->updateFlag |= SM_MODEL_UPDATE_NORMALS + \
			     SM_MODEL_UPDATE_MATERIALS + \
			     SM_MODEL_UPDATE_POSITION + \
			     SM_MODEL_UPDATE_TEXTURES + \
			     SM_MODEL_UPDATE_BBOX;
	return TCL_OK;
    }

    if ((v = (Vertex *) ckalloc(argc * sizeof(Vertex))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for vertex list.", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    for (i = 0; i < (const) argc; i++) {

	if (Tcl_SplitList(interp, argv[i], &nc, &comp) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) v);
	    return TCL_ERROR;
	}

	if (nc != 3) {
	    Tcl_AppendResult(interp, "bad vertex value \"", argv[i], "\": three components expected.", (char *) NULL);
	    (void) ckfree((void *) comp);
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) v);
	    return TCL_ERROR;
	}

	if (Tcl_GetDouble(interp, comp[0], &lv0) != TCL_OK ||
	    Tcl_GetDouble(interp, comp[1], &lv1) != TCL_OK ||
	    Tcl_GetDouble(interp, comp[2], &lv2) != TCL_OK) {
	    (void) ckfree((void *) comp);
	    (void) ckfree((void *) argv);
	    (void) ckfree((void *) v);
	    return TCL_ERROR;
	}
	v[i].lv[0] = (DOUBLE) lv0;
	v[i].lv[1] = (DOUBLE) lv1;
	v[i].lv[2] = (DOUBLE) lv2;
	(void) ckfree((void *) comp);
    }
    model->nv = argc;
    if (model->v) (void) ckfree((void *) model->v);
    (void) ckfree((void *) argv);
    model->v = v;
    model->updateFlag |= SM_MODEL_UPDATE_NORMALS + \
			 SM_MODEL_UPDATE_MATERIALS + \
			 SM_MODEL_UPDATE_POSITION + \
			 SM_MODEL_UPDATE_TEXTURES + \
			 SM_MODEL_UPDATE_BBOX;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelVerticesPrintProc:
 *
 * produce a printable string for the '-vertices' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelVerticesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i;
    char *buf;
    Vertex *v;
    Model *model = (Model *) widgRec;
    Tcl_Interp *interp = model->cell->interp;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE], c2[TCL_DOUBLE_SPACE];

    if (model->nv == 0) {
	*freeProcPtr = NULL;
	return "";
    }

    if ((buf = (char *) ckalloc(model->nv * 80)) == NULL) {
	*freeProcPtr = NULL;
	return "";
    }

    v = model->v;
    *buf = '\0';
    for (i = 0; i < (const) model->nv - 1; i++) {
	Tcl_PrintDouble(interp, (double) v[i].lv[0], c0);
	Tcl_PrintDouble(interp, (double) v[i].lv[1], c1);
	Tcl_PrintDouble(interp, (double) v[i].lv[2], c2);
	sprintf(buf + strlen(buf), "{%s %s %s} ", c0, c1, c2);
    }
    Tcl_PrintDouble(interp, (double) v[i].lv[0], c0);
    Tcl_PrintDouble(interp, (double) v[i].lv[1], c1);
    Tcl_PrintDouble(interp, (double) v[i].lv[2], c2);
    sprintf(buf + strlen(buf), "{%s %s %s}", c0, c1, c2);

    *freeProcPtr = (Tcl_FreeProc *) free;
    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * ModelSurfacesParseProc:
 *
 * process the '-surfaces' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelSurfacesParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int i, j, index, argc, nc;
    char **argv = NULL;
    char **comp = NULL;
    Surface *s = NULL;
    Model *model = (Model *) widgRec;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    
    if (model->s) {
	for (s = model->s, i = 0; i < (const) model->ns; i++) {
	    (void) ckfree((void *) s[i].index);
	    if (s[i].lvn) (void) ckfree((void *) s[i].lvn);
	    if (s[i].texcoords) (void) ckfree((void *) s[i].texcoords);
	    if (s[i].materials) (void) ckfree((void *) s[i].materials);
	    if (s[i].texture) {
		Tcl_Interp *interp = model->cell->interp;
		SmFreeTexture(interp, SmGetCellHeader(interp), s[i].texture);
	    }
	}
	(void) ckfree((void *) model->s);
	model->s = NULL;
    }
    if (argc == 0) {
	model->ns = 0;
	(void) ckfree((void *) argv);
	model->updateFlag |= SM_MODEL_UPDATE_NORMALS + \
			     SM_MODEL_UPDATE_MATERIALS + \
			     SM_MODEL_UPDATE_POSITION + \
			     SM_MODEL_UPDATE_TEXTURES + \
			     SM_MODEL_UPDATE_BBOX;
	return TCL_OK;
    }
    
    if ((model->s = (Surface *) ckalloc(argc * sizeof(Surface))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for surface list.", (char *) NULL);
	goto error;
    }

    for (s = model->s, i = 0; i < (const) argc; i++, s++) {
	s->index = NULL;
    }

    for (s = model->s, i = 0; i < (const) argc; i++, s++) {
	
	if (Tcl_SplitList(interp, argv[i], &nc, &comp) != TCL_OK) {
	    goto error;
	}

	if (nc < 3) {
	    Tcl_AppendResult(interp, "each surface must contain at least three component.", (char *) NULL);
	    goto error;
	}
	
	if ((s->index = (int *) ckalloc(nc * sizeof(int))) == NULL) {
	    Tcl_AppendResult(interp, "failed to allocate memory for surface vertex indices.", (char *) NULL);
	    goto error;
	}
	
	for (j = 0; j < (const) nc; j++) {
	    if (Tcl_GetInt(interp, comp[j], &index) != TCL_OK) {
		goto error;
	    }
	    if (index < 1) {
		Tcl_AppendResult(interp, "surface vertex index must be positive.", (char *) NULL);
		goto error;
	    }
	    s->index[j] = index - 1;
	}
	if (s->index[0] == s->index[nc-1]) {
	    if (nc == 3) {
		Tcl_AppendResult(interp, "each surface must contain at least three distinct component.", (char *) NULL);
		goto error;
	    }
	    nc--;
	}
	s->id = i + 1;
	s->vcnt = nc;
	s->model = model;
	s->normal = SM_VERTEX_NORMAL_DEFAULT;
	s->texcoords = NULL;
	s->lvn = NULL;
	s->materials = NULL;
	s->texture = NULL;
	
	(void) ckfree((void *) comp);
	comp = NULL;
    }
    
    (void) ckfree((void *) argv);
    model->ns = argc;
    model->updateFlag |= SM_MODEL_UPDATE_NORMALS + \
		         SM_MODEL_UPDATE_MATERIALS + \
			 SM_MODEL_UPDATE_POSITION + \
			 SM_MODEL_UPDATE_TEXTURES + \
			 SM_MODEL_UPDATE_BBOX;
    return TCL_OK;
    
  error:
    (void) ckfree((void *) argv);
    if (comp) (void) ckfree((void *) comp);
    if (model->s) {
	for (i = 0, s = model->s; i < (const) argc; i++, s++) {
	    if (s->index) (void) ckfree((void *) s->index);
	}
	(void) ckfree((void *) model->s);
	model->s = NULL;
    }
    model->ns = 0;
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ModelSurfacesPrintProc:
 *
 * produce a printable string for the '-surfaces' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelSurfacesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, j;
    int cnt = 0;
    char *buf;
    Surface *s;
    Model *model = (Model *) widgRec;

    if (model->ns == 0) {
	*freeProcPtr = NULL;
	return "";
    }

    for (s = model->s, i = 0; i < (const) model->ns; i++, s++)
	cnt += s->vcnt;

    if (cnt > 0) {
	if ((buf = (char *) ckalloc(cnt * 12)) == NULL) {
	    *freeProcPtr = NULL;
	    return "";
	}

	*buf = '\0';
	for (s = model->s, i = 0; i < (const) model->ns - 1; i++, s++) {
	    strcat(buf, "{");
	    for (j = 0; j < (const) s->vcnt - 1; j++) {
		sprintf(buf + strlen(buf), "%u ", s->index[j] + 1);
	    }
	    sprintf(buf + strlen(buf), "%u} ", s->index[j] + 1);
	}
	strcat(buf, "{");
	for (j = 0; j < (const) s->vcnt - 1; j++) {
	    sprintf(buf + strlen(buf), "%u ", s->index[j] + 1);
	}
	sprintf(buf + strlen(buf), "%u}", s->index[j] + 1);
	
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelLinesParseProc:
 *
 * process the '-lines' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelLinesParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *value;
    char *widgRec;
    int offset;
{
    int i, j, index, argc, nc, ns;
    char **argv = NULL;
    char **comp = NULL;
    Surface *lines = NULL;
    Model *model = (Model *) widgRec;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    if (model->s) {
	for (lines = model->s, i = 0; i < (const) model->ns; i++) {
	    (void) ckfree((void *) lines[i].index);
	    if (lines[i].texcoords) (void) ckfree((void *) lines[i].texcoords);
	    if (lines[i].lvn) (void) ckfree((void *) lines[i].lvn);
	    if (lines[i].materials) (void) ckfree((void *) lines[i].materials);
	    if (lines[i].texture) {
		Tcl_Interp *interp = model->cell->interp;
		SmFreeTexture(interp, SmGetCellHeader(interp), lines[i].texture);
	    }
	}
	(void) ckfree((void *) model->s);
	model->s = NULL;
    }
    if (argc == 0) {
	model->ns = 0;
	(void) ckfree((void *) argv);
	return TCL_OK;
    }
  
    if ((model->s = (Surface *) ckalloc(argc * sizeof(Surface))) == NULL) {
	Tcl_AppendResult(interp, "failed to allocate memory for line segments.", (char *) NULL);
	goto error;
    }

    for (lines = model->s, i = 0; i < (const) argc; i++, lines++) {
	lines->index = NULL;
    }

    ns = 0;
    for (lines = model->s, i = 0; i < (const) argc; i++, lines++) {
    
	if (Tcl_SplitList(interp, argv[i], &nc, &comp) != TCL_OK) {
	    goto error;
	}

	if (nc > 0) {
	    if ((lines->index = (int *) ckalloc(nc * sizeof(int))) == NULL) {
		Tcl_AppendResult(interp, "failed to allocate memory for line vertex indices.", (char *) NULL);
		goto error;
	    }
    
	    for (j = 0; j < (const) nc; j++) {
		if (Tcl_GetInt(interp, comp[j], &index) != TCL_OK) {
		    goto error;
		}
		if (index < 1) {
		    Tcl_AppendResult(interp, "line vertex index must be positive.", (char *) NULL);
		    goto error;
		}
		lines->index[j] = index - 1;
	    }
	    lines->id = i + 1;
	    lines->vcnt = nc;
	    lines->model = model;
	    lines->normal = SM_VERTEX_NORMAL_DEFAULT;
	    lines->texcoords = NULL;
	    lines->lvn = NULL;
	    lines->materials = NULL;
	    lines->texture = NULL;
	    ns++;
	}
	(void) ckfree((void *) comp);
	comp = NULL;
    }
  
    (void) ckfree((void *) argv);
    model->ns = ns;
    model->updateFlag |= SM_MODEL_UPDATE_NORMALS + \
			 SM_MODEL_UPDATE_MATERIALS + \
			 SM_MODEL_UPDATE_POSITION + \
			 SM_MODEL_UPDATE_TEXTURES + \
			 SM_MODEL_UPDATE_WIDTHS + \
			 SM_MODEL_UPDATE_STIPPLES;
    return TCL_OK;

  error:
    (void) ckfree((void *) argv);
    if (comp) (void) ckfree((void *) comp);
    if (model->s) {
	for (lines = model->s, i = 0; i < (const) argc; i++, lines++) {
	    if (lines->index) (void) ckfree((void *) lines->index);
	}
	(void) ckfree((void *) model->s);
    }
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ModelLinesPrintProc:
 *
 * produce a printable string for the '-lines' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelLinesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, j;
    int cnt = 0;
    char *buf;
    Surface *s;
    Model *model = (Model *) widgRec;

    if (model->ns == 0) {
	*freeProcPtr = NULL;
	return "";
    }

    for (s = model->s, i = 0; i < (const) model->ns; i++, s++)
	cnt += s->vcnt;

    if (cnt > 0) {
	if ((buf = (char *) ckalloc(cnt * 12)) == NULL) {
	    *freeProcPtr = NULL;
	    return "";
	}

	*buf = '\0';
	for (s = model->s, i = 0; i < (const) model->ns - 1; i++, s++) {
	    strcat(buf, "{");
	    for (j = 0; j < (const) s->vcnt - 1; j++) {
		sprintf(buf + strlen(buf), "%u ", s->index[j] + 1);
	    }
	    sprintf(buf + strlen(buf), "%u} ", s->index[j] + 1);
	}
	strcat(buf, "{");
	for (j = 0; j < (const) s->vcnt - 1; j++) {
	    sprintf(buf + strlen(buf), "%u ", s->index[j] + 1);
	}
	sprintf(buf + strlen(buf), "%u}", s->index[j] + 1);
	
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelMaterialsParseProc:
 *
 * process the '-materials' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelMaterialsParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->materials) {
	(void) ckfree((void *) model->materials);
    }

    if ((*value)) {
	if ((model->materials = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->materials, value);
	}
    }
    else {
	model->materials = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_MATERIALS;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelMaterialsPrintProc:
 *
 * produce a printable string for the '-materials' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelMaterialsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, j, len;
    char *buf;
    char r[TCL_DOUBLE_SPACE];
    char g[TCL_DOUBLE_SPACE];
    char b[TCL_DOUBLE_SPACE];
    char shininess[TCL_DOUBLE_SPACE];
    Model *model = (Model *) widgRec;
    Surface *s;
    Material *mat;
    Tcl_Interp *interp = model->cell->interp;

    len = 0;
    for (i = 0, s = model->s; i < (const) model->ns; i++, s++) {
	if (s->materials) {
	    len += s->vcnt;
	}
    }

    if ((len > 0) && (buf = (char *) ckalloc((TCL_DOUBLE_SPACE * 13 + 30) * len))) {
	*buf = '\0';
	for (i = 1, s = model->s; i <= (const) model->ns; i++, s++) {
	    if (s->materials) {
		if (*buf) {
		    sprintf(buf + strlen(buf), " {%d ", i);
		}
		else {
		    sprintf(buf, "{%d ", i);
		}
		for (j = 0, mat = s->materials; j < (const) s->vcnt - 1; j++, mat++) {
		    Tcl_PrintDouble(interp, (double) mat->shininess, shininess);

		    Tcl_PrintDouble(interp, (double) mat->ambientRGBA[0], r);
		    Tcl_PrintDouble(interp, (double) mat->ambientRGBA[1], g);
		    Tcl_PrintDouble(interp, (double) mat->ambientRGBA[2], b);
		    sprintf(buf + strlen(buf), "{{%s %s %s} ", r, g, b);

		    Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[0], r);
		    Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[1], g);
		    Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[2], b);
		    sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

		    Tcl_PrintDouble(interp, (double) mat->specularRGBA[0], r);
		    Tcl_PrintDouble(interp, (double) mat->specularRGBA[1], g);
		    Tcl_PrintDouble(interp, (double) mat->specularRGBA[2], b);
		    sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

		    Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[0], r);
		    Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[1], g);
		    Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[2], b);
		    sprintf(buf + strlen(buf), "{%s %s %s} %s} ", r, g, b, shininess);
		}
		Tcl_PrintDouble(interp, (double) mat->shininess, shininess);
		Tcl_PrintDouble(interp, (double) mat->ambientRGBA[0], r);
		Tcl_PrintDouble(interp, (double) mat->ambientRGBA[1], g);
		Tcl_PrintDouble(interp, (double) mat->ambientRGBA[2], b);
		sprintf(buf + strlen(buf), "{{%s %s %s} ", r, g, b);

		Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[0], r);
		Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[1], g);
		Tcl_PrintDouble(interp, (double) mat->diffuseRGBA[2], b);
		sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

		Tcl_PrintDouble(interp, (double) mat->specularRGBA[0], r);
		Tcl_PrintDouble(interp, (double) mat->specularRGBA[1], g);
		Tcl_PrintDouble(interp, (double) mat->specularRGBA[2], b);
		sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

		Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[0], r);
		Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[1], g);
		Tcl_PrintDouble(interp, (double) mat->emissiveRGBA[2], b);
		sprintf(buf + strlen(buf), "{%s %s %s} %s}} ", r, g, b, shininess);
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelNormalsParseProc:
 *
 * process the '-normals' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelNormalsParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->normals) {
	(void) ckfree((void *) model->normals);
    }

    if ((*value)) {
	if ((model->normals = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->normals, value);
	}
    }
    else {
	model->normals = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_NORMALS;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelNormalsPrintProc:
 *
 * produce a printable string for the '-normals' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelNormalsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, j, len;
    char *buf;
    Normal3D *lvn;
    Model *model = (Model *) widgRec;
    Surface *s = model->s;
    Tcl_Interp *interp = model->cell->interp;
    char x[TCL_DOUBLE_SPACE], y[TCL_DOUBLE_SPACE], z[TCL_DOUBLE_SPACE];

    len = 0;
    for (i = 0, s = model->s; i < (const) model->ns; i++, s++) {
	if (s->normal) {
	    len += s->vcnt;
	}
    }

    if ((len > 0) && (buf = (char *) ckalloc(20 * model->ns + (TCL_DOUBLE_SPACE + 5) * len))) {
	*buf = '\0';
	for (i = 1, s = model->s; i <= (const) model->ns; i++, s++) {
	    if (s->normal) {
		if (*buf) {
		    sprintf(buf + strlen(buf), " {%d ", i);
		}
		else {
		    sprintf(buf, "{%d ", i);
		}
		for (j = 0, lvn = s->lvn; j < (const) s->vcnt - 1; j++, lvn++) {
		    Tcl_PrintDouble(interp, (double) lvn->v[0], x);
		    Tcl_PrintDouble(interp, (double) lvn->v[1], y);
		    Tcl_PrintDouble(interp, (double) lvn->v[2], z);
		    sprintf(buf + strlen(buf), "{%s %s %s} ", x, y, z);
		}
		Tcl_PrintDouble(interp, (double) lvn->v[0], x);
		Tcl_PrintDouble(interp, (double) lvn->v[1], y);
		Tcl_PrintDouble(interp, (double) lvn->v[2], z);
		sprintf(buf + strlen(buf), "{%s %s %s}}", x, y, z);
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelStipplesParseProc:
 *
 * process the '-stipples' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelStipplesParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->stipples) {
	(void) ckfree((void *) model->stipples);
    }

    if ((*value)) {
	if ((model->stipples = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->stipples, value);
	}
    }
    else {
	model->stipples = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_STIPPLES;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelStipplesPrintProc:
 *
 * produce a printable string for the '-stipples' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelStipplesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, len;
    char *buf;
    Model *model = (Model *) widgRec;
    Surface *line;

    len = 0;
    for (line = model->s, i = 0; i < (const) model->ns; line++, i++) {
	if (!line->defaultStipple) {
	    len++;
	}
    }
    if ((len > 0) && (buf = (char *) ckalloc(len * 40))) {
	*buf = '\0';
	for (line = model->s, i = 1; i <= (const) model->ns; line++, i++) {
	    if (!line->defaultStipple) {
		sprintf(buf + strlen(buf), "{%d 0x%04x %d} ", i, line->lineStipple, line->lineFactor);
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelWidthsParseProc:
 *
 * process the '-widths' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelWidthsParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->widths) {
	(void) ckfree((void *) model->widths);
    }

    if ((*value)) {
	if ((model->widths = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->widths, value);
	}
    }
    else {
	model->widths = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_WIDTHS;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelWidthsPrintProc:
 *
 * produce a printable string for the '-widths' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelWidthsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, len;
    char *buf;
    Model *model = (Model *) widgRec;
    Surface *line;

    len = 0;
    for (line = model->s, i = 0; i < (const) model->ns; line++, i++) {
	if (!line->defaultWidth) {
	    len++;
	}
    }
    if ((len > 0) && (buf = (char *) ckalloc(len * 25))) {
	*buf = '\0';
	for (line = model->s, i = 1; i <= (const) model->ns; line++, i++) {
	    if (!line->defaultWidth) {
		sprintf(buf + strlen(buf), "{%d %f} ", i, line->lineWidth);
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelSizesParseProc:
 *
 * process the '-sizes' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelSizesParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->sizes) (void) ckfree((void *) model->sizes);

    if ((*value)) {
	if ((model->sizes = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->sizes, value);
	}
    }
    else {
	model->sizes = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_SIZES;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelSizesPrintProc:
 *
 * produce a printable string for the '-sizes' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelSizesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i;
    char *buf, size[TCL_DOUBLE_SPACE];
    DOUBLE *sizes;
    Model *model = (Model *) widgRec;
    Tcl_Interp *interp = model->cell->interp;

    if ((sizes = model->pointSizes) && (buf = (char *) ckalloc(20 * TCL_DOUBLE_SPACE))) {
	*buf = '\0';
	for (i = 0; i < (const) model->nv - 1; i++, sizes++) {
	    Tcl_PrintDouble(interp, (double) *sizes, size);
	    sprintf(buf + strlen(buf), "%s ", size);
	}
	Tcl_PrintDouble(interp, (double) *sizes, size);
	sprintf(buf + strlen(buf), "%s ", size);
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelTexturesParseProc:
 *
 * process the '-textures' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelTexturesParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *value;
    char *widgRec;
    int offset;
{
    Model *model = (Model *) widgRec;

    /* we simply save a copy of the value string here; actual parsing is done elsewhere */
    if (model->textures) {
	(void) ckfree((void *) model->textures);
    }

    if ((*value)) {
	if ((model->textures = (char *) ckalloc(strlen(value) + 1))) {
	    strcpy(model->textures, value);
	}
    }
    else {
	model->textures = NULL;
    }
    model->updateFlag |= SM_MODEL_UPDATE_TEXTURES;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelTexturesPrintProc:
 *
 * produce a printable string for the '-textures' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelTexturesPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    int i, j, len;
    char *buf;
    char blend[TCL_DOUBLE_SPACE * 3 + 20];
    char r[TCL_DOUBLE_SPACE], g[TCL_DOUBLE_SPACE], b[TCL_DOUBLE_SPACE];
    char x[TCL_DOUBLE_SPACE], y[TCL_DOUBLE_SPACE];
    Model *model = (Model *) widgRec;
    Surface *s;
    TexCoords *texcoords;
    Tcl_Interp *interp = model->cell->interp;

    len = 0;
    for (i = 0, s = model->s; i < (const) model->ns; i++, s++) {
	if (s->texture) {
	    len += (s->vcnt + 3) * (TCL_DOUBLE_SPACE + 10) + 100;
	}
    }

    if ((len > 0) && (buf = (char *) ckalloc(len))) {
	*buf = '\0';
	for (i = 1, s = model->s; i <= (const) model->ns; i++, s++) {
	    if (s->texture) {

		if (s->func == GL_BLEND) {
		    Tcl_PrintDouble(interp, (double) s->texColor[0], r);
		    Tcl_PrintDouble(interp, (double) s->texColor[1], g);
		    Tcl_PrintDouble(interp, (double) s->texColor[2], b);
		    sprintf(blend, "{blend %s %s %s}", r, g, b);
		}

		Tcl_PrintDouble(interp, (double) s->bdColor[0], r);
		Tcl_PrintDouble(interp, (double) s->bdColor[1], g);
		Tcl_PrintDouble(interp, (double) s->bdColor[2], b);

		if (*buf) {
		    strcat(buf, " ");
		}
		sprintf(buf + strlen(buf), "{%d %s %d %s %s %s %s %s {%s %s %s} {",
			i, s->texture->label, s->component,
			(s->func == GL_DECAL) ? "decal" : (s->func == GL_MODULATE) ? "modulate" : blend,
			(s->smode == GL_CLAMP) ? "clamp" : "repeat",
			(s->tmode == GL_CLAMP) ? "clamp" : "repeat",
			(s->magFilter == GL_NEAREST) ? "nearest" : "linear",
			(s->minFilter == GL_NEAREST) ? "nearest" : "linear",
			r, g, b);
		for (j = 0, texcoords = s->texcoords; j < (const) s->vcnt - 1; j++, texcoords++) {
		    Tcl_PrintDouble(interp, (double) texcoords->v[0], x);
		    Tcl_PrintDouble(interp, (double) texcoords->v[1], y);
		    sprintf(buf + strlen(buf), "{%s %s} ", x, y);
		}
		Tcl_PrintDouble(interp, (double) texcoords->v[0], x);
		Tcl_PrintDouble(interp, (double) texcoords->v[1], y);
		sprintf(buf + strlen(buf), "{%s %s}}}", x, y);
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	return buf;
    }
    else {
	*freeProcPtr = NULL;
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelShadingParseProc:
 *
 * process the '-shading' option of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelShadingParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    char c;
    Model *model = (Model *) widgRec;

    c = *value;
    len = strlen(value);
    if ((c == 'f') && (strncmp(value, "flat", len) == 0) && (len >= 2)) {
	model->shading = GL_FLAT;
	return TCL_OK;
    }
    else if ((c == 's') && (strncmp(value, "smooth", len) == 0) && (len >= 2)) {
	model->shading = GL_SMOOTH;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad shading method \"", value, "\": should be either flat or smooth.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ModelShadingPrintProc:
 *
 * produce a printable string for the '-shading' option.
 *
 *--------------------------------------------------------------
 */

static char *
ModelShadingPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    *freeProcPtr = NULL;
    switch (((Model *) widgRec)->shading) {

      case GL_FLAT:
	return "flat";

      case GL_SMOOTH:
	return "smooth";

      default:
	return "unknown";
    }
}

/*
 *--------------------------------------------------------------
 *
 * ModelCoefficientParseProc:
 *
 * process reflection coefficient options of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelCoefficientParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int argc;
    char **argv;
    double r, g, b;
    Model *model = (Model *) widgRec;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc != 3) {
	Tcl_AppendResult(interp, "bad format: must specify three coefficients.", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[0], &r) != TCL_OK ||
	Tcl_GetDouble(interp, argv[1], &g) != TCL_OK ||
	Tcl_GetDouble(interp, argv[2], &b) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if ((r < (double) 0) || (r > (double) 1) ||
	(g < (double) 0) || (g > (double) 1) ||
	(b < (double) 0) || (b > (double) 1)) {
	sprintf(interp->result, "reflection coefficients must be between 0 and 1: \"%f %f %f\"", r, g, b);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    (void) ckfree((void *) argv);
    switch((int) clientData) {

      case SM_MODEL_AMBIENT:
	model->ambient[0] = (DOUBLE) r;
	model->ambient[1] = (DOUBLE) g;
	model->ambient[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_AMBIENT;
	break;

      case SM_MODEL_DIFFUSE:
	model->diffuse[0] = (DOUBLE) r;
	model->diffuse[1] = (DOUBLE) g;
	model->diffuse[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_DIFFUSE;
	break;

      case SM_MODEL_EMISSIVE:
	model->emissive[0] = (DOUBLE) r;
	model->emissive[1] = (DOUBLE) g;
	model->emissive[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_EMISSIVE;
	break;

      case SM_MODEL_BF_AMBIENT:
	model->bfambient[0] = (DOUBLE) r;
	model->bfambient[1] = (DOUBLE) g;
	model->bfambient[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_BF_AMBIENT;
	break;

      case SM_MODEL_BF_DIFFUSE:
	model->bfdiffuse[0] = (DOUBLE) r;
	model->bfdiffuse[1] = (DOUBLE) g;
	model->bfdiffuse[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_BF_DIFFUSE;
	break;

      case SM_MODEL_BF_EMISSIVE:
	model->bfemissive[0] = (DOUBLE) r;
	model->bfemissive[1] = (DOUBLE) g;
	model->bfemissive[2] = (DOUBLE) b;
	model->updateFlag |= SM_MODEL_UPDATE_BF_EMISSIVE;
	break;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ModelCoefficientPrintPRoc:
 *
 * produce a printable string for reflection coefficients.
 *
 *--------------------------------------------------------------
 */

static char *
ModelCoefficientPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    static char buf[256];
    DOUBLE *v = NULL;
    Model *model = (Model *) widgRec;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE];
    Tcl_Interp *interp = model->cell->interp;

    switch((int) clientData) {
      case SM_MODEL_AMBIENT:
	v = model->ambient;
	break;

      case SM_MODEL_DIFFUSE:
	v = model->diffuse;
	break;

      case SM_MODEL_EMISSIVE:
	v = model->emissive;
	break;

      case SM_MODEL_BF_AMBIENT:
	v = model->bfambient;
	break;

      case SM_MODEL_BF_DIFFUSE:
	v = model->bfdiffuse;
	break;

      case SM_MODEL_BF_EMISSIVE:
	v = model->bfemissive;
	break;
    }
    Tcl_PrintDouble(interp, (double) v[0], c0);
    Tcl_PrintDouble(interp, (double) v[1], c1);
    Tcl_PrintDouble(interp, (double) v[2], c2);
    sprintf(buf, "%s %s %s", c0, c1, c2);
    *freeProcPtr = NULL;
    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * ModelSpecularParseProc:
 *
 * process specular reflection options of a model.
 *
 *--------------------------------------------------------------
 */

static int
ModelSpecularParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int argc;
    char **argv;
    double r, g, b, exponent;
    Model *model = (Model *) widgRec;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc != 4) {
	Tcl_AppendResult(interp, "bad format: must specify {r g b exponent}.", (char *) NULL);
	goto err;
    }

    if ((Tcl_GetDouble(interp, argv[0], &r) != TCL_OK) ||
	(Tcl_GetDouble(interp, argv[1], &g) != TCL_OK) ||
	(Tcl_GetDouble(interp, argv[2], &b) != TCL_OK) ||
	(Tcl_GetDouble(interp, argv[3], &exponent) != TCL_OK)) {
	goto err;
    }

    if (r > 1) r = 1;
    if (r < 0) r = 0;
    if (g > 1) g = 1;
    if (g < 0) g = 0;
    if (b > 1) b = 1;
    if (b < 0) b = 0;

    if (exponent < 0) {
	Tcl_AppendResult(interp, "bad value: specular reflection exponent must be non-negative.", (char *) NULL);
	goto err;
    }
    if (clientData == (ClientData) SM_MODEL_SPECULAR) {
	model->specular.RGBA[0] = (float) r;
	model->specular.RGBA[1] = (float) g;
	model->specular.RGBA[2] = (float) b;
	model->specular.RGBA[3] = 1.0;
	model->specular.exponent = (DOUBLE) exponent;
    } else if (clientData == (ClientData) SM_MODEL_BF_SPECULAR) {
	model->bfspecular.RGBA[0] = (float) r;
	model->bfspecular.RGBA[1] = (float) g;
	model->bfspecular.RGBA[2] = (float) b;
	model->bfspecular.RGBA[3] = 1.0;
	model->bfspecular.exponent = (DOUBLE) exponent;
    }
    (void) ckfree((void *) argv);
    return TCL_OK;

  err:
    (void) ckfree((void *) argv);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ModelSpeculartPrintPRoc:
 *
 * produce a printable string for specular reflection parameters.
 *
 *--------------------------------------------------------------
 */

static char *
ModelSpecularPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    char r[TCL_DOUBLE_SPACE];
    char g[TCL_DOUBLE_SPACE];
    char b[TCL_DOUBLE_SPACE];
    char e[TCL_DOUBLE_SPACE];
    char *elements[4];
    Model *model = (Model *) widgRec;
    Tcl_Interp *interp = model->cell->interp;

    *freeProcPtr = (Tcl_FreeProc *) free;

    elements[0] = r;
    elements[1] = g;
    elements[2] = b;
    elements[3] = e;

    if (clientData == (ClientData) SM_MODEL_SPECULAR) {
	Tcl_PrintDouble(interp, (double) model->specular.RGBA[0], r);
	Tcl_PrintDouble(interp, (double) model->specular.RGBA[1], g);
	Tcl_PrintDouble(interp, (double) model->specular.RGBA[2], b);
	Tcl_PrintDouble(interp, (double) model->specular.exponent, e);
    } else {
	Tcl_PrintDouble(interp, (double) model->bfspecular.RGBA[0], r);
	Tcl_PrintDouble(interp, (double) model->bfspecular.RGBA[1], g);
	Tcl_PrintDouble(interp, (double) model->bfspecular.RGBA[2], b);
	Tcl_PrintDouble(interp, (double) model->bfspecular.exponent, e);
    }
    return Tcl_Merge(4, elements);
}

/*
 *--------------------------------------------------------------
 *
 * PolygonModeParseProc:
 *
 * process the polygon drawing mode options (-frontmode, -backmode)
 * of a model.
 *
 *--------------------------------------------------------------
 */

static int
PolygonModeParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    GLenum *mode;
    Model *model = (Model *) widgRec;

    len = strlen(value);
    mode = (((GLenum) clientData) == GL_FRONT) ? &model->frontmode : &model->backmode;
    if ((strncmp("point", value, len) == 0) && (len >= 2)) {
	*mode = GL_POINT;
	return TCL_OK;
    }
    if ((strncmp("line", value, len) == 0) && (len >= 2)) {
	*mode = GL_LINE;
	return TCL_OK;
    }
    if ((strncmp("fill", value, len) == 0) && (len >= 2)) {
	*mode = GL_FILL;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad polygon drawing mode \"", value, "\": should be one of point, line, or fill.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * PolygonModePrintProc:
 *
 * produce a printable string for the polygon drawing mode
 * (-frontmode, -backmode).
 *
 *--------------------------------------------------------------
 */

static char *
PolygonModePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *model = (Model *) widgRec;
    GLenum *mode;

    *freeProcPtr = NULL;
    mode = (((GLenum) clientData) == GL_FRONT) ? &model->frontmode : &model->backmode;
    if (*mode == GL_POINT) {
	return "point";
    }
    if (*mode == GL_LINE) {
	return "line";
    }
    if (*mode == GL_FILL) {
	return "fill";
    }
    return "unknown polygon drawing mode";
}

/*
 *--------------------------------------------------------------
 *
 * PolygonCullFaceParseProc:
 *
 * process the polygon face culling option -cull.
 *
 *--------------------------------------------------------------
 */

static int
PolygonCullFaceParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    Model *model = (Model *) widgRec;

    len = strlen(value);

    if ((strncmp("none", value, len) == 0) && (len >= 2)) {
	model->cull = 0;
	return TCL_OK;
    }
    if ((strncmp("front", value, len) == 0) && (len >= 2)) {
	model->cull = GL_FRONT;
	return TCL_OK;
    }
    if ((strncmp("back", value, len) == 0) && (len >= 2)) {
	model->cull = GL_BACK;
	return TCL_OK;
    }
    if ((strncmp("both", value, len) == 0) && (len >= 2)) {
	model->cull = GL_FRONT_AND_BACK;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad polygon face culling mode \"", value, "\": should be one of none, front, back, or both.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * PolygonCullFacePrintProc:
 *
 * produce a printable string for the polygon culling mode
 * (-cull).
 *
 *--------------------------------------------------------------
 */

static char *
PolygonCullFacePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *model = (Model *) widgRec;

    *freeProcPtr = NULL;
    if (model->cull == 0) {
	return "none";
    }
    if (model->cull == GL_FRONT) {
	return "front";
    }
    if (model->cull == GL_BACK) {
	return "back";
    }
    if (model->cull == GL_FRONT_AND_BACK) {
	return "both";
    }
    return "unknown polygon culling mode";
}

/*
 *--------------------------------------------------------------
 *
 * QuadricDrawStyleParseProc:
 *
 * process the drawing style options (-drawstyle)
 * of a quadric model.
 *
 *--------------------------------------------------------------
 */

static int
QuadricDrawStyleParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    Model *model = (Model *) widgRec;

    len = strlen(value);
    if ((strncmp("point", value, len) == 0) && (len >= 2)) {
	model->drawStyle = (GLenum) GLU_POINT;
	return TCL_OK;
    }
    if ((strncmp("line", value, len) == 0) && (len >= 2)) {
	model->drawStyle = (GLenum) GLU_LINE;
	return TCL_OK;
    }
    if ((strncmp("fill", value, len) == 0) && (len >= 2)) {
	model->drawStyle = (GLenum) GLU_FILL;
	return TCL_OK;
    }
    if ((strncmp("silhouette", value, len) == 0) && (len >= 2)) {
	model->drawStyle = (GLenum) GLU_SILHOUETTE;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad quadric drawing style \"", value,
		     "\": should be one of point, line, silhouette, or fill.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * QuadricDrawStylePrintProc:
 *
 * produce a printable string for the quadric drawing style
 * (-drawstyle).
 *
 *--------------------------------------------------------------
 */

static char *
QuadricDrawStylePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *model = (Model *) widgRec;

    *freeProcPtr = NULL;
    if (model->drawStyle == GLU_POINT) {
	return "point";
    }
    if (model->drawStyle == GLU_LINE) {
	return "line";
    }
    if (model->drawStyle == GLU_FILL) {
	return "fill";
    }
    if (model->drawStyle == GLU_SILHOUETTE) {
	return "silhouette";
    }
    return "unknown quadric drawing style";
}

/*
 *--------------------------------------------------------------
 *
 * QuadricNormalsParseProc:
 *
 * process the normals options (-normals)
 * of a quadric model.
 *
 *--------------------------------------------------------------
 */

static int
QuadricNormalsParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    Model *model = (Model *) widgRec;

    len = strlen(value);
    if ((strncmp("none", value, len) == 0) && (len >= 2)) {
	model->quadNormals = GLU_NONE;
	return TCL_OK;
    }
    if ((strncmp("flat", value, len) == 0) && (len >= 2)) {
	model->quadNormals = GLU_FLAT;
	return TCL_OK;
    }
    if ((strncmp("smooth", value, len) == 0) && (len >= 2)) {
	model->quadNormals = GLU_SMOOTH;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad quadric normals \"", value,
		     "\": should be one of none, flat, or smooth.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * QuadricNormalsPrintProc:
 *
 * produce a printable string for the quadric drawing style
 * (-normals).
 *
 *--------------------------------------------------------------
 */

static char *
QuadricNormalsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *model = (Model *) widgRec;

    *freeProcPtr = NULL;
    if (model->quadNormals == GLU_NONE) {
	return "none";
    }
    if (model->quadNormals == GLU_FLAT) {
	return "flat";
    }
    if (model->quadNormals == GLU_SMOOTH) {
	return "smooth";
    }
    return "unknown quadric normals";
}

/*
 *--------------------------------------------------------------
 *
 * QuadricOrientationParseProc:
 *
 * process the orientation options (-orientation)
 * of a quadric model.
 *
 *--------------------------------------------------------------
 */

static int
QuadricOrientationParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int len;
    Model *model = (Model *) widgRec;

    len = strlen(value);
    if ((strncmp("outside", value, len) == 0) && (len >= 2)) {
	model->orientation = GLU_OUTSIDE;
	return TCL_OK;
    }
    if ((strncmp("inside", value, len) == 0) && (len >= 2)) {
	model->orientation = GLU_INSIDE;
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad quadric orientation \"", value,
		     "\": should be either inside or outside.", (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * QuadricOrientationPrintProc:
 *
 * produce a printable string for the quadric drawing style
 * (-orientation).
 *
 *--------------------------------------------------------------
 */

static char *
QuadricOrientationPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    Model *model = (Model *) widgRec;

    *freeProcPtr = NULL;
    if (model->orientation == GLU_OUTSIDE) {
	return "outside";
    }
    if (model->orientation == GLU_INSIDE) {
	return "inside";
    }
    return "unknown quadric orientation";
}
