/*
 * smtexture.c --
 *
 *	This file implements (single level) texture maps.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <tk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "smInt.h"


/********************************************************************************
 *                                                                              *
 * The following data structures are lifted from tkImgPhoto.c, and are used by  *
 * photo images internally. They are needed here so that we can go from an      *
 * Tk_Image token directly to the pixel data.                                   *
 *                                                                              *
 ********************************************************************************/

typedef struct Image {
    Tk_Window tkwin;            /* Window passed to Tk_GetImage (needed to
                                 * "re-get" the image later if the manager
                                 * changes). */
    Display *display;           /* Display for tkwin.  Needed because when
                                 * the image is eventually freed tkwin may
                                 * not exist anymore. */
    struct ImageMaster *masterPtr;
                                /* Master for this image (identifiers image
                                 * manager, for example). */
    ClientData instanceData;
                                /* One word argument to pass to image manager
                                 * when dealing with this image instance. */
    Tk_ImageChangedProc *changeProc;
                                /* Code in widget to call when image changes
                                 * in a way that affects redisplay. */
    ClientData widgetClientData;
                                /* Argument to pass to changeProc. */
    struct Image *nextPtr;      /* Next in list of all image instances
                                 * associated with the same name. */

} Image;

typedef struct ImageMaster {
    Tk_ImageType *typePtr;      /* Information about image type.  NULL means
                                 * that no image manager owns this image:  the
                                 * image was deleted. */
    ClientData masterData;      /* One-word argument to pass to image mgr
                                 * when dealing with the master, as opposed
                                 * to instances. */
    int width, height;          /* Last known dimensions for image. */
    Tcl_HashTable *tablePtr;    /* Pointer to hash table containing image
                                 * (the imageTable field in some TkMainInfo
                                 * structure). */
    Tcl_HashEntry *hPtr;        /* Hash entry in mainPtr->imageTable for
                                 * this structure (used to delete the hash
                                 * entry). */
    Image *instancePtr;         /* Pointer to first in list of instances
                                 * derived from this name. */
} ImageMaster;


typedef struct PhotoMaster {
    Tk_ImageMaster tkMaster;    /* Tk's token for image master.  NULL means
                                 * the image is being deleted. */
    Tcl_Interp *interp;         /* Interpreter associated with the
                                 * application using this image. */
    Tcl_Command imageCmd;       /* Token for image command (used to delete
                                 * it when the image goes away).  NULL means
                                 * the image command has already been
                                 * deleted. */
    int flags;                  /* Sundry flags, defined below. */
    int width, height;          /* Dimensions of image. */
    int userWidth, userHeight;  /* User-declared image dimensions. */
    Tk_Uid palette;             /* User-specified default palette for
                                 * instances of this image. */
    double gamma;               /* Display gamma value to correct for. */
    char *fileString;           /* Name of file to read into image. */
    char *dataString;           /* String value to use as contents of image. */
    char *format;               /* User-specified format of data in image
                                 * file or string value. */
    unsigned char *pix24;       /* Local storage for 24-bit image. */
    int ditherX, ditherY;       /* Location of first incorrectly
                                 * dithered pixel in image. */
    Region validRegion;         /* X region indicating which parts of
                                 * the image have valid image data. */
    void *instancePtr;
                                /* First in the list of instances
                                 * associated with this master. */
} PhotoMaster;


/*
 * Prototypes for procedures defined externally:
 */

CellHeader	*SmGetCellHeader _ANSI_ARGS_ ((Tcl_Interp *interp));


/*
 * Prototypes for procedures defined in this file that are exported:
 */

int		SmTextureCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp, int argc, char **argv));

Texture *	SmGetTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, char *label));

void		SmFreeTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, Texture *texture));

Texture *	SmNewTextureMapFromSFImage _ANSI_ARGS_((Tcl_Interp *interp, long width, long height, long comp, unsigned long *data));


/*
 * Prototypes for procedures defined in this file that are static:
 */

static int	NewTextureMapFromData _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, int argc, char **argv));

static int	NewTextureMapFromImage _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, int argc, char **argv));

static int	PowerOfTwo _ANSI_ARGS_ ((GLsizei num));

static long	RoundToPowerOfTwo _ANSI_ARGS_ ((long num));



/*
 *--------------------------------------------------------------
 *
 * SmTextureCmd --
 *
 * process texture commands.
 *
 * valid commands are:
 *
 *		create data <width> <height> <border> <data>
 *		create image <src_x> <src_y> <width> <height> <border> <image>
 *		list
 *		destry <texturemap>
 *
 *--------------------------------------------------------------
 */

int
SmTextureCmd(clientData, interp, argc, argv)
     ClientData clientData;	/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;	/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
    int row, col, length;
    char c, buf[128];
    unsigned char *p;
    Texture *texture, *last;
    CellHeader *header = (CellHeader *) clientData;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " cmd ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "create", length) == 0) && (length >= 2)) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " create type parameters\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (strcmp(argv[2], "data") == 0) {
	    if (argc != 7) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
				 " create data <width> <height> <border> <data>\"",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    return NewTextureMapFromData(interp, header, argc-3, argv+3);
	} else if (strcmp(argv[2], "image") == 0) {
	    if (argc != 9) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
				 " create image <src_x> <src_y> <width> <height> <border> <image>\"",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    return NewTextureMapFromImage(interp, header, argc-3, argv+3);
	} else {
	    Tcl_AppendResult(interp, "unknown type \"", argv[2],
			     "\": must be data or image", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0) && (length >= 2)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " list\"", (char *) NULL);
	    return TCL_ERROR;
	}

	for (texture = header->textures; texture; texture = texture->next) {
	    Tcl_AppendElement(interp, texture->label);
	}
	return TCL_OK;
    }
    else if ((c == 'd') && (strncmp(argv[1], "destroy", length) == 0) && (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " destroy texturemap\"", (char *) NULL);
	    return TCL_ERROR;
	}

	last = NULL;
	for (texture = header->textures; texture; last = texture, texture = texture->next) {
	    if (strcmp(texture->label, argv[2]) == 0) {
		if (last == NULL) {
		    header->textures = texture->next;
		}
		else {
		    last->next = texture->next;
		}
		if (texture->refCount == 0) {
		    (void) ckfree((void *) texture->data);
		    (void) ckfree((void *) texture->label);
		    (void) ckfree((void *) texture);
		} else {
		    texture->refCount = -texture->refCount;
		}
		return TCL_OK;
	    }
	}
	Tcl_AppendResult(interp, "texture map not found: \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0) && (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " get texturemap\"", (char *) NULL);
	    return TCL_ERROR;
	}

	last = NULL;
	for (texture = header->textures; texture; last = texture, texture = texture->next) {
	    if (strcmp(texture->label, argv[2]) == 0) {
		p = (unsigned char *) texture->data;
		for (row = 0; row < (const) texture->height; row++) {
		    for (col = 0; col < (const) texture->width; col++) {
			sprintf(buf, "%d %d %d", *p++, *p++, *p++);
			Tcl_AppendElement(interp, buf);
		    }
		}
		return TCL_OK;
	    }
	}
	Tcl_AppendResult(interp, "texture map not found: \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    else {
	Tcl_AppendResult(interp, "bad command \"", argv[1],
			 "\": must be create, destroy, or list", (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmGetTexture --
 *
 * return pointer to texture record.
 *
 *--------------------------------------------------------------
 */

Texture *
SmGetTexture(interp, header, label)
     Tcl_Interp *interp;
     CellHeader *header;
     char *label;
{
    Texture *texture;

    for (texture = header->textures; texture; texture = texture->next) {
	if (strcmp(texture->label, label) == 0) {
	    texture->refCount++;
	    return texture;
	}
    }
    return (Texture *) NULL;
}

/*
 *--------------------------------------------------------------
 *
 * SmFreeTexture --
 *
 * release a texture record.
 *
 *--------------------------------------------------------------
 */

void
SmFreeTexture(interp, header, texture)
     Tcl_Interp *interp;
     CellHeader *header;
     Texture *texture;
{
    if (texture->refCount < 0) {
	texture->refCount++;
	if (texture->refCount == 0) {
	    (void) ckfree((void *) texture->data);
	    (void) ckfree((void *) texture->label);
	    (void) ckfree((void *) texture);
	}
    }
    else {
	texture->refCount--;
    }
}

/*
 *--------------------------------------------------------------
 *
 * NewTextureMapFromData --
 *
 * create a new texture map from a list of RGB entries.
 *
 * input: <width> <height> <border> <data>
 *
 *--------------------------------------------------------------
 */

static int
NewTextureMapFromData(interp, header, argc, argv)
     Tcl_Interp *interp;
     CellHeader *header;
     int argc;
     char **argv;
{
    int i, len, num;
    int r, g, b;
    int width, height, border;
    char **items, **colorvalues, buf[128];
    Texture *new;
    unsigned char *bytes, *p;

    /* parse width and height: must be non-negative */
    if ((Tcl_GetInt(interp, argv[0], &width) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[1], &height) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[2], &border) != TCL_OK)) {
	return TCL_ERROR;
    }

    if (width < 0) {
	Tcl_AppendResult(interp, "width out of range \"", argv[0],
			 "\":  must be non-negative", (char *) NULL);
	return TCL_ERROR;
    }

    if (height < 0) {
	Tcl_AppendResult(interp, "height out of range \"", argv[1],
			 "\":  must be non-negative", (char *) NULL);
	return TCL_ERROR;
    }

    if ((border < 0) || (border > 1)) {
	Tcl_AppendResult(interp, "illegal border value \"", argv[2],
			 "\":  must be 0 or 1", (char *) NULL);
	return TCL_ERROR;
    }

    if (!PowerOfTwo((GLsizei) (width - 2 * border))) {
	sprintf(interp->result, "width (%d) minus 2 * border (%d) must be an integral power of two",
		width, border);
	return TCL_ERROR;
    }

    if (!PowerOfTwo((GLsizei) (height - 2 * border))) {
	sprintf(interp->result, "height (%d) minus 2 * border (%d) must be an integral power of two",
		height, border);
	return TCL_ERROR;
    }

    if (Tcl_SplitList(interp, argv[3], &len, &items) != TCL_OK) {
	return TCL_ERROR;
    }

    if (len != width * height) {
	Tcl_AppendResult(interp, "bad data length: data length must equal width x height",
			 (char *) NULL);
	(void) ckfree((void *) items);
	return TCL_ERROR;
    }

    if ((new = (Texture *) ckalloc(sizeof(Texture))) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture",
			 (char *) NULL);
	(void) ckfree((void *) items);
	return TCL_ERROR;
    }

    sprintf(buf, "texture%ld", header->id);
    if ((new->label = (char *) ckalloc(strlen(buf) + 1)) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture label",
			 (char *) NULL);
	(void) ckfree((void *) items);
	(void) ckfree((void *) new);
	return TCL_ERROR;
    }
    if ((bytes = (unsigned char *) ckalloc(3 * len)) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture data",
			 (char *) NULL);
	(void) ckfree((void *) items);
	(void) ckfree((void *) new);
	(void) ckfree((void *) new->label);
	return TCL_ERROR;
    }

    for (p = bytes, i = 0; i < (const) len; i++) {

	/* each item consists of RGB values */
	if (Tcl_SplitList(interp, items[i], &num, &colorvalues) != TCL_OK) {
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) new);
	    (void) ckfree((void *) bytes);
	    (void) ckfree((void *) new->label);
	    return TCL_ERROR;
	}

	/* make sure there are exactly three color components */
	if (num != 3) {
	    Tcl_AppendResult(interp, "wrong # args \"", items[i],
			     "\": expected a list of 3 elements", (char *) NULL);
	    (void) ckfree((void *) colorvalues);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) new);
	    (void) ckfree((void *) bytes);
	    (void) ckfree((void *) new->label);
	    return TCL_ERROR;
	}

	if ((Tcl_GetInt(interp, colorvalues[0], &r) != TCL_OK) ||
	    (Tcl_GetInt(interp, colorvalues[1], &g) != TCL_OK) ||
	    (Tcl_GetInt(interp, colorvalues[2], &b) != TCL_OK)) {
	    (void) ckfree((void *) colorvalues);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) new);
	    (void) ckfree((void *) bytes);
	    (void) ckfree((void *) new->label);
	    return TCL_ERROR;
	}

	if ((r < 0) || (r > 255) ||
	    (g < 0) || (g > 255) ||
	    (b < 0) || (b > 255)) {
	    Tcl_AppendResult(interp, "RGB component(s) out of range \"", items[i],
			     "\": must be between 0 and 255", (char *) NULL);
	    (void) ckfree((void *) colorvalues);
	    (void) ckfree((void *) items);
	    (void) ckfree((void *) new);
	    (void) ckfree((void *) bytes);
	    (void) ckfree((void *) new->label);
	    return TCL_ERROR;
	}
	(void) ckfree((void *) colorvalues);
	*p++ = (unsigned char) (r & 0x00ff);
	*p++ = (unsigned char) (g & 0x00ff);
	*p++ = (unsigned char) (b & 0x00ff);
    }
    (void) ckfree((void *) items);

    new->width = width;
    new->height = height;
    new->border = border;
    new->data = (void *) bytes;
    new->refCount = 0;
    new->next = header->textures;
    header->textures = new;
    header->id++;
    strcpy(new->label, buf);
    strcpy(interp->result, buf);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * NewTextureMapFromImage --
 *
 * create a new texture map from a Photo image.
 *
 * input: <src_x> <src_y> <width> <height> <border> <image>
 *
 *--------------------------------------------------------------
 */

static int
NewTextureMapFromImage(interp, header, argc, argv)
     Tcl_Interp *interp;
     CellHeader *header;
     int argc;
     char **argv;
{
    int x, y, offset;
    int src_x, src_y, width, height, border;
    char *p, *bytes, buf[128];
    Texture *new;

    Image *imagePtr;
    ImageMaster *masterPtr;
    PhotoMaster *photoMaster;
    unsigned char *pixelPtr, *pixelBasePtr;

    /* parse src_x, src_y, width and height */
    if ((Tcl_GetInt(interp, argv[0], &src_x) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[1], &src_y) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[2], &width) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[3], &height) != TCL_OK) ||
	(Tcl_GetInt(interp, argv[4], &border) != TCL_OK) ||
	((imagePtr = (Image *) Tk_GetImage(interp, Tk_MainWindow(interp), argv[5], NULL, NULL)) == (Image *) NULL)) {
	return TCL_ERROR;
    }
    masterPtr = imagePtr->masterPtr;
    Tk_FreeImage((Tk_Image) imagePtr);

    if (strcmp(masterPtr->typePtr->name, "photo") != 0) {
	Tcl_AppendResult(interp, "wrong image type: \"", masterPtr->typePtr->name, "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if ((width < 1) || (height < 1)) {
	Tcl_AppendResult(interp, "image region is empty", (char *) NULL);
	return TCL_ERROR;
    }

    if ((border < 0) || (border > 1)) {
	Tcl_AppendResult(interp, "illegal border value \"", argv[2],
			 "\":  must be 0 or 1", (char *) NULL);
	return TCL_ERROR;
    }

    if ((src_x < 0) || (src_x + width > masterPtr->width) ||
	(src_y < 0) || (src_y + height > masterPtr->height)) {
	Tcl_AppendResult(interp, "image region out of bounds", (char *) NULL);
	return TCL_ERROR;
    }

    if (!PowerOfTwo((GLsizei) (width - 2 * border))) {
	sprintf(interp->result, "width (%d) minus 2 * border (%d) must be an integral power of two",
		width, border);
	return TCL_ERROR;
    }

    if (!PowerOfTwo((GLsizei) (height - 2 * border))) {
	sprintf(interp->result, "height (%d) minus 2 * border (%d) must be an integral power of two",
		height, border);
	return TCL_ERROR;
    }

    if ((new = (Texture *) ckalloc(sizeof(Texture))) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture",
			 (char *) NULL);
	return TCL_ERROR;
    }

    sprintf(buf, "texture%ld", header->id);
    if ((new->label = (char *) ckalloc(strlen(buf) + 1)) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture label",
			 (char *) NULL);
	(void) ckfree((void *) new);
	return TCL_ERROR;
    }
    if ((bytes = (char *) ckalloc(3 * width * height)) == NULL) {
	Tcl_AppendResult(interp, "memory error: failed to allocate memory for texture data",
			 (char *) NULL);
	(void) ckfree((void *) new);
	(void) ckfree((void *) new->label);
	return TCL_ERROR;
    }

    p = bytes;
    photoMaster = (PhotoMaster *) masterPtr->masterData;
    offset = photoMaster->width * 3;
    pixelBasePtr = photoMaster->pix24 + src_y * offset;
    for (y = src_y; y < (const) src_y + height; y++) {
	pixelPtr = pixelBasePtr + src_x * 3;
	for (x = src_x; x < (const) src_x + width; x++) {
	    *p++ = *pixelPtr++;
	    *p++ = *pixelPtr++;
	    *p++ = *pixelPtr++;
	}
	pixelBasePtr += offset;
    }
    new->width = width;
    new->height = height;
    new->border = border;
    new->data = (void *) bytes;
    new->refCount = 0;
    new->next = header->textures;
    header->textures = new;
    header->id++;
    strcpy(new->label, buf);
    strcpy(interp->result, buf);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SmNewTextureMapFromSFImage --
 *
 * create a new texture map from an SFImage
 *
 *--------------------------------------------------------------
 */

Texture *
SmNewTextureMapFromSFImage(interp, width, height, comp, data)
     Tcl_Interp *interp;
     long width;
     long height;
     long comp;
     unsigned long *data;
{
    int x, y;
    char *p, *bytes, buf[128];
    long w, h;
    unsigned long *d;
    Texture *new;
    CellHeader *header = SmGetCellHeader(interp);

    w = RoundToPowerOfTwo(width);
    h = RoundToPowerOfTwo(height);

    if ((w < 1) || (h < 1)) {
	return NULL;
    }

    if ((new = (Texture *) ckalloc(sizeof(Texture))) == NULL) {
	return NULL;
    }

    sprintf(buf, "texture%ld", header->id);
    if ((new->label = (char *) ckalloc(strlen(buf) + 1)) == NULL) {
	(void) ckfree((void *) new);
	return NULL;
    }
    if ((bytes = (char *) ckalloc(3 * w * h)) == NULL) {
	(void) ckfree((void *) new);
	(void) ckfree((void *) new->label);
	return NULL;
    }

    p = bytes;
    for (y = 0; y < (const) h; y++) {
	d = data + width * y;
	for (x = 0; x < (const) w; x++, d++) {
	    switch(comp) {

	      case 1:
		*p++ = (char) (*d & 0x000000ff);
		*p++ = *p;
		*p++ = *p;
		break;

	      case 2:
		*p++ = (char) ((*d & 0x0000ff00) >> 8);
		*p++ = *p;
		*p++ = *p;
		break;

	      case 3:
		*p++ = (char) ((*d & 0x00ff0000) >> 16);
		*p++ = (char) ((*d & 0x0000ff00) >> 8);
		*p++ = (char) (*d & 0x000000ff);
		break;

	      case 4:
		*p++ = (char) ((*d & 0xff000000) >> 24);
		*p++ = (char) ((*d & 0x00ff0000) >> 16);
		*p++ = (char) ((*d & 0x0000ff00) >> 8);
		break;
	    }
	}
    }
    new->width = w;
    new->height = h;
    new->border = 0;
    new->data = (void *) bytes;
    new->refCount = 0;
    new->next = header->textures;
    header->textures = new;
    header->id++;
    strcpy(new->label, buf);
    return new;
}

/*
 *--------------------------------------------------------------
 *
 * PowerOfTwo:
 *
 * return 1 if the integer is an even power of two.
 *
 *--------------------------------------------------------------
 */

static int
PowerOfTwo(num)
     GLsizei num;
{
    int i, on;
    GLsizei n;

    for (i = 0, on = 0, n = num; i < (const) (sizeof(GLsizei) * 8); i++, n >>= 1) {
	if (n & (GLsizei) 0x0001) {
	    if (on) {
		return 0;
	    }
	    else {
		on = 1;
	    }
	}
    }
    return 1;
}

/*
 *--------------------------------------------------------------
 *
 * RoundToPowerOfTwo:
 *
 * round integer down to nearest power of two.
 *
 *--------------------------------------------------------------
 */

static long
RoundToPowerOfTwo(num)
     long num;
{
    int i;
    GLsizei n;

    for (i = 0, n = num; n; i++, n >>= 1);
    if (i > 0) {
	return (1 << (i - 1));
    }
    else {
	return 0;
    }
}
