/* 
 * smview.c --
 *
 *	This file implements canvas viewport widgets.
 *
 * Portion of this code based on tkRectOval.c
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "copyright.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <tk.h>
#include <sys/types.h>
#include <X11/Xutil.h>
#include <tkCanvas.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "smInt.h"


#define	WHICH_SIDE(n, x0, y0, z0, x, y, z)	(n[0] * (x - x0) + n[1] * (y - y0) + n[2] * (z - z0))


/*
 * Prototypes for procedures defined externally
 */

void		SmRemovePortFromCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

int		SmViewportChanged _ANSI_ARGS_ ((PortItem *port));

void		SmFreeTexture _ANSI_ARGS_ ((Tcl_Interp *interp, CellHeader *header, Texture *texture));

CellHeader	*SmGetCellHeader _ANSI_ARGS_ ((Tcl_Interp *interp));

int		SmGetBoundingBox _ANSI_ARGS_ ((Model *model, DOUBLE *x0, DOUBLE *y0, DOUBLE *z0,
					       DOUBLE *x1, DOUBLE *y1, DOUBLE *z1, int recursive));


/*
 * Prototypes for procedures defined in this file that are exported:
 */

int		SmNormalizeVector3D _ANSI_ARGS_ ((DOUBLE vin[], DOUBLE vout[], double epsilon));

void		SmCrossVector3D _ANSI_ARGS_ ((DOUBLE a[], DOUBLE b[], DOUBLE c[]));

void		SmDisplayViewport _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, Display *dpy, Drawable dst,
						int x, int y, int width, int height));

void		SmDisplayModels _ANSI_ARGS_ ((PortItem *port, int pick));

void		SmSelectViewport _ANSI_ARGS_ ((PortItem *port));

XVisualInfo	*SmGetVisualInfo _ANSI_ARGS_ ((Display *dpy, Tk_Window tkwin, int doulebuffer));

int		SmConfigureViewport _ANSI_ARGS_ ((Tcl_Interp *interp,
						  Tk_Canvas canvas, Tk_Item *itemPtr, int argc, char **argv, int flags));

int		SmComputeViewTransform _ANSI_ARGS_ ((PortItem *port));

void		SmCreateDrawable _ANSI_ARGS_ ((PortItem *port));


/*
 * Prototypes for procedures defined in this file that are static:
 */

static int	CreateViewport _ANSI_ARGS_ ((Tcl_Interp *interp,
					     Tk_Canvas canvas, struct Tk_Item *itemPtr, int argc, char **argv));

static int	PortCoords _ANSI_ARGS_ ((Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr,
					 int argc, char **argv));

static void	DeleteViewport _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, Display *dpy));

static void	ComputeViewportBbox _ANSI_ARGS_ ((Tk_Canvas canvas, PortItem *port));

static void	SetupLightSources _ANSI_ARGS_ ((PortItem *port));

static double	PortToPoint _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, double *pointPtr));

static int	PortToArea _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, double *areaPtr));

static void	ScaleViewport _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, double originX, double originY,
					    double scaleX, double scaleY));

static void	TranslateViewport _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, double deltaX, double deltaY));

static int	Equal _ANSI_ARGS_ ((DOUBLE a, DOUBLE b, double epsilon));

static int	EqualVector3D _ANSI_ARGS_ ((DOUBLE a[], DOUBLE b[], double epsilon));

static int	DrawModel _ANSI_ARGS_ ((PortItem *port, Model *model, GLuint name));


/*
 * Data structure declarations
 */

extern Cell *cell;
extern Tk_ConfigSpec canvasConfigSpecs[];
extern Tk_ConfigSpec widgetConfigSpecs[];


/*
 * The structures below defines the viewport item type
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkViewportType = {
    "viewport",				/* name */
    sizeof(PortItem),			/* itemSize */
    CreateViewport,			/* createProc */
    canvasConfigSpecs,			/* configSpecs */
    SmConfigureViewport,		/* configureProc */
    PortCoords,				/* coordProc */
    DeleteViewport,			/* deleteProc */
    SmDisplayViewport,			/* displayProc */
    0,					/* alwaysRedraw */
    PortToPoint,			/* pointProc */
    PortToArea,				/* areaProc */
    (Tk_ItemPostscriptProc *) NULL,	/* postscriptProc */
    ScaleViewport,			/* scaleProc */
    TranslateViewport,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};



/*
 *--------------------------------------------------------------
 *
 * CreateViewport:
 *
 * create a new viewport item in a canvas.
 *
 *--------------------------------------------------------------
 */

static int
CreateViewport(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    int argc;
    char **argv;
{
    int i;
    XGCValues gcValues;
    unsigned long mask;
    PortItem *port = (PortItem *) itemPtr;

    if (argc < 4) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), "\" create ",
		itemPtr->typePtr->name, " x1 y1 x2 y2 ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    port->redraw = 1;
    port->dpy = Tk_Display(Tk_CanvasTkwin(canvas));
    port->canvas = canvas;
    port->tkwin = NULL;
    port->widgetCmd = NULL;
    mask = GCFunction | GCPlaneMask;
    gcValues.function = GXcopy;
    gcValues.plane_mask = AllPlanes;
    port->initialized = 0;
    port->visinfo = NULL;
    port->pm = None;
    port->pmGC = Tk_GetGC(Tk_CanvasTkwin(canvas), mask, &gcValues);
    port->glxpm = None;
    port->glxpmGC = None;
    port->outlineColor = NULL;
    port->fillColor = NULL;
    port->outlineGC = None;
    port->interp = interp;
    port->invalid = 0;
    port->cell = NULL;
    port->next = NULL;
    port->pick[0] = 0;
    port->ambientLight = NULL;
    port->ambient[0] = port->ambient[1] = port->ambient[2] = 0;
    port->ambient[3] = 1.0;
    port->depthcue.on = 0;
    port->depthcue.RGBA[3] = 1.0;
    port->hsr = 1;
    port->doublebuffer = 0;
    port->texture = 1;
    port->picking = 0;
    port->epsilon = 1e-10;
    port->srcx = port->srcy = 0;
    port->srcw = port->srch = 0;
    port->invertedVisibleMask = 0;
    port->visibleMaskStr = NULL;

    for (i = 0; i < 8; i++) {
	port->lights[i].on = 0;
    }

    Tcl_InitHashTable(&port->visibleHash, TCL_STRING_KEYS);

    /*
     * Process the arguments to fill in the item record.
     */

    if ((Tk_CanvasGetCoord(interp, canvas, argv[0], &port->bbox[0]) != TCL_OK)
	|| (Tk_CanvasGetCoord(interp, canvas, argv[1], &port->bbox[1]) != TCL_OK)
	|| (Tk_CanvasGetCoord(interp, canvas, argv[2], &port->bbox[2]) != TCL_OK)
	|| (Tk_CanvasGetCoord(interp, canvas, argv[3], &port->bbox[3]) != TCL_OK)) {
	DeleteViewport(canvas, itemPtr, port->dpy);
	return TCL_ERROR;
    }
    port->w = ceil(port->bbox[2] - port->bbox[0]);
    port->h = ceil(port->bbox[3] - port->bbox[1]);

    if (SmConfigureViewport(interp, canvas, itemPtr, argc-4, argv+4, 0) != TCL_OK) {
	DeleteViewport(canvas, itemPtr, port->dpy);
	return TCL_ERROR;
    }

    /* get desired visual & context */
    if (!(port->visinfo = SmGetVisualInfo(port->dpy, Tk_CanvasTkwin(canvas), 0))) {
	Tcl_AppendResult(interp, "parent visual does not meet required specifications", (char *) NULL);
	DeleteViewport(canvas, itemPtr, port->dpy);
	return TCL_ERROR;
    }

    /* create drawing buffer */
    SmCreateDrawable(port);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortCoords:
 *
 * process the "coords" widget command on a viewport.
 *
 *--------------------------------------------------------------
 */

static int
PortCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    int argc;
    char **argv;
{
    PortItem *port = (PortItem *) itemPtr;
    double x1, y1, x2, y2;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    if (argc == 0) {
	Tcl_PrintDouble(interp, (double) port->bbox[0], c0);
	Tcl_PrintDouble(interp, (double) port->bbox[1], c1);
	Tcl_PrintDouble(interp, (double) port->bbox[2], c2);
	Tcl_PrintDouble(interp, (double) port->bbox[3], c3);
	Tcl_AppendResult(interp, c0, " ", c1, " ", c2, " ", c3,	(char *) NULL);
    } else if (argc == 4) {
	if ((Tk_CanvasGetCoord(interp, canvas, argv[0], &x1) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, canvas, argv[1], &y1) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, canvas, argv[2], &x2) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, canvas, argv[3], &y2) != TCL_OK)) {
	    return TCL_ERROR;
	}

	/* make sure viewport isn't too big */

	port->w = x2 - x1;
	port->h = y2 - y1;
	port->bbox[0] = x1;
	port->bbox[1] = y1;
	port->bbox[2] = x2;
	port->bbox[3] = y2;
	ComputeViewportBbox(canvas, port);

	SmCreateDrawable(port);
	port->invalid = SmComputeViewTransform(port);
	port->redraw = 1;
    } else {
	sprintf(interp->result,	"wrong # coordinates:  expected 0 or 4, got %d", argc);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SmConfigureViewport:
 *
 * configure a viewport item or a viewport widget.
 *
 *--------------------------------------------------------------
 */

int
SmConfigureViewport(interp, canvas, itemPtr, argc, argv, flags)
     Tcl_Interp *interp;
     Tk_Canvas canvas;
     Tk_Item *itemPtr;
     int argc;
     char **argv;
     int flags;
{
    GC newGC;
    unsigned long mask;
    XGCValues gcValues;
    Tk_Window tkwin;
    Tk_ConfigSpec *specs;
    PortItem *port = (PortItem *) itemPtr;
    XColor *ambient = port->ambientLight;

    if (port->canvas) {
	tkwin = Tk_CanvasTkwin(port->canvas);
	specs= canvasConfigSpecs;
    } else {
	tkwin = (Tk_Window) canvas;
	specs = widgetConfigSpecs;
    }

    port->picking = 0;
    if (Tk_ConfigureWidget(interp, tkwin, specs, argc, argv, (char *) port, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * if the only option modified is -pick, don't redraw the port
     */
    if (port->picking) {
	port->picking = 0;
	if (argc == 2) {
	    return TCL_OK;
	}
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (port->width < 1) {
	port->width = 1;
    }
    if (port->outlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = port->outlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	gcValues.line_width = port->width;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (port->outlineGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), port->outlineGC);
    }
    port->outlineGC = newGC;
    port->redraw = 1;

    if (port->canvas) {
	ComputeViewportBbox(canvas, port);
    }

    if (port->ambientLight != ambient) {
	port->ambient[0] = (DOUBLE) port->ambientLight->red / (DOUBLE) 65535.0;
	port->ambient[1] = (DOUBLE) port->ambientLight->green / (DOUBLE) 65535.0;
	port->ambient[2] = (DOUBLE) port->ambientLight->blue / (DOUBLE) 65535.0;
    }
    if ((!port->initialized) || SmViewportChanged(port)) {
	if ((port->invalid = SmComputeViewTransform(port)))
	  return TCL_ERROR;
	port->initialized = 1;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteViewport:
 *
 * clean up the data structure associated with a viewport item.
 *
 *--------------------------------------------------------------
 */

static void
DeleteViewport(canvas, itemPtr, dpy)
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    Display *dpy;
{
    Tcl_HashEntry *t;
    CurrentTexture *curr;
    PortItem *port = (PortItem *) itemPtr;
    CellHeader *header = SmGetCellHeader(port->interp);

    Tcl_DeleteHashTable(&port->visibleHash);

    if (port->ambientLight != NULL) {
	Tk_FreeColor(port->ambientLight);
    }
    if (port->outlineColor != NULL) {
	Tk_FreeColor(port->outlineColor);
    }
    if (port->fillColor != NULL) {
	Tk_FreeColor(port->fillColor);
    }
    if (port->outlineGC != None) {
	Tk_FreeGC(dpy, port->outlineGC);
    }
    if (port->cell != NULL) {
	SmRemovePortFromCell(port->cell, port);
    }
    if (port->glxpm != None) {
	glXDestroyGLXPixmap(port->dpy, port->glxpm);
    }
    if (port->glxpmGC != None) {
	t = Tcl_FindHashEntry(&header->glxGC, (char *) port->glxpmGC);
	curr = (CurrentTexture *) Tcl_GetHashValue(t);
	if (--curr->refCount == 0) {
	    Tcl_DeleteHashEntry(t);
	    (void) ckfree((void *) curr);
	}
	glXDestroyContext(port->dpy, port->glxpmGC);
    }
    if (port->pm != None) {
	XFreePixmap(port->dpy, port->pm);
    }
    if (port->pmGC != None) {
	Tk_FreeGC(dpy, port->pmGC);
    }
    if (port->visinfo != NULL) {
	(void) XFree((char *) port->visinfo);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeViewportBbox:
 *
 * compute the bounding box of all the pixels that may be drawn
 * as part of a viewport.
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
static void
ComputeViewportBbox(canvas, port)
    Tk_Canvas canvas;
    PortItem *port;
{
    int bloat;

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (port->bbox[1] > port->bbox[3]) {
	double tmp;
	tmp = port->bbox[3];
	port->bbox[3] = port->bbox[1];
	port->bbox[1] = tmp;
    }
    if (port->bbox[0] > port->bbox[2]) {
	double tmp;
	tmp = port->bbox[2];
	port->bbox[2] = port->bbox[0];
	port->bbox[0] = tmp;
    }

    if (port->outlineColor == NULL) {
	bloat = 0;
    } else {
	bloat = (port->width+1)/2;
    }
    port->header.x1 = port->bbox[0] - bloat;
    port->header.y1 = port->bbox[1] - bloat;
    port->header.x2 = port->bbox[2] + bloat + 1;
    port->header.y2 = port->bbox[3] + bloat + 1;
}

/*
 *--------------------------------------------------------------
 *
 * SmGetVisualInfo:
 *
 * return visualinfo for a window.
 *
 *--------------------------------------------------------------
 */

XVisualInfo *
SmGetVisualInfo(dpy,  tkwin, doublebuffer)
     Display *dpy;
     Tk_Window tkwin;
     int doublebuffer;
{
    int i, depth, nitems, value;
    XWindowAttributes attr;
    XVisualInfo vinfo, *visinfo, *vis;
    long vinfoMask;

#ifdef MESA
    int attrs[] = {
	GLX_RGBA,	1,
	GLX_RED_SIZE,	2,
	GLX_GREEN_SIZE,	2,
	GLX_BLUE_SIZE,	2,
	GLX_DEPTH_SIZE,	16,
	GLX_DOUBLEBUFFER, 1,
	None,		1,
    };

    if (!doublebuffer) {
	attrs[10] = None;
    }
#endif

    /* get parent visual info */
    Tk_MapWindow(tkwin);

    depth = Tk_Depth(tkwin);
    vinfoMask = VisualIDMask;

    XGetWindowAttributes(dpy, Tk_WindowId(tkwin), &attr);
    vinfo.visualid = XVisualIDFromVisual(attr.visual);
    visinfo = XGetVisualInfo(dpy, vinfoMask, &vinfo, &nitems);
    if (visinfo == NULL) return NULL;

    /* check returned visuals for required visual attributes */

    for (i = 0, vis = visinfo; i < nitems; i++, vis++) {

	/* USE_GL */
	(void) glXGetConfig(dpy, vis, GLX_USE_GL, &value);
	if (value == False) continue;

	/* RGBA */
	(void) glXGetConfig(dpy, vis, GLX_RGBA, &value);
	if (value == False) continue;

	/* RED, GREEN, and BLUE sizes */
	(void) glXGetConfig(dpy, vis, GLX_RED_SIZE, &value);
	if (value < (int) (depth / 3)) continue;

	(void) glXGetConfig(dpy, vis, GLX_GREEN_SIZE, &value);
	if (value < (int) (depth / 3)) continue;

	(void) glXGetConfig(dpy, vis, GLX_BLUE_SIZE, &value);
	if (value < (int) (depth / 3)) continue;

	/* DEPTH_SIZE */
	(void) glXGetConfig(dpy, vis, GLX_DEPTH_SIZE, &value);
	if (value < 16) continue;

	/* DOUBLEBUFFER */
	if (doublebuffer) {
	    (void) glXGetConfig(dpy, vis, GLX_DOUBLEBUFFER, &value);
	    if (value == False) continue;
	}

#ifdef MESA
	return glXChooseVisual(dpy, vis->screen, attrs);
#else
	return vis;
#endif

    }
    (void) XFree((char *) visinfo);
    return NULL;
}

/*
 *--------------------------------------------------------------
 *
 * SmDisplayViewport:
 *
 * draw a viewport item in a given drawable.
 *
 *--------------------------------------------------------------
 */

void
SmDisplayViewport(canvas, itemPtr, dpy, drawable, x, y, width, height)
     Tk_Canvas canvas;
     Tk_Item *itemPtr;
     Display *dpy;
     Drawable drawable;
     int x, y, width, height;
{
    short x1, y1, x2, y2;
    int w, h;
    int srcx, srcy, tx, ty;
    int srcw, srch;
    int tgtx, tgty;
    int immediate, extra;
    unsigned int twidth, theight, tborder, tdepth;
    Window troot;
    GLbitfield clear_mask;

    PortItem *port = (PortItem *) itemPtr;
    Cell *cell = port->cell;
    Tk_Window tkwin;

    if ((width <= 0) || (height <= 0)) return;
    if ((port->pm == None) && (port->tkwin == NULL)) return;

    if (canvas) {
	tkwin = Tk_CanvasTkwin(canvas);
	immediate = (drawable == Tk_WindowId(tkwin));

	/*
	 * Compute the screen coordinates of the bounding box for the item.
	 * Make sure that the bbox is at least one pixel large, since some
	 * X servers will die if it isn't.
	 */

	if (immediate) {
	    Tk_CanvasWindowCoords(canvas, port->bbox[0], port->bbox[1], &x1, &y1);
	    Tk_CanvasWindowCoords(canvas, port->bbox[2], port->bbox[3], &x2, &y2);
	}
	else {
	    Tk_CanvasDrawableCoords(canvas, port->bbox[0], port->bbox[1], &x1, &y1);
	    Tk_CanvasDrawableCoords(canvas, port->bbox[2], port->bbox[3], &x2, &y2);
	}

	w = x2 - x1;
	h = y2 - y1;
	if (w <= 0) w = 1;
	if (h <= 0) h = 1;
    
	/*
	 * Display filled part first (if wanted), then cell models, then
	 * outline (if wanted).
	 */

	XGetGeometry(dpy, drawable, &troot, &tx, &ty, &twidth, &theight, &tborder, &tdepth);

	if (immediate) {
	    
	    /* need to get canvas's border width and highlight thickness */

	    extra = ((TkCanvas *) canvas)->inset;
	    twidth -= (extra + extra);
	    theight -= (extra + extra);
	    if (x1 < extra) {
		srcx = extra;
		srcw = w + x1 - extra;
	    }
	    else {
		srcx = x1;
		srcw = w;
	    }
	    if (y1 < extra) {
		srcy = extra;
		srch = h + y1 - extra;
	    }
	    else {
		srcy = y1;
		srch = h;
	    }
	    if (srcx + srcw > twidth) {
		srcw = twidth - srcx;
	    }
	    if (srcy + srch > theight) {
		srch = theight - srcy;
	    }
	}
	else {
	    if (x1 < tx) {
		srcx = tx;
		srcw = w - (tx - x1);
	    }
	    else {
		srcx = x1;
		srcw = w;
	    }
	    if (y1 < ty) {
		srcy = ty;
		srch = h - (ty - y1);
	    }
	    else {
		srcy = y1;
		srch = h;
	    }
	    if (srcx + srcw > tx + twidth) {
		srcw = tx + twidth - srcx;
	    }
	    if (srcy + srch > ty + theight) {
		srch = ty + theight - srcy;
	    }
	}
	if ((srcw < 1) || (srch < 1)) return;
	tgtx = (srcx - x1);
	tgty = (srcy - y1);

	/* force redraw if previous drawing area does not full contain current drawing area */
	if ((!port->redraw) &&
	    ((tgtx < port->srcx) ||
	     (tgty < port->srcy) ||
	     (tgtx + srcw > port->srcx + port->srcw) ||
	     (tgty + srch > port->srcy + port->srch))) {
	    port->redraw = 1;

#ifdef DEBUG_VIEW
	printf("current drawing area: (%d %d %u %u), previous drawing area: (%d %d %u %u)\n",
	       tgtx, tgty, srcw, srch,
	       port->srcx, port->srcy, port->srcw, port->srch);
#endif
	}
    } else {
	w = width - port->width;
	h = height - port->width;
	x1 = y1 = port->width / 2;
    }

    if (port->redraw) {
	    
	/* draw models */

	SmSelectViewport(port);

	/* set up color buffer clearing value */
	glClearColor((GLclampf) port->fillColor->red / (DOUBLE) 65535.0,
		     (GLclampf) port->fillColor->green / (DOUBLE) 65535.0,
		     (GLclampf) port->fillColor->blue / (DOUBLE) 65535.0,
		     0.0);

	/* clear buffers */
	clear_mask = GL_COLOR_BUFFER_BIT;

	if (port->hsr) {
	    glEnable(GL_DEPTH_TEST);
	    clear_mask |= GL_DEPTH_BUFFER_BIT;
	} else {
	    glDisable(GL_DEPTH_TEST);
	}
	glClear(clear_mask);

	/* enable lighting */
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, port->ambient);
	SetupLightSources(port);

	if (port->depthcue.on) {

	    glHint(GL_FOG_HINT, port->fogHint);

	    glFogi(GL_FOG_MODE, port->depthcue.mode);
	    glFogfv(GL_FOG_COLOR, port->depthcue.RGBA);
	    if ((port->depthcue.mode == GL_EXP) || (port->depthcue.mode == GL_EXP2)) {
		glFogf(GL_FOG_DENSITY, (float) port->depthcue.density);
	    } else if (port->depthcue.mode == GL_LINEAR) {
		glFogf(GL_FOG_START, (float) port->depthcue.start);
		glFogf(GL_FOG_END, (float) port->depthcue.end);
	    }
	    glEnable(GL_FOG);
	} else {
	    glDisable(GL_FOG);
	}

	if ((!port->invalid) && (cell != NULL)) {
	    SmDisplayModels(port, 0);
	}
	if (canvas) {
	    port->srcx = tgtx;
	    port->srcy = tgty;
	    port->srcw = srcw;
	    port->srch = srch;
	    port->redraw = 0;

#ifdef DEBUG_VIEW
	    printf("drawing area: %d %d %u %u, destination origin: %d %d\n", tgtx, tgty, srcw, srch, srcx, srcy);
#endif
	} else {
	    if (port->doublebuffer) {
		glXSwapBuffers(dpy, drawable);
	    }
	}
	glXWaitGL();
    }

    if (canvas) {
	if ((port->pmGC != None) && (port->pm != None)) {
	    XCopyArea(dpy, port->pm, drawable, port->pmGC, tgtx, tgty,
		      srcw, srch, srcx, srcy);
	}
    }

    if (port->outlineGC != None) {
	XDrawRectangle(dpy, drawable, port->outlineGC, x1, y1, w, h);
    }
}

/*
 *--------------------------------------------------------------
 *
 * SetupLightSources:
 *
 * set up all light sources.
 *
 *--------------------------------------------------------------
 */
static void
SetupLightSources(port)
     PortItem *port;
{
    int i, visible;
    LightSource *l;
    Tcl_HashEntry *e;
    Model *model, *m;
    GLfloat x, y, z, nx, ny, nz, position[4];
    GLfloat di, dj, dk, ni, nj, nk, direction[3];
    static GLenum source[] = {
	GL_LIGHT0,
	GL_LIGHT1,
	GL_LIGHT2,
	GL_LIGHT3,
	GL_LIGHT4,
	GL_LIGHT5,
	GL_LIGHT6,
	GL_LIGHT7
    };

    for (i = 0; i < 8; i++) {
	l = &port->lights[i];

	if (l->on) {

	    if ((model = l->model)) {
		e = Tcl_FindHashEntry(&port->visibleHash, model->label);
		if (model->visible
		    && (((e == NULL) && !port->invertedVisibleMask) ||
			((e != NULL) && port->invertedVisibleMask))) {
		    visible = 1;
		    x = model->modelT[0] * l->position[0] + model->modelT[4] * l->position[1] + model->modelT[8] * l->position[2] + model->modelT[12];
		    y = model->modelT[1] * l->position[0] + model->modelT[5] * l->position[1] + model->modelT[9] * l->position[2] + model->modelT[13];
		    z = model->modelT[2] * l->position[0] + model->modelT[6] * l->position[1] + model->modelT[10] * l->position[2] + model->modelT[14];

		    di = model->modelT[0] * l->direction[0] + model->modelT[4] * l->direction[1] + model->modelT[8] * l->direction[2];
		    dj = model->modelT[1] * l->direction[0] + model->modelT[5] * l->direction[1] + model->modelT[9] * l->direction[2];
		    dk = model->modelT[2] * l->direction[0] + model->modelT[6] * l->direction[1] + model->modelT[10] * l->direction[2];

		    for (m = model->parent; m && visible; m = m->parent) {
			if (!m->visible) {
			    visible = 0;
			}
			else {

			    /* convert light source from local to parent coordinates */
			    nx = m->modelT[0] * x + m->modelT[4] * y + m->modelT[8] * z + m->modelT[12];
			    ny = m->modelT[1] * x + m->modelT[5] * y + m->modelT[9] * z + m->modelT[13];
			    nz = m->modelT[2] * x + m->modelT[6] * y + m->modelT[10] * z + m->modelT[14];

			    x = nx;
			    y = ny;
			    z = nz;

			    ni = m->modelT[0] * di + m->modelT[4] * dj + m->modelT[8] * dk;
			    nj = m->modelT[1] * di + m->modelT[5] * dj + m->modelT[9] * dk;
			    nk = m->modelT[2] * di + m->modelT[6] * dj + m->modelT[10] * dk;

			    di = ni;
			    dj = nj;
			    dk = nk;
			}
		    }
		    if (visible) {
			position[0] = x;
			position[1] = y;
			position[2] = z;
			position[3] = l->position[3];

			direction[0] = di;
			direction[1] = dj;
			direction[2] = dk;

			glEnable(source[i]);

			/* create light source */
			glLightfv(source[i], GL_AMBIENT, l->ambientRGBA);
			glLightfv(source[i], GL_DIFFUSE, l->diffuseRGBA);
			glLightfv(source[i], GL_SPECULAR, l->specularRGBA);
			glLightf(source[i], GL_SPOT_EXPONENT, l->exponent);
			glLightf(source[i], GL_SPOT_CUTOFF, l->cutoff);
			glLightf(source[i], GL_CONSTANT_ATTENUATION, l->attenuation[0]);
			glLightf(source[i], GL_LINEAR_ATTENUATION, l->attenuation[1]);
			glLightf(source[i], GL_QUADRATIC_ATTENUATION, l->attenuation[2]);

			glLightfv(source[i], GL_SPOT_DIRECTION, direction);
			glLightfv(source[i], GL_POSITION, position);
		    } else {
			glDisable(source[i]);
		    }
		} else {
		    glDisable(source[i]);
		}
	    }
	    else {

		glEnable(source[i]);

		/* create light source */
		glLightfv(source[i], GL_AMBIENT, l->ambientRGBA);
		glLightfv(source[i], GL_DIFFUSE, l->diffuseRGBA);
		glLightfv(source[i], GL_SPECULAR, l->specularRGBA);
		glLightfv(source[i], GL_SPOT_DIRECTION, l->ndirection);
		glLightf(source[i], GL_SPOT_EXPONENT, l->exponent);
		glLightf(source[i], GL_SPOT_CUTOFF, l->cutoff);
		glLightf(source[i], GL_CONSTANT_ATTENUATION, l->attenuation[0]);
		glLightf(source[i], GL_LINEAR_ATTENUATION, l->attenuation[1]);
		glLightf(source[i], GL_QUADRATIC_ATTENUATION, l->attenuation[2]);
		glLightfv(source[i], GL_POSITION, l->position);
	    }
	} else {
	    glDisable(source[i]);
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortToPoint:
 *
 * computes the distance from a given point to a given viewport,
 * in canvas units.
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
    static double
PortToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    double *pointPtr;
{
    PortItem *port = (PortItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc, tmp;

    /*
     * Generate a new larger viewport that includes the border width, if there is one.
     */

    x1 = port->bbox[0];
    y1 = port->bbox[1];
    x2 = port->bbox[2];
    y2 = port->bbox[3];
    if (port->outlineGC != None) {
	inc = port->width/2.0;
	x1 -= inc;
	y1 -= inc;
	x2 += inc;
	y2 += inc;
    }

    /*
     * If the point is inside the viewport, handle specially:
     * distance is 0 if viewport is filled, otherwise compute
     * distance to nearest edge of viewport and subtract width
     * of edge.
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2) && (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
	if ((port->fillColor != NULL) || (port->outlineGC == None)) {
	    return 0.0;
	}
	xDiff = pointPtr[0] - x1;
	tmp = x2 - pointPtr[0];
	if (tmp < xDiff) {
	    xDiff = tmp;
	}
	yDiff = pointPtr[1] - y1;
	tmp = y2 - pointPtr[1];
	if (tmp < yDiff) {
	    yDiff = tmp;
	}
	if (yDiff < xDiff) {
	    xDiff = yDiff;
	}
	xDiff -= port->width;
	if (xDiff < 0.0) {
	    return 0.0;
	}
	return xDiff;
    }

    /*
     * Point is outside viewport.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
	xDiff = pointPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
	yDiff = pointPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * PortToArea:
 *
 * determine whether an item lies entirely inside, entirely
 * outside, or overlapping a given viewport.
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
static int
PortToArea(canvas, itemPtr, areaPtr)
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    double *areaPtr;
{
    PortItem *port = (PortItem *) itemPtr;
    double halfWidth;

    halfWidth = port->width/2.0;
    if (port->outlineGC == None) {
	halfWidth = 0.0;
    }

    if ((areaPtr[2] <= (port->bbox[0] - halfWidth))
	|| (areaPtr[0] >= (port->bbox[2] + halfWidth))
	|| (areaPtr[3] <= (port->bbox[1] - halfWidth))
	|| (areaPtr[1] >= (port->bbox[3] + halfWidth))) {
	return -1;
    }
    if ((port->fillColor == NULL) && (port->outlineGC != None)
	&& (areaPtr[0] >= (port->bbox[0] + halfWidth))
	&& (areaPtr[1] >= (port->bbox[1] + halfWidth))
	&& (areaPtr[2] <= (port->bbox[2] - halfWidth))
	&& (areaPtr[3] <= (port->bbox[3] - halfWidth))) {
	return -1;
    }
    if ((areaPtr[0] <= (port->bbox[0] - halfWidth))
	&& (areaPtr[1] <= (port->bbox[1] - halfWidth))
	&& (areaPtr[2] >= (port->bbox[2] + halfWidth))
	&& (areaPtr[3] >= (port->bbox[3] + halfWidth))) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleViewport:
 *
 * rescale a viewport item.
 *
 *--------------------------------------------------------------
 */

static void
ScaleViewport(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    double originX, originY;
    double scaleX;
    double scaleY;
{
    PortItem *port = (PortItem *) itemPtr;

    port->bbox[0] = originX + scaleX*(port->bbox[0] - originX);
    port->bbox[1] = originY + scaleY*(port->bbox[1] - originY);
    port->bbox[2] = originX + scaleX*(port->bbox[2] - originX);
    port->bbox[3] = originY + scaleY*(port->bbox[3] - originY);
    ComputeViewportBbox(canvas, port);

    SmCreateDrawable(port);
    port->invalid = SmComputeViewTransform(port);
    port->redraw = 1;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateViewport:
 *
 * move a viewport by a given amount.
 *
 *--------------------------------------------------------------
 */

static void
TranslateViewport(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;
    Tk_Item *itemPtr;
    double deltaX, deltaY;
{
    PortItem *port = (PortItem *) itemPtr;

    port->bbox[0] += deltaX;
    port->bbox[1] += deltaY;
    port->bbox[2] += deltaX;
    port->bbox[3] += deltaY;
    ComputeViewportBbox(canvas, port);
}

/*
 *--------------------------------------------------------------
 *
 * SmComputeViewTransform:
 *
 * compute the view transform matrix.
 *
 *--------------------------------------------------------------
 */

int
SmComputeViewTransform(port)
     PortItem *port;
{
    Tcl_Interp *interp = port->interp;
    double epsilon = port->epsilon;

    DOUBLE vup[3];
    DOUBLE rx[3], ry[3], rz[3];
    DOUBLE *mr, msh[16] = {
	1, 0, 0, 0,
	0, 1, 0, 0,
	0, 0, 1, 0,
	0, 0, 0, 1,
    };

    if (SmNormalizeVector3D(port->vpn, rz, epsilon)) {
	Tcl_AppendResult(interp, "invalid VPN (view projection normal) value.", (char *) NULL);
	return 1;
    }

    if (SmNormalizeVector3D(port->vup, ry, epsilon)) {
	Tcl_AppendResult(interp, "invalid VUP (view up) value.", (char *) NULL);
	return 1;
    }

    SmCrossVector3D(port->vup, rz, rx);
    if (SmNormalizeVector3D(rx, rx, epsilon)) {
	if (EqualVector3D(vup, rz, epsilon)) {
	    Tcl_AppendResult(interp, "VPN and VUP must not point in the same direction.", (char *) NULL);
	} else {
	    Tcl_AppendResult(interp, "invalid VUP (view up) value.", (char *) NULL);
	}
	return 1;
    }

    SmCrossVector3D(rz, rx, ry);

    if (Equal(port->maxu, port->minu, epsilon) ||
	Equal(port->maxv, port->minv, epsilon) ||
	(port->maxu <= port->minu) ||
	(port->maxv <= port->minv)) {
	Tcl_AppendResult(interp, "invalid view window specification.", (char *) NULL);
	return 1;
    }

    if (port->prp[2] <= epsilon) {
	Tcl_AppendResult(interp, "prp (projection reference point) must have a positive z component.", (char *) NULL);
	return 1;
    }

    memcpy(port->msh, msh, sizeof(DOUBLE) * 16);

    /* set up rotation matrix */
    mr = port->mr;
    mr[0] = rx[0];
    mr[1] = ry[0];
    mr[2] = rz[0];
    mr[3] = 0;

    mr[4] = rx[1];
    mr[5] = ry[1];
    mr[6] = rz[1];
    mr[7] = 0;

    mr[8] = rx[2];
    mr[9] = ry[2];
    mr[10] = rz[2];
    mr[11] = 0;

    mr[12] = 0;
    mr[13] = 0;
    mr[14] = 0;
    mr[15] = 1;

    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * SmSelectViewport:
 *
 * make viewport's transform matrices current.
 *
 *--------------------------------------------------------------
 */

void
SmSelectViewport(port)
     PortItem *port;
{
    /* set up the right OpenGL/Mesa context */
    if (port->canvas) {
	glXMakeCurrent(port->dpy, port->glxpm, port->glxpmGC);
    } else {
	glXMakeCurrent(port->dpy, Tk_WindowId(port->tkwin), port->glxpmGC);
    }
    glViewport(0, 0, port->w, port->h);

    /* set up PROJECTION transform matrix */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    if (!port->invalid) {

	switch(port->projection) {
	  case SM_PORT_PARALLEL:
	    port->msh[8] = -port->prp[0] / port->prp[2];
	    port->msh[9] = -port->prp[1] / port->prp[2];
	    glOrtho(port->minu - port->prp[0],
		    port->maxu - port->prp[0],
		    port->minv - port->prp[1],
		    port->maxv - port->prp[1],
		    port->prp[2], port->prp[2] - port->bp);
#ifdef USE_DOUBLE
	    glMultMatrixd(port->msh);
#else
	    glMultMatrixf(port->msh);
#endif
	break;

	  case SM_PORT_PERSPECTIVE:
	    glFrustum(port->minu - port->prp[0],
		      port->maxu - port->prp[0],
		      port->minv - port->prp[1],
		      port->maxv - port->prp[1],
		      port->prp[2], port->prp[2] - port->bp);
	    break;
	}
    }
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    if (!port->invalid) {
#ifdef USE_DOUBLE
	glTranslated(-port->prp[0], -port->prp[1], -port->prp[2]);
	glMultMatrixd(port->mr);
	glTranslated(-port->vrp[0], -port->vrp[1], -port->vrp[2]);
#else
	glTranslatef(-port->prp[0], -port->prp[1], -port->prp[2]);
	glMultMatrixf(port->mr);
	glTranslatef(-port->vrp[0], -port->vrp[1], -port->vrp[2]);
#endif
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmNormalizeVector3D:
 *
 * normalize a 3D vector that is represented as three scalar
 * values.
 *
 *--------------------------------------------------------------
 */

int
SmNormalizeVector3D(vin, vout, epsilon)
     DOUBLE vin[];
     DOUBLE vout[];
     double epsilon;
{
    DOUBLE mag;

    mag = (DOUBLE) sqrt((double) (vin[0] * vin[0] + vin[1] * vin[1] + vin[2] * vin[2]));
    if (mag <= epsilon) {
	vout[0] = vout[1] = vout[2] = 0;
	return 1;
    }
    mag = 1.0 / mag;
    vout[0] = vin[0] * mag;
    vout[1] = vin[1] * mag;
    vout[2] = vin[2] * mag;
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * SmCrossVector3D:
 *
 * compute the cross product of two 3D vectors, each of which
 * is represented by 3 scalar values.
 *
 *--------------------------------------------------------------
 */

void
SmCrossVector3D(a, b, c)
     DOUBLE a[];
     DOUBLE b[];
     DOUBLE c[];
{
    c[0] = a[1] * b[2] - a[2] * b[1];
    c[1] = a[2] * b[0] - a[0] * b[2];
    c[2] = a[0] * b[1] - a[1] * b[0];
}

/*
 *--------------------------------------------------------------
 *
 * Equal:
 *
 * determine if two floating point values are close enough
 * (within epsilon) to one another to be considered equal.
 *
 *--------------------------------------------------------------
 */

static int
Equal(DOUBLE a, DOUBLE b, double epsilon)
{
    return (fabs((double) (a - b)) <= epsilon) ? 1 : 0;
}

/*
 *--------------------------------------------------------------
 *
 * EqualVector3D:
 *
 * determine if two 3D vectors are close enough (each scalar
 * componnet within epsilon of its counterpart) to one another
 * to be considered equal.
 *
 *--------------------------------------------------------------
 */

static int
EqualVector3D(a, b, epsilon)
     DOUBLE a[], b[];
     double epsilon;
{
    return ((fabs((double) (a[0] - b[0])) <= epsilon)
	    && (fabs((double) (a[1] - b[1])) <= epsilon)
	    && (fabs((double) (a[2] - b[2])) <= epsilon));
}

/*
 *--------------------------------------------------------------
 *
 * SmDisplayModels:
 *
 * draw surfacea in a given drawable.
 *
 *--------------------------------------------------------------
 */

static float *fambientMat;
static float *fdiffuseMat;
static float *femissiveMat;
static float *fspecularMat;
static DOUBLE *fshininessMat;

static float *bambientMat;
static float *bdiffuseMat;
static float *bemissiveMat;
static float *bspecularMat;
static DOUBLE *bshininessMat;

void
SmDisplayModels(port, pick)
     PortItem *port;
     int pick;
{
    Model *model;
    GLuint name;

    double nvrpx, nvrpy, nvrpz;
    DOUBLE x0, y0, z0, x1, y1, z1;

    if (pick) {
	name = 1;
    } else {
	name = 0;

	/* perspective correction? */
	if (port->projection == SM_PORT_PERSPECTIVE) {
	    glHint(GL_PERSPECTIVE_CORRECTION_HINT, port->perspHint);
	}

	fambientMat = bambientMat = NULL;
	fdiffuseMat = bdiffuseMat = NULL;
	fspecularMat = bspecularMat = NULL;
	femissiveMat = bemissiveMat = NULL;
	fshininessMat = bshininessMat = NULL;

    }

    /* draw toplevel models */
    for (model = port->cell->models; model; model = model->next) {

	if (model->parent) continue;
	if (model->invalid) continue;
	if (SmGetBoundingBox(model, &x0, &y0, &z0, &x1, &y1, &z1, 1)) {
	    continue;
	}

	/* draw model only if model is potentially visible */
	/*
	 * 1. does the model extent lie completely behind the back clipping plane?
	 * 2. does the model extent lie completely behind the viewing plane?
	 * 3. does the 4 rays intersect the bounding box?
	 * 4. does any corner of the bounding box intersect the viewing voluem?
	 *
	 */

	nvrpx = port->vrp[0] + port->bp * port->mr[2];
	nvrpy = port->vrp[1] + port->bp * port->mr[6];
	nvrpz = port->vrp[2] + port->bp * port->mr[10];

	if ((WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x0, y0, z0) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x0, y0, z1) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x0, y1, z0) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x0, y1, z1) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x1, y0, z0) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x1, y0, z1) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x1, y1, z0) >= 0)
	    || (WHICH_SIDE(port->vpn, nvrpx, nvrpy, nvrpz, x1, y1, z1) >= 0)) {
	    if ((WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x0, y0, z0) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x0, y0, z1) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x0, y1, z0) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x0, y1, z1) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x1, y0, z0) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x1, y0, z1) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x1, y1, z0) <= 0)
		|| (WHICH_SIDE(port->vpn, port->vrp[0], port->vrp[1], port->vrp[2], x1, y1, z1) <= 0)) {
		name = DrawModel(port, model, name);
	    }
	    else {
		model->pickname = 0;
	    }
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * SetMaterial:
 *
 * change current material properties
 *
 *--------------------------------------------------------------
 */

static void
SetMaterial(face, ambientRGBA, diffuseRGBA, emissiveRGBA, specularRGBA, shininess)
     GLenum face;
     float *ambientRGBA;
     float *diffuseRGBA;
     float *emissiveRGBA;
     float *specularRGBA;
     DOUBLE *shininess;
{
    float *ambientMat = (face == GL_FRONT) ? fambientMat : bambientMat;
    float *diffuseMat = (face == GL_FRONT) ? fdiffuseMat : bdiffuseMat;
    float *emissiveMat = (face == GL_FRONT) ? femissiveMat : bemissiveMat;
    float *specularMat = (face == GL_FRONT) ? fspecularMat : bspecularMat;
    DOUBLE *shininessMat = (face == GL_FRONT) ? fshininessMat : bshininessMat;

    if ((!ambientMat) ||
	(ambientMat[0] != ambientRGBA[0]) ||
	(ambientMat[1] != ambientRGBA[1]) ||
	(ambientMat[2] != ambientRGBA[2]) ||
	(ambientMat[3] != ambientRGBA[3])) {
	glMaterialfv(face, GL_AMBIENT, ambientRGBA);
	if (face == GL_FRONT) 
	  fambientMat = ambientRGBA;
	else
	  bambientMat = ambientRGBA;
    }
    if ((!diffuseMat) ||
	(diffuseMat[0] != diffuseRGBA[0]) ||
	(diffuseMat[1] != diffuseRGBA[1]) ||
	(diffuseMat[2] != diffuseRGBA[2]) ||
	(diffuseMat[3] != diffuseRGBA[3])) {
	glMaterialfv(face, GL_DIFFUSE, diffuseRGBA);
	if (face == GL_FRONT)
	  fdiffuseMat = diffuseRGBA;
	else
	  bdiffuseMat = diffuseRGBA;
    }
    if ((!emissiveMat) ||
	(emissiveMat[0] != emissiveRGBA[0]) ||
	(emissiveMat[1] != emissiveRGBA[1]) ||
	(emissiveMat[2] != emissiveRGBA[2]) ||
	(emissiveMat[3] != emissiveRGBA[3])) {
	glMaterialfv(face, GL_EMISSION, emissiveRGBA);
	if (face == GL_FRONT)
	  femissiveMat = emissiveRGBA;
	else
	  bemissiveMat = emissiveRGBA;
    }
    if ((!specularMat) ||
	(specularMat[0] != specularRGBA[0]) ||
	(specularMat[1] != specularRGBA[1]) ||
	(specularMat[2] != specularRGBA[2]) ||
	(specularMat[3] != specularRGBA[3])) {
	glMaterialfv(face, GL_SPECULAR, specularRGBA);
	if (face == GL_FRONT)
	  fspecularMat = specularRGBA;
	else
	  bspecularMat = specularRGBA;
    }
    if ((!shininessMat) || (*shininessMat != *shininess)) {
	glMaterialf(face, GL_SHININESS, (float) *shininess);
	if (face == GL_FRONT)
	  fshininessMat = shininess;
	else
	  bshininessMat = shininess;
    }
}

/*
 *--------------------------------------------------------------
 *
 * DrawModel:
 *
 * recursivel draw an hierarchical model.
 *
 *--------------------------------------------------------------
 */

static int
DrawModel(port, model, name)
     PortItem *port;
     Model *model;
     GLuint name;
{
    int i, j, *index, modelMaterial;
    int useTexture;
    Vertex *v;
    Surface *s;
    Model *child;
    DOUBLE *size;

    TexCoords *c;
    Material *mat;
    Tcl_HashEntry *e, *t;
    CurrentTexture *curr;
    Tcl_Interp *interp = port->interp;
    CellHeader *header = SmGetCellHeader(interp);

    glPushMatrix();

#ifdef USE_DOUBLE
    glMultMatrixd(model->modelT);
#else
    glMultMatrixf(model->modelT);
#endif

    modelMaterial = 0;
    e = Tcl_FindHashEntry(&port->visibleHash, model->label);
    t = Tcl_FindHashEntry(&header->glxGC, (char *) port->glxpmGC);
    curr = (CurrentTexture *) Tcl_GetHashValue(t);

    if (model->visible
	&& ((model->type > SM_TYPE_POLYGON) || ((model->ns > 0) && (model->nv > 0)))
	&& (((e == NULL) && !port->invertedVisibleMask) ||
	    ((e != NULL) && port->invertedVisibleMask))) {

	/* model shading model */

	if (name != 0) {
	    model->pickname = name;
	    glLoadName(name++);
	}

	glShadeModel(model->shading);

	if (model->type >= SM_TYPE_POLYGON) {

	    /* select front facing orientation */
	    if (model->ccw) {
		glFrontFace(GL_CCW);
	    } else {
		glFrontFace(GL_CW);
	    }

            /* front/back face culling */
            if (model->cull) {
                glCullFace(model->cull);
                glEnable(GL_CULL_FACE);
		if (model->cull != GL_FRONT) {
                    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
		}
		else {
                    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
		}
            } else {
                glDisable(GL_CULL_FACE);
                glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
            }

	    /* polygon drawing mode */
	    glPolygonMode(GL_FRONT, model->frontmode);
	    glPolygonMode(GL_BACK, model->backmode);

	    if (model->type > SM_TYPE_POLYGON) glDisable(GL_TEXTURE_2D);
	}

	/* assign name for picking, if necessary */
	if (name != 0) {
	    glDisable(GL_LIGHTING);
	    useTexture = 0;
	    glDisable(GL_TEXTURE_2D);
	} else {
	    glEnable(GL_LIGHTING);

	    /* Model material */
	    SetMaterial(GL_FRONT, model->ambientRGBA, model->diffuseRGBA, model->emissiveRGBA,
			model->specular.RGBA, &model->specular.exponent);
	    if (model->type >= SM_TYPE_POLYGON) {
		SetMaterial(GL_BACK, model->bfambientRGBA, model->bfdiffuseRGBA, model->bfemissiveRGBA,
			    model->bfspecular.RGBA, &model->bfspecular.exponent);
	    }
	    modelMaterial = 1;
	}

	if (model->type == SM_TYPE_CYLINDER) {
	    gluQuadricNormals(model->quadric, model->quadNormals);
	    gluQuadricOrientation(model->quadric, model->orientation);
	    gluQuadricDrawStyle(model->quadric, model->drawStyle);
	    gluCylinder(model->quadric, (GLdouble) model->baseRadius,
			(GLdouble) model->topRadius, (GLdouble) model->height,
			(GLint) model->slices, (GLint) model->stacks);
	}
	else if (model->type == SM_TYPE_DISK) {
	    gluQuadricNormals(model->quadric, model->quadNormals);
	    gluQuadricOrientation(model->quadric, model->orientation);
	    gluQuadricDrawStyle(model->quadric, model->drawStyle);
	    gluDisk(model->quadric, (GLdouble) model->innerRadius,
		    (GLdouble) model->outerRadius, (GLint) model->slices, (GLint) model->loops);
	}
	else if (model->type == SM_TYPE_SPHERE) {
	    gluQuadricNormals(model->quadric, model->quadNormals);
	    gluQuadricOrientation(model->quadric, model->orientation);
	    gluQuadricDrawStyle(model->quadric, model->drawStyle);
	    gluSphere(model->quadric, (GLdouble) model->radius,
		      (GLint) model->slices, (GLint) model->stacks);
	}

	/********************/
	/* POLYGON          */
	/********************/

	else if (model->type == SM_TYPE_POLYGON) {
	    for (i = 0, s = model->s; i < (const) model->ns; i++, s++) {

		if (name == 0) {
		    if ((mat = s->materials) == NULL) {
			if (!modelMaterial) {

			    /* Model material */
			    SetMaterial(GL_FRONT, model->ambientRGBA, model->diffuseRGBA, model->emissiveRGBA,
					model->specular.RGBA, &model->specular.exponent);
			    modelMaterial = 1;
			}
		    } else {
			modelMaterial = 0;
		    }

		    /* set up texture map, if necessary */
		    if (model->useTexture && port->texture && s->texture) {
			if (s->texture->refCount < 0) {
			    SmFreeTexture(interp, header, s->texture);
			    s->texture = NULL;
			    glDisable(GL_TEXTURE_2D);
			    useTexture = 0;
			}
			else {
			    glEnable(GL_TEXTURE_2D);
			    useTexture = 1;

			    if ((curr->component != s->component) || (curr->texture != s->texture)) {
				curr->component = s->component;
				curr->texture = s->texture;
				glTexImage2D(GL_TEXTURE_2D, 0, curr->component, curr->texture->width,
					     curr->texture->height, curr->texture->border, GL_RGB, GL_UNSIGNED_BYTE,
					     (GLvoid *) curr->texture->data);
			    }
			    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, s->func);
			    if (s->func == GL_BLEND) {
				glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, s->texColor);
			    }
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, s->smode);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, s->tmode);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, s->magFilter);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, s->minFilter);
			    glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, s->bdColor);
			}
		    }
		    else {
			useTexture = 0;
			glDisable(GL_TEXTURE_2D);
		    }
		}

		glBegin(GL_POLYGON);
		if (s->normal == SM_VERTEX_NORMAL_DEFAULT) {

		    if (model->useAverage) {
			
			/* use per-vertex normal obtained from adjacent surface normals */
			for (j = 0, v = model->v, index = s->index, c = s->texcoords; j < (const) s->vcnt; j++, index++, c++) {

			    if (name == 0) {
				if (mat) {
				    SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
						mat->specularRGBA, &mat->shininess);
				    mat++;
				}
				
#ifdef USE_DOUBLE
				glNormal3dv(v[*index].lvn.v);
				if (useTexture) {
				    glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
				}
#else
				glNormal3fv(v[*index].lvn.v);
				if (useTexture) {
				    glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
				}
#endif
			    }

#ifdef USE_DOUBLE			    
			    glVertex3dv(v[*index].lv);
#else
			    glVertex3fv(v[*index].lv);
#endif
			}
		    }
		    else {

			/* use surface normal */

#ifdef USE_DOUBLE
			glNormal3dv(s->ln);
#else
			glNormal3fv(s->ln);
#endif
			for (j = 0, v = model->v, index = s->index, c = s->texcoords; j < (const) s->vcnt; j++, index++, c++) {

			    if ((name == 0) && mat) {
				SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					    mat->specularRGBA, &mat->shininess);
				mat++;
			    }

#ifdef USE_DOUBLE
			    if (useTexture) {
				glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			    }
			    glVertex3dv(v[*index].lv);
#else
			    if (useTexture) {
				glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			    }
			    glVertex3fv(v[*index].lv);
#endif
			}
		    }
		}
		else {
		
		    /* use user-specified vertex normals */
		    for (j = 0, v = model->v, index = s->index, c = s->texcoords; j < (const) s->vcnt; j++, index++, c++) {

			if (name == 0) {
			    if (mat) {
				SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					    mat->specularRGBA, &mat->shininess);
				mat++;
			    }

#ifdef USE_DOUBLE
			    glNormal3dv(s->lvn[j].v);
			    if (useTexture) {
				glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			    }
#else
			    glNormal3fv(s->lvn[j].v);
			    if (useTexture) {
				glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			    }
#endif

			}

#ifdef USE_DOUBLE
			glVertex3dv(v[*index].lv);
#else
			glVertex3fv(v[*index].lv);
#endif
		    }
		}
		glEnd();
	    }
	}

	/********************/
	/* LINE             */
	/********************/

	else if (model->type == SM_TYPE_LINE) {
	    for (i = 0, s = model->s; i < (const) model->ns; i++, s++) {

		if (name == 0) {
		    if ((mat = s->materials) == NULL) {
			if (!modelMaterial) {

			    /* Model material */
			    SetMaterial(GL_FRONT, model->ambientRGBA, model->diffuseRGBA, model->emissiveRGBA,
					model->specular.RGBA, &model->specular.exponent);
			    modelMaterial = 1;
			}
		    } else {
			modelMaterial = 0;
		    }

		    /* set up texture map, if necessary */
		    if (port->texture && s->texture) {
			if (s->texture->refCount < 0) {
			    SmFreeTexture(interp, header, s->texture);
			    s->texture = NULL;
			    glDisable(GL_TEXTURE_2D);
			    useTexture = 0;
			}
			else {
			    glEnable(GL_TEXTURE_2D);
			    useTexture = 1;
			    if ((curr->component != s->component) || (curr->texture != s->texture)) {
				curr->component = s->component;
				curr->texture = s->texture;
				glTexImage2D(GL_TEXTURE_2D, 0, curr->component, curr->texture->width,
					     curr->texture->height, curr->texture->border, GL_RGB, GL_UNSIGNED_BYTE,
					     (GLvoid *) curr->texture->data);
			    
			    }
			    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, s->func);
			    if (s->func == GL_BLEND) {
				glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, s->texColor);
			    }
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, s->smode);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, s->tmode);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, s->magFilter);
			    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, s->minFilter);
			    glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, s->bdColor);
			}
		    }
		    else {
			useTexture = 0;
			glDisable(GL_TEXTURE_2D);
		    }
		}

		glLineWidth((GLfloat) s->lineWidth);
		if ((s->lineFactor >= 1) && (s->lineFactor <= 255)) {
		    glEnable(GL_LINE_STIPPLE);
		    glLineStipple((GLint) s->lineFactor, (GLushort) s->lineStipple);
		}
		else {
		    glDisable(GL_LINE_STIPPLE);
		}

		glBegin(GL_LINE_STRIP);

		if (s->normal == SM_VERTEX_NORMAL_DEFAULT) {

		    /* use default normal */
		    glNormal3f((GLfloat) 0.0, (GLfloat) 0.0, (GLfloat) 1.0);

		    for (j = 0, v = model->v, index = s->index, c = s->texcoords; j < (const) s->vcnt; j++, index++, c++) {

			if (name == 0) {
			    if (mat) {
				SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					    mat->specularRGBA, &mat->shininess);
				mat++;
			    }

#ifdef USE_DOUBLE
			    if (useTexture) {
				glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			    }
#else
			    if (useTexture) {
				glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			    }
#endif
			}

#ifdef USE_DOUBLE
			glVertex3dv(v[*index].lv);
#else
			glVertex3fv(v[*index].lv);
#endif
		    }
		} else {
		
		    /* use user-specified vertex normals */

		    for (j = 0, v = model->v, index = s->index, c = s->texcoords; j < (const) s->vcnt; j++, index++, c++) {

			if (name == 0) {
			    if (mat) {
				SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					    mat->specularRGBA, &mat->shininess);
				mat++;
			    }

#ifdef USE_DOUBLE
			    glNormal3dv(s->lvn[j].v);
			    if (useTexture) {
				glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			    }
#else
			    glNormal3fv(s->lvn[j].v);
			    if (useTexture) {
				glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			    }
#endif
			}

#ifdef USE_DOUBLE
			glVertex3dv(v[*index].lv);
#else
			glVertex3fv(v[*index].lv);
#endif
		    }
		}
		glEnd();
	    }
	}
	/********************/
	/* POINT             */
	/********************/

	else if (model->type == SM_TYPE_POINT) {
	    s = model->s;
	    if (name == 0) {
		if ((mat = s->materials) == NULL) {
		    if (!modelMaterial) {

			/* Model material */
			SetMaterial(GL_FRONT, model->ambientRGBA, model->diffuseRGBA, model->emissiveRGBA,
				    model->specular.RGBA, &model->specular.exponent);
			modelMaterial = 1;
		    }
		} else {
		    modelMaterial = 0;
		}

		/* set up texture map, if necessary */
		if (port->texture && s->texture) {
		    if (s->texture->refCount < 0) {
			SmFreeTexture(interp, header, s->texture);
			s->texture = NULL;
			glDisable(GL_TEXTURE_2D);
			useTexture = 0;
		    }
		    else {
			glEnable(GL_TEXTURE_2D);
			useTexture = 1;
			if ((curr->component != s->component) || (curr->texture != s->texture)) {
			    curr->component = s->component;
			    curr->texture = s->texture;
			    glTexImage2D(GL_TEXTURE_2D, 0, curr->component, curr->texture->width,
					 curr->texture->height, curr->texture->border, GL_RGB, GL_UNSIGNED_BYTE,
					 (GLvoid *) curr->texture->data);
			}
			glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, s->func);
			if (s->func == GL_BLEND) {
			    glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, s->texColor);
			}
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, s->smode);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, s->tmode);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, s->magFilter);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, s->minFilter);
			glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, s->bdColor);
		    }
		}
		else {
		    useTexture = 0;
		    glDisable(GL_TEXTURE_2D);
		}
	    }

	    if (!(size = model->pointSizes)) {
		glPointSize((GLfloat) model->pointSize);
	    }

	    if (s->normal == SM_VERTEX_NORMAL_DEFAULT) {

		/* use default normal */
		glNormal3f((GLfloat) 0.0, (GLfloat) 0.0, (GLfloat) 1.0);

		if (!size) {
		    glBegin(GL_POINTS);
		}

		for (j = 0, v = model->v, c = s->texcoords; j < (const) model->nv; j++, v++, c++) {

		    if (name == 0) {
			if (mat) {
			    SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					mat->specularRGBA, &mat->shininess);
			    mat++;
			}

#ifdef USE_DOUBLE
			if (useTexture) {
			    glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			}
#else
			if (useTexture) {
			    glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			}
#endif
		    }
		    if (size) {
			glPointSize(*size++);
			glBegin(GL_POINTS);
		    }
#ifdef USE_DOUBLE
		    glVertex3dv(v->lv);
#else
		    glVertex3fv(v->lv);
#endif
		    if (size) {
			glEnd();
		    }
		}
		if (!size) {
		    glEnd();
		}
	    } else {
		
		/* use user-specified vertex normals */

		if (!size) {
		    glBegin(GL_POINTS);
		}

		for (j = 0, v = model->v, c = s->texcoords; j < (const) s->vcnt; j++, v++, c++) {

		    if (name == 0) {
			if (mat) {
			    SetMaterial(GL_FRONT, mat->ambientRGBA, mat->diffuseRGBA, mat->emissiveRGBA,
					mat->specularRGBA, &mat->shininess);
			}

#ifdef USE_DOUBLE
			glNormal3dv(s->lvn[j].v);
			if (useTexture) {
			    glTexCoord2d((GLdouble) c->v[0], (GLdouble) c->v[1]);
			}
#else
			glNormal3fv(s->lvn[j].v);
			if (useTexture) {
			    glTexCoord2f((GLfloat) c->v[0], (GLfloat) c->v[1]);
			}
#endif
		    }
		    if (size) {
			glPointSize((GLfloat) *size++);
			glBegin(GL_POINTS);
		    }

#ifdef USE_DOUBLE
		    glVertex3dv(v->lv);
#else
		    glVertex3fv(v->lv);
#endif
		    if (size) {
			glEnd();
		    }
		}
		if (!size) {
		    glEnd();
		}
	    }
	}

    }
    if (model->visible) {

	/* recursively draw children with current model's transform in effect */
	for (child = model->child; child; child = child->sibling) {
	    name = DrawModel(port, child, name);
	}
    }
    glPopMatrix();

    return name;
}

/*
 *--------------------------------------------------------------
 *
 * SmCreateDrawable:
 *
 * create window/pixmap used for rendering by OpenGL/Mesa.
 *
 *--------------------------------------------------------------
 */


void
SmCreateDrawable(port)
     PortItem *port;
{
    int new;
    Tk_Window tkwin;
    Tcl_HashEntry *t;
    CurrentTexture *curr;
    Display *dpy = port->dpy;
    CellHeader *header = SmGetCellHeader(port->interp);

#ifdef MESA
    Window win;
    XWindowAttributes attr;
    unsigned long pixel;
#endif

    if (port->canvas) {

	/* if canvas item, use pixmap as off-screen buffer */
	if (port->pm != None) {
	    XFreePixmap(dpy, port->pm);
	    port->pm = None;
	}
	if (port->glxpm != None) {
	    glXDestroyGLXPixmap(dpy, port->glxpm);
	    port->glxpm = None;
	}
	if ((port->w <= 0) || (port->h <= 0)) return;

	tkwin = Tk_CanvasTkwin(port->canvas);

	port->pm = XCreatePixmap(dpy, Tk_WindowId(tkwin), port->w, port->h, port->visinfo->depth);
	port->glxpm = glXCreateGLXPixmap(dpy, port->visinfo, port->pm);
	if (port->glxpmGC == None) {
	    port->glxpmGC = glXCreateContext(dpy, port->visinfo, NULL, False);
	    t = Tcl_CreateHashEntry(&header->glxGC, (char *) port->glxpmGC, &new);
	    if (new) {
		curr = (CurrentTexture *) ckalloc(sizeof(CurrentTexture));
		curr->texture = NULL;
		curr->component = 0;
		curr->refCount = 1;
		Tcl_SetHashValue(t, (ClientData) curr);
	    } else {
		curr = (CurrentTexture *) Tcl_GetHashValue(t);
		curr->refCount++;
	    }
	}

#ifdef MESA

    /* create a dummy window to make Mesa use the right window */
    /* size.    					       */

	pixel = BlackPixel(dpy, port->visinfo->screen);
	XGetWindowAttributes(dpy, Tk_WindowId(tkwin), &attr);
	win = XCreateSimpleWindow(dpy, Tk_WindowId(tkwin),
				  0, 0, port->w, port->h, 0, pixel, pixel);
	XSetWindowColormap(dpy, win, attr.colormap);
	glXMakeCurrent(dpy, win, port->glxpmGC);
#endif

	glXMakeCurrent(dpy, port->glxpm, port->glxpmGC);

#ifdef MESA
	XDestroyWindow(dpy, win);
#endif

    } else {
	if (port->glxpmGC == None) {
	    port->glxpmGC = glXCreateContext(dpy, port->visinfo, NULL, True);
	    t = Tcl_CreateHashEntry(&header->glxGC, (char *) port->glxpmGC, &new);
	    if (new) {
		curr = (CurrentTexture *) ckalloc(sizeof(CurrentTexture));
		curr->texture = NULL;
		curr->component = 0;
		curr->refCount = 1;
		Tcl_SetHashValue(t, (ClientData) curr);
	    } else {
		curr = (CurrentTexture *) Tcl_GetHashValue(t);
		curr->refCount++;
	    }
	}
	glXMakeCurrent(dpy, Tk_WindowId(port->tkwin), port->glxpmGC);
    }

    glClearDepth(1.0);
    glDisable(GL_NORMALIZE);
    glDepthFunc(GL_LESS);
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
}
