/* 
 * smvoption.c --
 *
 *	This file implements canvas viewport options.
 *
 */

#include "copyright.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "smInt.h"


/*
 * Prototypes for procedures defined externally
 */

Cell *		SmFindCell _ANSI_ARGS_ ((Tcl_Interp *interp, char *label));

Model		*SmFindModel _ANSI_ARGS_ ((Cell *cell, char *label));

void		SmInsertPortIntoCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

void		SmRemovePortFromCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

void		SmDisplayModels _ANSI_ARGS_ ((PortItem *port, int pick));

void		SmSelectViewport _ANSI_ARGS_ ((PortItem *port));

void            SmViewportUpdate _ANSI_ARGS_ ((ClientData clientData));

int		SmParseColorCoefficients _ANSI_ARGS_ ((Tcl_Interp *interp, char *value,
						       double *r, double *g, double *b));


/*
 * Prototypes for procedures defined in this file that are exported:
 */

int		SmVectorParseProc _ANSI_ARGS_ ((ClientData clientData,
						Tcl_Interp *interp, Tk_Window tkwin,
						char *value, char *widgRec, int offset));

char *		SmVectorPrintProc _ANSI_ARGS_ ((ClientData clientData,
						Tk_Window tkwin, char *widgRec, int offset,
						Tcl_FreeProc **freeProcPtr));

void		SmSetClientData _ANSI_ARGS_ ((ClientData clientData));

int		SmViewportChanged _ANSI_ARGS_ ((PortItem *port));


/*
 * Prototypes for procedures defined in this file that are static:
 */

static int	PortProjectionParseProc _ANSI_ARGS_ ((ClientData clientData,
						      Tcl_Interp *interp, Tk_Window tkwin,
						      char *value, char *widgRec, int offset));

static char *	PortProjectionPrintProc _ANSI_ARGS_ ((ClientData clientData,
						      Tk_Window tkwin, char *widgRec, int offset,
						      Tcl_FreeProc **freeProcPtr));

static int	PortCellParseProc _ANSI_ARGS_ ((ClientData clientData,
						Tcl_Interp *interp, Tk_Window tkwin,
						char *value, char *widgRec, int offset));

static char *	PortCellPrintProc _ANSI_ARGS_ ((ClientData clientData,
						Tk_Window tkwin, char *widgRec, int offset,
						Tcl_FreeProc **freeProcPtr));

static int	PortModelVisibleMaskParseProc _ANSI_ARGS_ ((ClientData clientData,
							    Tcl_Interp *interp, Tk_Window tkwin,
							    char *value, char *widgRec, int offset));

static char *	PortModelVisibleMaskPrintProc _ANSI_ARGS_ ((ClientData clientData,
							    Tk_Window tkwin, char *widgRec, int offset,
							    Tcl_FreeProc **freeProcPtr));

static int	PortModelPickParseProc _ANSI_ARGS_ ((ClientData clientData,
						     Tcl_Interp *interp, Tk_Window tkwin,
						     char *value, char *widgRec, int offset));

static char *	PortModelPickPrintProc _ANSI_ARGS_ ((ClientData clientData,
						     Tk_Window tkwin, char *widgRec, int offset,
						     Tcl_FreeProc **freeProcPtr));

static int	PortLightSourceParseProc _ANSI_ARGS_ ((ClientData clientData,
						       Tcl_Interp *interp, Tk_Window tkwin,
						       char *value, char *widgRec, int offset));

static char *	PortLightSourcePrintProc _ANSI_ARGS_ ((ClientData clientData,
						       Tk_Window tkwin, char *widgRec, int offset,
						       Tcl_FreeProc **freeProcPtr));

static int	PortDepthCueParseProc _ANSI_ARGS_ ((ClientData clientData,
						    Tcl_Interp *interp, Tk_Window tkwin,
						    char *value, char *widgRec, int offset));

static char *	PortDepthCuePrintProc _ANSI_ARGS_ ((ClientData clientData,
						    Tk_Window tkwin, char *widgRec, int offset,
						    Tcl_FreeProc **freeProcPtr));


static int	ParseLightDirection _ANSI_ARGS_ ((Tcl_Interp *interp, char *value, float dir[3], float ndir[3]));

static int	ParseLightPosition _ANSI_ARGS_ ((Tcl_Interp *interp, char *value, float pos[4]));

static int	ParseLightAttenuation _ANSI_ARGS_ ((Tcl_Interp *interp, char *value, float att[3]));

static int	PortHintParseProc _ANSI_ARGS_ ((ClientData clientData,
						Tcl_Interp *interp, Tk_Window tkwin,
						char *value, char *widgRec, int offset));

static char *	PortHintPrintProc _ANSI_ARGS_ ((ClientData clientData,
						Tk_Window tkwin, char *widgRec, int offset,
						Tcl_FreeProc **freeProcPtr));


/*
 * Custom options for viewport item type
 */

static Tk_CustomOption tk_ViewportOptionProjection = {
    PortProjectionParseProc,
    PortProjectionPrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption tk_ViewportOptionVRP = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_PORT_VRP,
};

static Tk_CustomOption tk_ViewportOptionVPN = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_PORT_VPN,
};

static Tk_CustomOption tk_ViewportOptionVUP = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_PORT_VUP,
};

static Tk_CustomOption tk_ViewportOptionPRP = {
    SmVectorParseProc,
    SmVectorPrintProc,
    (ClientData) SM_PORT_PRP,
};

static Tk_CustomOption tk_ViewportOptionLightSource0 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_0,
};

static Tk_CustomOption tk_ViewportOptionLightSource1 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_1,
};

static Tk_CustomOption tk_ViewportOptionLightSource2 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_2,
};

static Tk_CustomOption tk_ViewportOptionLightSource3 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_3,
};

static Tk_CustomOption tk_ViewportOptionLightSource4 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_4,
};

static Tk_CustomOption tk_ViewportOptionLightSource5 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_5,
};

static Tk_CustomOption tk_ViewportOptionLightSource6 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_6,
};

static Tk_CustomOption tk_ViewportOptionLightSource7 = {
    PortLightSourceParseProc,
    PortLightSourcePrintProc,
    (ClientData) SM_LIGHT_SOURCE_7,
};

static Tk_CustomOption tk_ViewportOptionCell = {
    PortCellParseProc,
    PortCellPrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption tk_ViewportOptionVisibleMask = {
    PortModelVisibleMaskParseProc,
    PortModelVisibleMaskPrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption tk_ViewportOptionPick = {
    PortModelPickParseProc,
    PortModelPickPrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption tk_ViewportOptionDepthCue = {
    PortDepthCueParseProc,
    PortDepthCuePrintProc,
    (ClientData) NULL,
};

static Tk_CustomOption tk_ViewportOptionFogHint = {
    PortHintParseProc,
    PortHintPrintProc,
    (ClientData) SM_HINT_FOG,
};

static Tk_CustomOption tk_ViewportOptionPerspectiveHint = {
    PortHintParseProc,
    PortHintPrintProc,
    (ClientData) SM_HINT_PERSPECTIVE,
};

#ifdef USE_TCL75

static Tk_CustomOption tagsOption = {Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};

#endif


/*
 * Information used for parsing canvas viewport configuration specs:
 */

Tk_ConfigSpec canvasConfigSpecs[] = {

    {TK_CONFIG_COLOR, "-ambientlight", "ambientLight", "AmbientLight",
       "#000000000", Tk_Offset(PortItem, ambientLight), 0},

    {TK_CONFIG_DOUBLE, "-bp", "backplane", "Backplane",
       "-1e6", Tk_Offset(PortItem, bp), 0},

    {TK_CONFIG_CUSTOM, "-cell", "cell", "Cell",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionCell},

    {TK_CONFIG_CUSTOM, "-depthcue", "depthCue", "DepthCue",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionDepthCue},

    {TK_CONFIG_DOUBLE, "-epsilon", "epsilon", "Epsilon",
       "1e-6", Tk_Offset(PortItem, epsilon), 0},

    {TK_CONFIG_COLOR, "-fill", "background", "Background",
       "black", Tk_Offset(PortItem, fillColor), 0},

    {TK_CONFIG_CUSTOM, "-foghint", (char *) NULL, (char *) NULL,
       "none", 0, 0, &tk_ViewportOptionFogHint},

    {TK_CONFIG_BOOLEAN, "-hsr", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, hsr), 0},

    {TK_CONFIG_CUSTOM, "-light0", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource0},

    {TK_CONFIG_CUSTOM, "-light1", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource1},

    {TK_CONFIG_CUSTOM, "-light2", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource2},

    {TK_CONFIG_CUSTOM, "-light3", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource3},

    {TK_CONFIG_CUSTOM, "-light4", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource4},

    {TK_CONFIG_CUSTOM, "-light5", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource5},

    {TK_CONFIG_CUSTOM, "-light6", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource6},

    {TK_CONFIG_CUSTOM, "-light7", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource7},

    {TK_CONFIG_DOUBLE, "-maxu", (char *) NULL, (char *) NULL,
       "100.0", Tk_Offset(PortItem, maxu), 0},

    {TK_CONFIG_DOUBLE, "-maxv", (char *) NULL, (char *) NULL,
       "100.0", Tk_Offset(PortItem, maxv), 0},

    {TK_CONFIG_DOUBLE, "-minu", (char *) NULL, (char *) NULL,
       "-100.0", Tk_Offset(PortItem, minu), 0},

    {TK_CONFIG_DOUBLE, "-minv", (char *) NULL, (char *) NULL,
       "-100.0", Tk_Offset(PortItem, minv), 0},

    {TK_CONFIG_COLOR, "-outline", "foreground", "Foreground",
       "black", Tk_Offset(PortItem, outlineColor), TK_CONFIG_NULL_OK},

    {TK_CONFIG_CUSTOM, "-perspectivehint", (char *) NULL, (char *) NULL,
       "none", 0, 0, &tk_ViewportOptionPerspectiveHint},

    {TK_CONFIG_CUSTOM, "-pick", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionPick},

    {TK_CONFIG_CUSTOM, "-projection", (char *) NULL, (char *) NULL,
       "parallel", 0, 0, &tk_ViewportOptionProjection},

    {TK_CONFIG_CUSTOM, "-prp", (char *) NULL, (char *) NULL,
       "0.0 0.0 100.0", 0, 0, &tk_ViewportOptionPRP},

#ifdef USE_TCL75
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
       (char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
#else
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
       (char *) NULL, 0, TK_CONFIG_NULL_OK, &tk_CanvasTagsOption},
#endif

    {TK_CONFIG_BOOLEAN, "-texture", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, texture), 0},

    {TK_CONFIG_SYNONYM, "-vismask", "visibleMask", (char *) NULL,
       (char *) NULL, 0, 0, NULL},

    {TK_CONFIG_CUSTOM, "-visiblemask", "visibleMask", "VisibleMask",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionVisibleMask},

    {TK_CONFIG_CUSTOM, "-vpn", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &tk_ViewportOptionVPN},

    {TK_CONFIG_CUSTOM, "-vrp", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0",  0, 0, &tk_ViewportOptionVRP},

    {TK_CONFIG_CUSTOM, "-vup", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &tk_ViewportOptionVUP},

    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(PortItem, width), 0, NULL},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Information used for parsing viewport configuration specs:
 */

Tk_ConfigSpec widgetConfigSpecs[] = {

    {TK_CONFIG_COLOR, "-ambientlight", "ambientLight", "AmbientLight",
       "#000000000", Tk_Offset(PortItem, ambientLight), 0},

    {TK_CONFIG_BOOLEAN, "-asynchronous", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, async), 0},

    {TK_CONFIG_DOUBLE, "-bp", "backplane", "Backplane",
       "-1e6", Tk_Offset(PortItem, bp), 0},

    {TK_CONFIG_CUSTOM, "-cell", "cell", "Cell",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionCell},

    {TK_CONFIG_CUSTOM, "-depthcue", "depthCue", "DepthCue",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionDepthCue},

    {TK_CONFIG_BOOLEAN, "-doublebuffer", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, doublebuffer), 0},

    {TK_CONFIG_DOUBLE, "-epsilon", "epsilon", "Epsilon",
       "1e-6", Tk_Offset(PortItem, epsilon), 0},

    {TK_CONFIG_COLOR, "-fill", "background", "Background",
       "black", Tk_Offset(PortItem, fillColor), 0},

    {TK_CONFIG_CUSTOM, "-foghint", (char *) NULL, (char *) NULL,
       "none", 0, 0, &tk_ViewportOptionFogHint},

    {TK_CONFIG_BOOLEAN, "-hsr", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, hsr), 0},

    {TK_CONFIG_CUSTOM, "-light0", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource0},

    {TK_CONFIG_CUSTOM, "-light1", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource1},

    {TK_CONFIG_CUSTOM, "-light2", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource2},

    {TK_CONFIG_CUSTOM, "-light3", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource3},

    {TK_CONFIG_CUSTOM, "-light4", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource4},

    {TK_CONFIG_CUSTOM, "-light5", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource5},

    {TK_CONFIG_CUSTOM, "-light6", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource6},

    {TK_CONFIG_CUSTOM, "-light7", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionLightSource7},

    {TK_CONFIG_DOUBLE, "-maxu", (char *) NULL, (char *) NULL,
       "100.0", Tk_Offset(PortItem, maxu), 0},

    {TK_CONFIG_DOUBLE, "-maxv", (char *) NULL, (char *) NULL,
       "100.0", Tk_Offset(PortItem, maxv), 0},

    {TK_CONFIG_DOUBLE, "-minu", (char *) NULL, (char *) NULL,
       "-100.0", Tk_Offset(PortItem, minu), 0},

    {TK_CONFIG_DOUBLE, "-minv", (char *) NULL, (char *) NULL,
       "-100.0", Tk_Offset(PortItem, minv), 0},

    {TK_CONFIG_COLOR, "-outline", "foreground", "Foreground",
       "black", Tk_Offset(PortItem, outlineColor), TK_CONFIG_NULL_OK},

    {TK_CONFIG_CUSTOM, "-perspectivehint", (char *) NULL, (char *) NULL,
       "none", 0, 0, &tk_ViewportOptionPerspectiveHint},

    {TK_CONFIG_CUSTOM, "-pick", (char *) NULL, (char *) NULL,
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionPick},

    {TK_CONFIG_CUSTOM, "-projection", (char *) NULL, (char *) NULL,
       "parallel", 0, 0, &tk_ViewportOptionProjection},

    {TK_CONFIG_CUSTOM, "-prp", (char *) NULL, (char *) NULL,
       "0.0 0.0 100.0", 0, 0, &tk_ViewportOptionPRP},

    {TK_CONFIG_BOOLEAN, "-texture", (char *) NULL, (char *) NULL,
       "on", Tk_Offset(PortItem, texture), 0},

    {TK_CONFIG_SYNONYM, "-vismask", "visibleMask", (char *) NULL,
       (char *) NULL, 0, 0, NULL},

    {TK_CONFIG_CUSTOM, "-visiblemask", "visibleMask", "VisibleMask",
       "", 0, TK_CONFIG_NULL_OK, &tk_ViewportOptionVisibleMask},

    {TK_CONFIG_CUSTOM, "-vpn", (char *) NULL, (char *) NULL,
       "0.0 0.0 1.0", 0, 0, &tk_ViewportOptionVPN},

    {TK_CONFIG_CUSTOM, "-vrp", (char *) NULL, (char *) NULL,
       "0.0 0.0 0.0",  0, 0, &tk_ViewportOptionVRP},

    {TK_CONFIG_CUSTOM, "-vup", (char *) NULL, (char *) NULL,
       "0.0 1.0 0.0", 0, 0, &tk_ViewportOptionVUP},

    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
       "1", Tk_Offset(PortItem, width), 0, NULL},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};



/*
 *--------------------------------------------------------------
 *
 * SmViewportChanged:
 *
 * determine if viewport specification has changed.
 *
 *--------------------------------------------------------------
 */

int
SmViewportChanged(port)
     PortItem *port;
{
    if (port->canvas) {
	return ((canvasConfigSpecs[1].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* bp */
		(canvasConfigSpecs[4].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* epsilon */
		(canvasConfigSpecs[16].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* maxu */
		(canvasConfigSpecs[17].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* maxv */
		(canvasConfigSpecs[18].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* minu */
		(canvasConfigSpecs[19].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* minv */
		(canvasConfigSpecs[23].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* projection */
		(canvasConfigSpecs[24].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* prp */
		(canvasConfigSpecs[29].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* vpn */
		(canvasConfigSpecs[30].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* vrp */
		(canvasConfigSpecs[31].specFlags & TK_CONFIG_OPTION_SPECIFIED)		/* vup */
		) ? 1 : 0;
    } else {
	return ((widgetConfigSpecs[2].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* bp */
		(widgetConfigSpecs[6].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* epsilon */
		(widgetConfigSpecs[18].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* maxu */
		(widgetConfigSpecs[19].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* maxv */
		(widgetConfigSpecs[20].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* minu */
		(widgetConfigSpecs[21].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* minv */
		(widgetConfigSpecs[25].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* projection */
		(widgetConfigSpecs[26].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* prp */
		(widgetConfigSpecs[30].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* vpn */
		(widgetConfigSpecs[31].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||	/* vrp */
		(widgetConfigSpecs[32].specFlags & TK_CONFIG_OPTION_SPECIFIED)		/* vup */
		) ? 1 : 0;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmSetClientData:
 *
 * set clientData to point to the appropriate cell header
 *
 *--------------------------------------------------------------
 */

void
SmSetClientData(clientData)
     ClientData clientData;
{
    tk_ViewportOptionCell.clientData = clientData;
}

/*
 *--------------------------------------------------------------
 *
 * SmVectorParseProc:
 *
 * process options for items that take vectors.
 *
 *--------------------------------------------------------------
 */

int
SmVectorParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;              /* An index that indicates which option *
					 * is being parsed */
    Tcl_Interp *interp;                 /* Used for reporting errors. */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *value;                        /* Value of option (3 doubles). */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Offset into item (ignored). */
{
    register PortItem *port = (PortItem *) widgRec;
    register Model *model = (Model *) widgRec;
    int argc;
    char **argv;
    double x, y, z;

    /* split the option value into components */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
        return TCL_ERROR;
    }

    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: 3 components required", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }

    if (Tcl_GetDouble(interp, argv[0], &x) != TCL_OK ||
	Tcl_GetDouble(interp, argv[1], &y) != TCL_OK ||
	Tcl_GetDouble(interp, argv[2], &z) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    (void) ckfree((void *) argv);
    switch((int) clientData) {

      case SM_PORT_VRP:
	port->vrp[0] = (DOUBLE) x;
	port->vrp[1] = (DOUBLE) y;
	port->vrp[2] = (DOUBLE) z;
	break;

      case SM_PORT_VPN:
	port->vpn[0] = (DOUBLE) x;
	port->vpn[1] = (DOUBLE) y;
	port->vpn[2] = (DOUBLE) z;
	break;
	
      case SM_PORT_VUP:
	port->vup[0] = (DOUBLE) x;
	port->vup[1] = (DOUBLE) y;
	port->vup[2] = (DOUBLE) z;
	break;
	
      case SM_PORT_PRP:
	port->prp[0] = (DOUBLE) x;
	port->prp[1] = (DOUBLE) y;
	port->prp[2] = (DOUBLE) z;
	break;

      case SM_MODEL_FWD:
	model->fwd[0] = (DOUBLE) x;
	model->fwd[1] = (DOUBLE) y;
	model->fwd[2] = (DOUBLE) z;
	model->updateFlag |= SM_MODEL_UPDATE_POSITION;
	break;

      case SM_MODEL_UP:
	model->up[0] = (DOUBLE) x;
	model->up[1] = (DOUBLE) y;
	model->up[2] = (DOUBLE) z;
	model->updateFlag |= SM_MODEL_UPDATE_POSITION;
	break;

      case SM_MODEL_POS:
	model->pos[0] = (DOUBLE) x;
	model->pos[1] = (DOUBLE) y;
	model->pos[2] = (DOUBLE) z;
	model->updateFlag |= SM_MODEL_UPDATE_POSITION;
	break;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SmVectorPrintProc:
 *
 * produce a printable string for those options for items
 * that take vector values.
 *
 *--------------------------------------------------------------
 */

char *
SmVectorPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;              /* An index that indicates which option *
					 * is being parsed */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Ignored. */
    Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
                                         * information about how to reclaim
                                         * storage for return string. */
{
    static char buf[128];
    DOUBLE *arrayPtr = NULL;
    Tcl_Interp *interp = NULL;

    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE];

#if defined(__alpha__)
    switch((long) clientData) {
#else
    switch((int) clientData) {
#endif

      case SM_PORT_VRP:
	arrayPtr = ((PortItem *) widgRec)->vrp;
	interp = ((PortItem *) widgRec)->interp;
	break;

      case SM_PORT_VPN:
	arrayPtr = ((PortItem *) widgRec)->vpn;
	interp = ((PortItem *) widgRec)->interp;
	break;
	
      case SM_PORT_VUP:
	arrayPtr = ((PortItem *) widgRec)->vup;
	interp = ((PortItem *) widgRec)->interp;
	break;
	
      case SM_PORT_PRP:
	arrayPtr = ((PortItem *) widgRec)->prp;
	interp = ((PortItem *) widgRec)->interp;
	break;

      case SM_MODEL_FWD:
	arrayPtr = ((Model *) widgRec)->fwd;
	interp = ((Model *) widgRec)->cell->interp;
	break;

      case SM_MODEL_UP:
	arrayPtr = ((Model *) widgRec)->up;
	interp = ((Model *) widgRec)->cell->interp;
	break;

      case SM_MODEL_POS:
	arrayPtr = ((Model *) widgRec)->pos;
	interp = ((Model *) widgRec)->cell->interp;
	break;

    }
    Tcl_PrintDouble(interp, (double) arrayPtr[0], c0);
    Tcl_PrintDouble(interp, (double) arrayPtr[1], c1);
    Tcl_PrintDouble(interp, (double) arrayPtr[2], c2);
    sprintf(buf, "%s %s %s", c0, c1, c2);
    *freeProcPtr = NULL;
    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * PortProjectionParseProc:
 *
 * process '-projection' option for viewport items.
 *
 *--------------------------------------------------------------
 */

static int
PortProjectionParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;              /* not used */
    Tcl_Interp *interp;                 /* Used for reporting errors. */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *value;                        /* Value of option (parallel or perspective). */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Offset into item (ignored). */
{
    register PortItem *port = (PortItem *) widgRec;
    char c;
    int len;

    c = *value;
    len = strlen(value);

    if ((c == 'p') && (len > 2) && (strncmp(value, "parallel", len) == 0)) {
	port->projection = SM_PORT_PARALLEL;
    } else if ((c == 'p') && (len > 2) && (strncmp(value, "perspective", len) == 0)) {
	port->projection = SM_PORT_PERSPECTIVE;
    } else {
	Tcl_AppendResult(interp, "illegal projection type: ", value,
			 ". should be one of: parallel, perspective", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortProjectionPrintProc:
 *
 * produce a printable string for the '-projection' option.
 *
 *--------------------------------------------------------------
 */

static char *
PortProjectionPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;              /* not used */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Ignored. */
    Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
                                         * information about how to reclaim
                                         * storage for return string. */
{
    register PortItem *port = (PortItem *) widgRec;

    *freeProcPtr = NULL;
    switch(port->projection) {
      case SM_PORT_PARALLEL:
	return "parallel";

      case SM_PORT_PERSPECTIVE:
	return "perspective";

      default:
	return "unknown";
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortCellParseProc:
 *
 * process '-cell' option for viewport items.
 *
 *--------------------------------------------------------------
 */

static int
PortCellParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;              /* not used */
    Tcl_Interp *interp;                 /* Used for reporting errors. */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *value;                        /* Value of option (parallel or perspective). */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Offset into item (ignored). */
{
    int i;
    LightSource *l;
    register PortItem *port = (PortItem *) widgRec;
    Cell *cell;

    if (!(*value)) {
	if (port->cell) {
	    SmRemovePortFromCell(port->cell, port);
	    port->cell = NULL;

	    /* turn off all tracking light sources */
	    for (i = 0, l = port->lights; i < 8; i++, l++) {
		if (l->model) l->on = 0;
	    }
	}
    }
    else if ((cell = SmFindCell(interp, value)) == NULL) {
	Tcl_AppendResult(interp, "cell \"", value, "\" does not exist.", (char *) NULL);
	return TCL_ERROR;
    }
    else if (cell != port->cell) {
	if (port->cell) {
	    SmRemovePortFromCell(port->cell, port);
	}
	port->cell = cell;
	SmInsertPortIntoCell(cell, port);

	/* turn off all tracking light sources */
	for (i = 0, l = port->lights; i < 8; i++, l++) {
	    if (l->model) l->on = 0;
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortCellPrintProc:
 *
 * produce a printable string for the '-cell' option.
 *
 *--------------------------------------------------------------
 */

static char *
PortCellPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;              /* not used */
     Tk_Window tkwin;                    /* Window containing canvas widget. */
     char *widgRec;                      /* Pointer to record for item. */
     int offset;                         /* Ignored. */
     Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
					  * information about how to reclaim
					  * storage for return string. */
{
    register PortItem *port = (PortItem *) widgRec;

    *freeProcPtr = NULL;

    if (port->cell) {
	return port->cell->label;
    } else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortModelVisibleMaskParseProc:
 *
 * process '-visibleMask' option for a viewport item.
 *
 *--------------------------------------------------------------
 */

static int
PortModelVisibleMaskParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;              /* not used */
    Tcl_Interp *interp;                 /* Used for reporting errors. */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *value;                        /* Value of option (list of strings) */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Offset into item (ignored). */
{
    int i, argc, new;
    int *ptr;
    char **argv;
    Tcl_HashTable *hash;

    int inverted = 0;
    register PortItem *port = (PortItem *) widgRec;

    hash = &port->visibleHash;
    ptr = &port->invertedVisibleMask;

    Tcl_DeleteHashTable(hash);
    Tcl_InitHashTable(hash, TCL_STRING_KEYS);
    if (*value) {
	if (*value == '^') {
	    inverted = 1;
	    if (Tcl_SplitList(interp, value+1, &argc, &argv) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	else if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	    return TCL_ERROR;
	}

	for (i = 0; i < (const) argc; i++) {
	    (void) Tcl_CreateHashEntry(hash, argv[i], &new);
	}
	(void) ckfree((void *) argv);
    }
    if (port->visibleMaskStr) (void) ckfree((void *) port->visibleMaskStr);
    if ((port->visibleMaskStr = (char *) ckalloc(strlen(value) + 1))) {
	strcpy(port->visibleMaskStr, value);
    }
    *ptr = inverted;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortModelVisibleMaskPrintProc:
 *
 * produce a printable string for model visible mask options.
 *
 *--------------------------------------------------------------
 */

static char *
PortModelVisibleMaskPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;              /* not used */
     Tk_Window tkwin;                    /* Window containing canvas widget. */
     char *widgRec;                      /* Pointer to record for item. */
     int offset;                         /* Ignored. */
     Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
					  * information about how to reclaim
					  * storage for return string. */
{
    register PortItem *port = (PortItem *) widgRec;

    *freeProcPtr = NULL;
    return port->visibleMaskStr;
}

/*
 *--------------------------------------------------------------
 *
 * PortModelPickParseProc:
 *
 * process '-pick' option for a viewport item.
 *
 * syntax: {x y w h}
 *
 *--------------------------------------------------------------
 */

static int
PortModelPickParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    PortItem *port = (PortItem *) widgRec;

    int argc;
    char **argv;

    port->picking = 1;
    if (!*value) {
	port->pick[0] = 0;
	return TCL_OK;
    } else {
	if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	    return TCL_ERROR;
	}

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be {x y w h}", (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, argv[0], &port->pick[1]) != TCL_OK ||
	    Tcl_GetInt(interp, argv[1], &port->pick[2]) != TCL_OK ||
	    Tcl_GetInt(interp, argv[2], &port->pick[3]) != TCL_OK ||
	    Tcl_GetInt(interp, argv[3], &port->pick[4]) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	(void) ckfree((void *) argv);
	port->pick[0] = 1;
	return TCL_OK;
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortModelPickPrintProc:
 *
 * produce the result of the pick operations.
 *
 *--------------------------------------------------------------
 */

static char *
PortModelPickPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;              /* not used */
     Tk_Window tkwin;                    /* Window containing canvas widget. */
     char *widgRec;                      /* Pointer to record for item. */
     int offset;                         /* Ignored. */
     Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
					  * information about how to reclaim
					  * storage for return string. */
{
    static char buf[512];

    int i, j, k;
    int x, y;
    char **labels, *result;
    short cx, cy;
    GLuint **list;
    GLint nhits, viewport[4];
    GLuint *hit, selectBuf[2048];
    Model *model;
    DOUBLE msh[16] = {
	1, 0, 0, 0,
	0, 1, 0, 0,
	0, 0, 1, 0,
	0, 0, 0, 1,
    };


    PortItem *port = (PortItem *) widgRec;
    int *pick = port->pick;

    *freeProcPtr = NULL;
    if (port->cell && port->pick[0]) {

	for (model = port->cell->models; model; model = model->next) {
	    model->pickname = 0;
	}

	SmSelectViewport(port);
	glGetIntegerv(GL_VIEWPORT, viewport);

	if (port->canvas) {
	    Tk_CanvasWindowCoords(port->canvas, (double) 0, (double) 0, &cx, &cy);

	    x = pick[1] - port->bbox[0] - cx;
	    y = viewport[3] - (pick[2] - port->bbox[1] - cy);
	} else {
	    x = pick[1];
	    y = viewport[3] - pick[2];
	}

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPickMatrix((GLdouble) x,
		      (GLdouble) y,
		      (GLdouble) pick[3],
		      (GLdouble) pick[4],
		      viewport);

	switch(port->projection) {
	  case SM_PORT_PARALLEL:
	    msh[8] = -port->prp[0] / port->prp[2];
	    msh[9] = -port->prp[1] / port->prp[2];
	    glOrtho(port->minu - port->prp[0],
		    port->maxu - port->prp[0],
		    port->minv - port->prp[1],
		    port->maxv - port->prp[1],
		    port->prp[2], port->prp[2] - port->bp);
#ifdef USE_DOUBLE
	    glMultMatrixd(msh);
#else
	    glMultMatrixf(msh);
#endif
	    break;
	    
	  case SM_PORT_PERSPECTIVE:
	    glFrustum(port->minu - port->prp[0],
		      port->maxu - port->prp[0],
		      port->minv - port->prp[1],
		  port->maxv - port->prp[1],
		      port->prp[2], port->prp[2] - port->bp);
	    break;
	}
	glSelectBuffer(2048, selectBuf);
	(void) glRenderMode(GL_SELECT);
	glInitNames();
	glPushName(-1);
	glMatrixMode(GL_MODELVIEW);
	
	SmDisplayModels(port, 1);
	nhits = glRenderMode(GL_RENDER);
	
	glMatrixMode(GL_PROJECTION);
	
	if (nhits == 0) {
	    sprintf(buf, "{%d %d %d %d}", pick[1], pick[2], pick[3], pick[4]);
	    return buf;
	}
	
	/* sort picked models */

	if ((list = (GLuint **) ckalloc(sizeof(GLuint *) * nhits)) == NULL) {
	    sprintf(buf, "{%d %d %d %d}", pick[1], pick[2], pick[3], pick[4]);
	    return buf;
	}

	for (hit = selectBuf, i = 0; i < (const) nhits; hit += 3 + *hit, i++) {

	    /* find slot where current hit record belongs */
	    for (j = 0; j < i; j++) {
		if (hit[1] < (list[j])[1]) break;
	    }

	    /* shift hit records down one slot */
	    for (k = i - 1; k >= j; k--) {
		list[k+1] = list[k];
	    }

	    /* insert current hit record into list */
	    list[j] = hit;
	}

	if ((labels = (char **) ckalloc(sizeof(char *) * (nhits + 1))) == NULL) {
	    (void) ckfree((void *) list);
	    sprintf(buf, "{%d %d %d %d}", pick[1], pick[2], pick[3], pick[4]);
	    return buf;
	}
    
	sprintf(buf, "%d %d %d %d", pick[1], pick[2], pick[3], pick[4]);
	labels[0] = buf;
	for (i = 0; i < (const) nhits; i++) {
	    for (model = port->cell->models; model; model = model->next) {
		if (model->visible && model->pickname == (const) (list[i])[3]) {
		    labels[i+1] = model->label;
		    break;
		}
	    }
	}
	*freeProcPtr = (Tcl_FreeProc *) free;
	result = Tcl_Merge(nhits + 1, labels);
	ckfree((void *) list);
	ckfree((void *) labels);
	return result;
    } else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortLightSourceParseProc:
 *
 * process the '-light<x>' option for a viewport item.
 *
 * syntax:
 *
 * {ambient diffuse specular {x y z w} {dx dy dz} exponent cutoff {k0 k1 k2} model}
 *
 * The first four parameters are required, others are optional.
 * If an optional parameter is not specified, default value will
 * be assigned instead:
 *
 * {dx dy dz}	= {0.0 0.0 -1.0}
 * exponent	= 0.0
 * cutoff	= 180.0
 * {k0 k1 k2}	= {1.0 0.0 0.0}
 *
 *--------------------------------------------------------------
 */

static int
PortLightSourceParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int index;
    int argc;
    char **argv;
    double r, g, b;
    PortItem *port = (PortItem *) widgRec;
    LightSource *light;
    float pos[4], dir[3], att[3], ndir[3];
    double exponent, cutoff;
    Model *model;

    index = (int) clientData;
    light = &port->lights[index];
    model = NULL;

    if (*value) {

	if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((argc < 4) || (argc > 9)) {
	    Tcl_AppendResult(interp,
			     "wrong # args: should be {ambient diffuse specular position [direction exponent cutoff attenuation model]}",
			     (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	/* get ambient, diffuse, and specular color coefficients */
	if (SmParseColorCoefficients(interp, argv[0], &r, &g, &b) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	light->ambientRGBA[0] = (float) r;
	light->ambientRGBA[1] = (float) g;
	light->ambientRGBA[2] = (float) b;
	light->ambientRGBA[3] = (float) 1.0;

	if (SmParseColorCoefficients(interp, argv[1], &r, &g, &b) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	light->diffuseRGBA[0] = (float) r;
	light->diffuseRGBA[1] = (float) g;
	light->diffuseRGBA[2] = (float) b;
	light->diffuseRGBA[3] = (float) 1.0;
	
	if (SmParseColorCoefficients(interp, argv[2], &r, &g, &b) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	light->specularRGBA[0] = (float) r;
	light->specularRGBA[1] = (float) g;
	light->specularRGBA[2] = (float) b;
	light->specularRGBA[3] = (float) 1.0;
	
	/* parse light position */
	if (ParseLightPosition(interp, argv[3], pos) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	
	/* parse light direction */
	if (argc > 4) {
	    if (ParseLightDirection(interp, argv[4], dir, ndir) != TCL_OK) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	} else {
	    dir[0] = dir[1] = ndir[0] = ndir[1] = 0.0;
	    dir[2] = ndir[2] = -1.0;
	}
	
	/* parse spotlight exponent */
	if (argc > 5) {
	    if (Tcl_GetDouble(interp, argv[5], &exponent) != TCL_OK) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	} else {
	    exponent = 0.0;
	}
	
	/* parse spotlight cutoff */
	if (argc > 6) {
	    if (Tcl_GetDouble(interp, argv[6], &cutoff) != TCL_OK) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	} else {
	    cutoff = 180.0;
	}
	
	/* parse light attenuation factors */
	if (argc > 7) {
	    if (ParseLightAttenuation(interp, argv[7], att) != TCL_OK) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	} else {
	    att[0] = 1.0;
	    att[1] = att[2] = 0.0;
	}

	/* parse track target */
	if (argc > 8) {
	    if ((*argv[8]) && !(model = SmFindModel(port->cell, argv[8]))) {
		Tcl_AppendResult(interp, "model \"", argv[8], "\" not found in cell \"", port->cell->label, "\".",
				 (char *) NULL);
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	}
	(void) ckfree((void *) argv);
	
	/* light specification legal */
	light->position[0] = pos[0];
	light->position[1] = pos[1];
	light->position[2] = pos[2];
	light->position[3] = pos[3];
	light->direction[0] = dir[0];
	light->direction[1] = dir[1];
	light->direction[2] = dir[2];
	light->ndirection[0] = ndir[0];
	light->ndirection[1] = ndir[1];
	light->ndirection[2] = ndir[2];
	light->exponent = (DOUBLE) exponent;
	light->cutoff = cutoff;
	light->attenuation[0] = att[0];
	light->attenuation[1] = att[1];
	light->attenuation[2] = att[2];
	light->model = model;
	light->on = 1;
    }
    else {
	light->on = 0;
    }
    return TCL_OK;
}
/*
 *--------------------------------------------------------------
 *
 * ParseLightDirection:
 *
 * process light direction value.
 *
 *--------------------------------------------------------------
 */

static int
ParseLightDirection(interp, value, dir, ndir)
     Tcl_Interp *interp;
     char *value;
     float dir[];
     float ndir[];
{
    int argc;
    char **argv;
    double d[3], e;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: light direction should be {x y z}.", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[0], &d[0]) != TCL_OK ||
	Tcl_GetDouble(interp, argv[1], &d[1]) != TCL_OK ||
	Tcl_GetDouble(interp, argv[2], &d[2]) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    dir[0] = (float) d[0];
    dir[1] = (float) d[1];
    dir[2] = (float) d[2];

    if ((e = d[0] * d[0] + d[1] * d[1] + d[2] * d[2]) > 0) {
	e  = 1 / sqrt(e);
	ndir[0] = (float) (d[0] * e);
	ndir[1] = (float) (d[1] * e);
	ndir[2] = (float) (d[2] * e);
    } else {
	ndir[0] = ndir[1] = ndir[2] = 0;
    }
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseLightPosition:
 *
 * process light position value.
 *
 * light position is given as {x y z w}. w must be >= 0. If
 * w == 0, light is directional, otherwise it is positional,
 * 
 *--------------------------------------------------------------
 */

static int
ParseLightPosition(interp, value, pos)
     Tcl_Interp *interp;
     char *value;
     float pos[];
{
    int argc;
    char **argv;
    double p0, p1, p2, p3;
    
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: light position must be {x y z w}", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[0], &p0) != TCL_OK ||
	Tcl_GetDouble(interp, argv[1], &p1) != TCL_OK ||
	Tcl_GetDouble(interp, argv[2], &p2) != TCL_OK ||
	Tcl_GetDouble(interp, argv[3], &p3) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if (p3 < (DOUBLE) 0.0) {
	Tcl_AppendResult(interp, "bad value: w must be greater than or equal to 0", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    pos[0] = (float) p0;
    pos[1] = (float) p1;
    pos[2] = (float) p2;
    pos[3] = (float) p3;
    (void) ckfree((void *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseLightAttenuation:
 *
 * process light source attenuation factors.
 *
 *--------------------------------------------------------------
 */

static int
ParseLightAttenuation(interp, value, att)
     Tcl_Interp *interp;
     char *value;
     float att[];
{
    int argc;
    char **argv;
    double a0, a1, a2;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: light source attenuation has 3 elements.", (char *) NULL);
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    if (Tcl_GetDouble(interp, argv[0], &a0) != TCL_OK ||
	Tcl_GetDouble(interp, argv[1], &a1) != TCL_OK ||
	Tcl_GetDouble(interp, argv[2], &a2) != TCL_OK) {
	(void) ckfree((void *) argv);
	return TCL_ERROR;
    }
    (void) ckfree((void *) argv);

    /* make sure factors have valid values */
    if ((a0 <= (double) 0.0) || (a1 < (double) 0.0) || (a2 < (double) 0.0)) {
	Tcl_AppendResult(interp, "bad value: light source attenuation factors must be positive.", (char *) NULL);
	return TCL_ERROR;
    }
    att[0] = (float) a0;
    att[1] = (float) a1;
    att[2] = (float) a2;

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortLightSourcePrintProc:
 *
 * produce a printable string for the '-light' option.
 *
 *--------------------------------------------------------------
 */

static char *
PortLightSourcePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    char r[TCL_DOUBLE_SPACE], g[TCL_DOUBLE_SPACE], b[TCL_DOUBLE_SPACE];
    char x[TCL_DOUBLE_SPACE], y[TCL_DOUBLE_SPACE], z[TCL_DOUBLE_SPACE], w[TCL_DOUBLE_SPACE];
    char dx[TCL_DOUBLE_SPACE], dy[TCL_DOUBLE_SPACE], dz[TCL_DOUBLE_SPACE];
    char exponent[TCL_DOUBLE_SPACE], cutoff[TCL_DOUBLE_SPACE];
    char k0[TCL_DOUBLE_SPACE], k1[TCL_DOUBLE_SPACE], k2[TCL_DOUBLE_SPACE];
    static char buf[256];

    PortItem *port = (PortItem *) widgRec;
    Tcl_Interp *interp = port->interp;
    LightSource *light = &port->lights[(int) clientData];

    *freeProcPtr = NULL;
    if (light->on) {
	Tcl_PrintDouble(interp, (double) light->position[0], x);
	Tcl_PrintDouble(interp, (double) light->position[1], y);
	Tcl_PrintDouble(interp, (double) light->position[2], z);
	Tcl_PrintDouble(interp, (double) light->position[3], w);

	Tcl_PrintDouble(interp, (double) light->direction[0], dx);
	Tcl_PrintDouble(interp, (double) light->direction[1], dy);
	Tcl_PrintDouble(interp, (double) light->direction[2], dz);

	Tcl_PrintDouble(interp, (double) light->attenuation[0], k0);
	Tcl_PrintDouble(interp, (double) light->attenuation[1], k1);
	Tcl_PrintDouble(interp, (double) light->attenuation[2], k2);

	Tcl_PrintDouble(interp, (double) light->cutoff, cutoff);
	Tcl_PrintDouble(interp, (double) light->exponent, exponent);

	Tcl_PrintDouble(interp, (double) light->ambientRGBA[0], r);
	Tcl_PrintDouble(interp, (double) light->ambientRGBA[1], g);
	Tcl_PrintDouble(interp, (double) light->ambientRGBA[2], b);
	sprintf(buf, "{%s %s %s} ", r, g, b);

	Tcl_PrintDouble(interp, (double) light->diffuseRGBA[0], r);
	Tcl_PrintDouble(interp, (double) light->diffuseRGBA[1], g);
	Tcl_PrintDouble(interp, (double) light->diffuseRGBA[2], b);
	sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

	Tcl_PrintDouble(interp, (double) light->specularRGBA[0], r);
	Tcl_PrintDouble(interp, (double) light->specularRGBA[1], g);
	Tcl_PrintDouble(interp, (double) light->specularRGBA[2], b);
	sprintf(buf + strlen(buf), "{%s %s %s} ", r, g, b);

	sprintf(buf + strlen(buf), "{%s %s %s %s} {%s %s %s} %s %s {%s %s %s} %s",
		x, y, z, w,
		dx, dy, dz,
		exponent, cutoff, k0, k1, k2,
		light->model ? light->model->label : "{}");
	return buf;
    }
    return "";
}

/*
 *--------------------------------------------------------------
 *
 * PortDepthCueParseProc:
 *
 * process the '-depthcue' option for a viewport item.
 *
 * syntax: {{r g b} | color} mode {density | start end}
 *
 *--------------------------------------------------------------
 */

static int
PortDepthCueParseProc(clientData, interp, tkwin, value, widgRec, offset)
     ClientData clientData;
     Tcl_Interp *interp;
     Tk_Window tkwin;
     char *value;
     char *widgRec;
     int offset;
{
    int argc;
    char **argv;
    double r, g, b;
    double start, end, density;
    GLenum mode;
    PortItem *port = (PortItem *) widgRec;

    if (!(*value)) {
	port->depthcue.on = 0;
    }
    else {
	if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((argc != 3) && (argc != 4)) {
	    Tcl_AppendResult(interp,
			     "wrong # args: should be {{r g b} mode density} or {{r g b} mode start end}",
			     (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	/* parse depth cue color */
	if (SmParseColorCoefficients(interp, argv[0], &r, &g, &b) != TCL_OK) {
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}

	/* parse depth cue parameter values */
	if ((!strcmp(argv[1], "exp")) || (!strcmp(argv[1], "exp2"))) {
	    if (argc != 3) {
		Tcl_AppendResult(interp,
				 "wrong # args: should be {{r g b} ", argv[1], " density}",
				 (char *) NULL);
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	    if ((argv[1])[3] == '2') {
		mode = GL_EXP2;
	    } else {
		mode = GL_EXP;
	    }
	    if (Tcl_GetDouble(interp, argv[2], &density) != TCL_OK) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	}
	else if (!strcmp(argv[1], "linear")) {
	    if (argc != 4) {
		Tcl_AppendResult(interp,
				 "wrong # args: should be {{r g b} linear start end}",
				 (char *) NULL);
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	    if ((Tcl_GetDouble(interp, argv[2], &start) != TCL_OK) ||
		(Tcl_GetDouble(interp, argv[3], &end) != TCL_OK)) {
		(void) ckfree((void *) argv);
		return TCL_ERROR;
	    }
	    mode = GL_LINEAR;
	}
	else {
	    Tcl_AppendResult(interp, "unknown depthcue mode \"", argv[1], "\": must be linear, exp, or exp2",
			     (char *) NULL);
	    (void) ckfree((void *) argv);
	    return TCL_ERROR;
	}
	(void) ckfree((void *) argv);

	/* sanity check on values */

	if ((mode == GL_EXP) || (mode == GL_EXP2)) {
	    if (density < (double) 0.0) {
	      Tcl_AppendResult(interp,
				 "depth cue density must be non-negative",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	} else if (mode == GL_LINEAR) {
	    if (start < (double) 0.0) {
		Tcl_AppendResult(interp,
				 "depth cue parameter \"start\" must be non-negative",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    if (end < (double) 0.0) {
		Tcl_AppendResult(interp,
				 "depth cue parameter \"end\" must be non-negative",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    if (end <= start) {
		Tcl_AppendResult(interp,
				 "depth cue parameter \"end\" must be greater than \"start\"",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	}

	port->depthcue.on = 1;
	port->depthcue.mode = mode;
	port->depthcue.RGBA[0] = r;
	port->depthcue.RGBA[1] = g;
	port->depthcue.RGBA[2] = b;
	port->depthcue.density = (DOUBLE) density;
	port->depthcue.start = (DOUBLE) start;
	port->depthcue.end = (DOUBLE) end;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortDepthCuePrintProc:
 *
 * produce a printable string for the '-depthcue' option.
 *
 *--------------------------------------------------------------
 */

static char *
PortDepthCuePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
     ClientData clientData;
     Tk_Window tkwin;
     char *widgRec;
     int offset;
     Tcl_FreeProc **freeProcPtr;
{
    PortItem *port = (PortItem *) widgRec;
    Tcl_Interp *interp = port->interp;
    DepthCue *cue = &port->depthcue;
    char r[TCL_DOUBLE_SPACE], g[TCL_DOUBLE_SPACE], b[TCL_DOUBLE_SPACE];
    char density[TCL_DOUBLE_SPACE];
    char start[TCL_DOUBLE_SPACE], end[TCL_DOUBLE_SPACE];
    static char buf[256];

    *freeProcPtr = NULL;

    if (cue->on) {

	Tcl_PrintDouble(interp, (double) cue->RGBA[0], r);
	Tcl_PrintDouble(interp, (double) cue->RGBA[1], g);
	Tcl_PrintDouble(interp, (double) cue->RGBA[2], b);

	if (cue->mode == GL_EXP) {
	    Tcl_PrintDouble(interp, (double) cue->density, density);
	    sprintf(buf, "{%s %s %s} exp %s", r, g, b, density);
	} else if (cue->mode == GL_EXP2) {
	    Tcl_PrintDouble(interp, (double) cue->density, density);
	    sprintf(buf, "{%s %s %s} exp2 %s", r, g, b, density);
	} else if (cue->mode == GL_LINEAR) {
	    Tcl_PrintDouble(interp, (double) cue->start, start);
	    Tcl_PrintDouble(interp, (double) cue->end, end );
	    sprintf(buf, "{%s %s %s} linear %s %s", r, g, b, start, end);
	}
	return buf;
    }
    else {
	return "";
    }
}

/*
 *--------------------------------------------------------------
 *
 * PortHintParseProc:
 *
 * process '-foghint' and '-perspectivehint' options for viewport items.
 *
 *--------------------------------------------------------------
 */

static int
PortHintParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;              /* not used */
    Tcl_Interp *interp;                 /* Used for reporting errors. */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *value;                        /* Value of option (parallel or perspective). */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Offset into item (ignored). */
{
    register PortItem *port = (PortItem *) widgRec;
    char c;
    int len;
    GLenum hint;

    c = *value;
    len = strlen(value);

    if ((c == 'n') && (len > 2) && (strncmp(value, "nicest", len) == 0)) {
	hint = GL_NICEST;
    } else if ((c == 'f') && (len > 2) && (strncmp(value, "fastest", len) == 0)) {
	hint = GL_FASTEST;
    } else if ((c == 'n') && (len > 2) && (strncmp(value, "none", len) == 0)) {
	hint = GL_DONT_CARE;
    } else {
      Tcl_AppendResult(interp, "bad %s hint value \"", value,
		       ((const) clientData == SM_HINT_FOG) ? "fog" : "perspective",
		       "\": should be one of: none, nicest, fastest", (char *) NULL);
	return TCL_ERROR;
    }
    switch((const) clientData) {

      case SM_HINT_FOG:
	port->fogHint = hint;
	break;

      case SM_HINT_PERSPECTIVE:
	port->perspHint = hint;
	break;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PortHintPrintProc:
 *
 * produce a printable string for the '-foghint' option.
 *
 *--------------------------------------------------------------
 */

static char *
PortHintPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;              /* not used */
    Tk_Window tkwin;                    /* Window containing canvas widget. */
    char *widgRec;                      /* Pointer to record for item. */
    int offset;                         /* Ignored. */
    Tcl_FreeProc **freeProcPtr;         /* Pointer to variable to fill in with
                                         * information about how to reclaim
                                         * storage for return string. */
{
    register PortItem *port = (PortItem *) widgRec;
    GLenum hint;

    *freeProcPtr = NULL;
    switch((const) clientData) {
      case SM_HINT_FOG:
	hint = port->fogHint;
	break;

      case SM_HINT_PERSPECTIVE:
	hint = port->perspHint;
	break;
    }

    switch(hint) {
      case GL_NICEST:
	return "nicest";

      case GL_FASTEST:
	return "fastest";

      case GL_DONT_CARE:
	return "none";

      default:
	return "unknown";
    }
}
