/* 
 * smwidget.c --
 *
 *	This file implements viewport, an OpenGL/MESA widget.
 *
 * Portion of this code based on tkSquare.c
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms.TclTk" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "copyright.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <tk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include "smInt.h"



#define	MODIFIER_NONE		0
#define	MODIFIER_SHIFT		1
#define	MODIFIER_CONTROL	2


/*
 * Prototypes for procedures defined externally
 */

XVisualInfo	*SmGetVisualInfo _ANSI_ARGS_ ((Display *dpy, Tk_Window tkwin, int doublebuffer));

int             SmConfigureViewport _ANSI_ARGS_ ((Tcl_Interp *interp,
                                                  Tk_Canvas canvas, Tk_Item *itemPtr, int argc, char **argv, int flags));

void            SmRemovePortFromCell _ANSI_ARGS_ ((Cell *cell, PortItem *port));

void		SmCreateDrawable _ANSI_ARGS_ ((PortItem *port));

int		SmComputeViewTransform _ANSI_ARGS_ ((PortItem *port));

void		SmDisplayViewport _ANSI_ARGS_ ((Tk_Canvas canvas, Tk_Item *itemPtr, Display *dpy, Drawable dst,
						int x, int y, int width, int height));

CellHeader      *SmGetCellHeader _ANSI_ARGS_ ((Tcl_Interp *interp));


/*
 * Prototypes for procedures defined in this file that are exported:
 */

int		SmViewportCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp, int argc, char **argv));

void		SmViewportUpdate _ANSI_ARGS_ ((ClientData clientData));


/*
 * Prototypes for procedures defined in this file that are static:
 */

static void	ViewportCmdDeletedProc _ANSI_ARGS_ ((ClientData clientData));

static void	ViewportDestroy _ANSI_ARGS_ ((char *blockPtr));

static void	ViewportEventProc _ANSI_ARGS_ ((ClientData clientData, XEvent *eventPtr));

static int	ViewportWidgetCmd _ANSI_ARGS_ ((ClientData clientData, Tcl_Interp *interp,
						int argc, char **argv));

static void	ViewportNavigate _ANSI_ARGS_ ((PortItem *port, int x, int y, int rx, int ry,
					       double cx, double cy, double *dimensions, int modifier));

static void	ViewportTranslateEye _ANSI_ARGS_ ((PortItem *port, double f, int axis));

static void	ViewportRotateEye _ANSI_ARGS_ ((PortItem *port, double f, int axis));


/*
 * Data structure declarations
 */

extern Tk_ConfigSpec widgetConfigSpecs[];



/*
 *--------------------------------------------------------------
 *
 * SmViewportCmd --
 *
 * create a new viewport.
 *
 *--------------------------------------------------------------
 */

int
SmViewportCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    int i, width, height;
    XGCValues gcValues;
    unsigned long mask;
    PortItem *port;
    Tk_Window tkwin;
    Tk_Window main = (Tk_Window) clientData;

    if (argc < 4) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName width height ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[2], &width) != TCL_OK ||
	Tcl_GetInt(interp, argv[3], &height) != TCL_OK) {
	return TCL_ERROR;
    }

    if (width < 1 || height < 1) {
	Tcl_AppendResult(interp, "bad widget geometry: ", argv[2], "x", argv[3],
			 (char *) NULL);
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, "Viewport");
    Tk_GeometryRequest(tkwin, width, height);

    /*
     * allocate and initialize the widget record.
     */

    port = (PortItem *) ckalloc(sizeof(PortItem));

    port->tkwin = tkwin;
    port->dpy = Tk_Display(tkwin);
    port->interp = interp;
    port->widgetCmd = Tcl_CreateCommand(interp,
					Tk_PathName(port->tkwin), ViewportWidgetCmd,
					(ClientData) port, ViewportCmdDeletedProc);

    mask = GCFunction | GCPlaneMask;
    gcValues.function = GXcopy;
    gcValues.plane_mask = AllPlanes;

    port->redraw = 1;
    port->update = 0;
    port->canvas = NULL;
    port->initialized = 0;
    port->pm = None;
    port->pmGC = Tk_GetGC(tkwin, mask, &gcValues);
    port->glxpm = None;
    port->glxpmGC = None;
    port->outlineColor = NULL;
    port->fillColor = NULL;
    port->outlineGC = None;
    port->invalid = 0;
    port->cell = NULL;
    port->next = NULL;
    port->pick[0] = 0;
    port->ambientLight = NULL;
    port->ambient[0] = port->ambient[1] = port->ambient[2] = 0;
    port->ambient[3] = 1.0;
    port->depthcue.on = 0;
    port->depthcue.RGBA[3] = 1.0;
    port->hsr = 1;
    port->doublebuffer = 1;
    port->async = 1;
    port->texture = 1;
    port->picking = 0;
    port->epsilon = 1e-10;
    port->srcx = port->srcy = 0;
    port->srcw = port->srch = 0;
    port->w = 0;
    port->h = 0;
    port->invertedVisibleMask = 0;
    port->visibleMaskStr = NULL;

    for (i = 0; i < 8; i++) {
	port->lights[i].on = 0;
    }
    Tcl_InitHashTable(&port->visibleHash, TCL_STRING_KEYS);

    if (SmConfigureViewport(interp, (Tk_Canvas) tkwin, (Tk_Item *) port, argc-4, argv+4, 0) != TCL_OK) {
	Tk_DestroyWindow(port->tkwin);
	return TCL_ERROR;
    }

    if (!(port->visinfo = SmGetVisualInfo(Tk_Display(tkwin), tkwin, port->doublebuffer))) {
	Tcl_AppendResult(interp, "parent visual does not meet required specifications", (char *) NULL);
	Tk_DestroyWindow(tkwin);
	return TCL_ERROR;
    }
    Tk_CreateEventHandler(port->tkwin, ExposureMask|StructureNotifyMask,
			  ViewportEventProc, (ClientData) port);

    interp->result = Tk_PathName(port->tkwin);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ViewportWidgetCmd --
 *
 * process the Tcl command that corresponds to a widget managed
 * by this module.
 *
 *--------------------------------------------------------------
 */

static int
ViewportWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Information about square widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    char c;
    int i, len, width, height;
    size_t length;
    int result = TCL_OK;
    PortItem *port = (PortItem *) clientData;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tk_Preserve((ClientData) port);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0) && (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " cget option\"",
			     (char *) NULL);
	    goto error;
	}
	result = Tk_ConfigureValue(interp, port->tkwin, widgetConfigSpecs, (char *) port, argv[2], 0);
    }
    else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0) && (length >= 2)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, port->tkwin, widgetConfigSpecs, (char *) port, (char *) NULL, 0);
	}
	else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, port->tkwin, widgetConfigSpecs, (char *) port, argv[2], 0);
	}
	else {

	    /* don't allow option '-doublebuffer' to be changed */
	    for (i = 2; i < argc; i++) {
		if ((len = strlen(argv[i])) < 2) continue;
		c = argv[i][1];
		if ((c == 'd') && (strncmp(argv[i], "-doublebuffer", len) == 0)) {
		    Tcl_AppendResult(interp, "can't modify ", argv[i], " option after widget is created",
				     (char *) NULL);
		    return TCL_ERROR;
		}
	    }
	    result = SmConfigureViewport(interp, (Tk_Canvas) port->tkwin, (Tk_Item *) port, argc-2, argv+2,
					 TK_CONFIG_ARGV_ONLY);
	    Tk_SetInternalBorder(port->tkwin, port->width);
	    if (port->async && !port->update) {
		port->update = 1;
		port->redraw = 1;
		Tk_DoWhenIdle(SmViewportUpdate, (ClientData) port);
	    }
	}
    }
    else if ((c == 'n') && (strncmp(argv[1], "navigate", length) == 0) && (length >= 2)) {

	int x, y, mod;
	int rx, ry;
	int count;
	char **elem;
	double cx, cy;
	double dimensions[3];

	if (argc != 10) {
	    Tcl_AppendResult(interp, "wrong # navigate arguments: must be x y rx ry cx cy dimensions modifier",
			     (char *) NULL);
	    result = TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[2], &x) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[3], &y) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &rx) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[5], &ry) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[6], &cx) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[7], &cy) != TCL_OK)) {
	    return TCL_ERROR;
	}
	if (Tcl_SplitList(interp, argv[8], &count, &elem) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (count != 3) {
	    Tcl_AppendResult(interp, "bad navigate argument: dimensions must have 3 elements",
			     (char *) NULL);
	    (void) ckfree((void *) elem);
	    return TCL_ERROR;
	}
	if ((Tcl_GetDouble(interp, elem[0], &dimensions[0]) != TCL_OK) ||
	    (Tcl_GetDouble(interp, elem[1], &dimensions[1]) != TCL_OK) ||
	    (Tcl_GetDouble(interp, elem[2], &dimensions[2]) != TCL_OK)) {
	    (void) ckfree((void *) elem);
	    return TCL_ERROR;
	}

	if (strcmp(argv[9], "None") == 0) {
	    mod = MODIFIER_NONE;
	}
	else if (strcmp(argv[9], "Control") == 0) {
	    mod = MODIFIER_CONTROL;
	}
	else if (strcmp(argv[9], "Shift") == 0) {
	    mod = MODIFIER_SHIFT;
	}
	else {
	    Tcl_AppendResult(interp, "bad modifier \"", argv[9], "\": should be one of None, Control, and Shift", (char *) NULL);
	    (void) ckfree((void *) elem);
	    return TCL_ERROR;
	}
	ViewportNavigate(port, x, y, rx, ry, cx, cy, dimensions, mod);
	(void) ckfree((void *) elem);
    }
    else if ((c == 's') && (strncmp(argv[1], "size", length) == 0) && (length >= 2)) {
	if (argc == 2) {
	    sprintf(interp->result, "%d %d", port->w, port->h);
	}
	else if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # size arguments: must be {width height}", (char *) NULL);
	    result = TCL_ERROR;
	}
	else if (Tcl_GetInt(interp, argv[2], &width) != TCL_OK ||
		   Tcl_GetInt(interp, argv[3], &height) != TCL_OK) {
	    result = TCL_ERROR;
	}
	else if (width < 1 || height < 1) {
	    Tcl_AppendResult(interp, "bad size value: width and height must be positive", (char *) NULL);
	    return TCL_ERROR;
	}
	else {
	    Tk_GeometryRequest(port->tkwin, width, height);
	    port->redraw = 1;
	    if (!port->update) {
		Tk_DoWhenIdle(SmViewportUpdate, (ClientData) port);
		port->update = 1;
	    }
	}
    }
    else if ((c == 'u') && (strncmp(argv[1], "update", length) == 0) && (length >= 2)) {

	/* 'update' command */

	if (argc != 2) {
	    Tcl_AppendResult(interp, "too many arguments", (char *) NULL);
	    result = TCL_ERROR;
	}
	if (port->update) {
	    port->update = 0;
	    Tk_CancelIdleCall(SmViewportUpdate, (ClientData) port);
	}
	if (port->redraw) {
	    SmViewportUpdate((ClientData) port);
	}
    }
    else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
			 "\":  must be cget, configure, or size",
			 (char *) NULL);
	goto error;
    }
    Tk_Release((ClientData) port);
    return result;

    error:
    Tk_Release((ClientData) port);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ViewportEventProc --
 *
 * process various events on viewports.
 *
 *--------------------------------------------------------------
 */

void
ViewportEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    PortItem *port = (PortItem *) clientData;
    Tk_Window tkwin = port->tkwin;

    if (eventPtr->type == Expose) {
	port->redraw = 1;
	if (!port->update) {
	    Tk_DoWhenIdle(SmViewportUpdate, (ClientData) port);
	    port->update = 1;
	}
    } else if (eventPtr->type == ConfigureNotify) {
	port->redraw = 1;
	if (!port->update) {
	    Tk_DoWhenIdle(SmViewportUpdate, (ClientData) port);
	    port->update = 1;
	}
	port->w = Tk_Width(tkwin);
	port->h = Tk_Height(tkwin);
	SmCreateDrawable(port);
    } else if (eventPtr->type == DestroyNotify) {
	if (port->tkwin != NULL) {
	    port->tkwin = NULL;
	    Tcl_DeleteCommand(port->interp,
			      Tcl_GetCommandName(port->interp,
						 port->widgetCmd));
	}
	if (port->update) {
	    Tk_CancelIdleCall(SmViewportUpdate, (ClientData) port);
	}
	Tk_EventuallyFree((ClientData) port, ViewportDestroy);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ViewportCmdDeletedProc --
 *
 * destroys a viewport widget.
 *
 *----------------------------------------------------------------------
 */

static void
ViewportCmdDeletedProc(clientData)
     ClientData clientData;	/* Pointer to widget record for widget. */
{
    PortItem *port = (PortItem *) clientData;
    Tk_Window tkwin = port->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	port->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

/*
 *--------------------------------------------------------------
 *
 * SmViewportUpdate:
 *
 * redraw the contents of a viewport widget.
 *
 *--------------------------------------------------------------
 */

void
SmViewportUpdate(clientData)
     ClientData clientData;	/* Information about window. */
{
    PortItem *port = (PortItem *) clientData;
    Tk_Window tkwin = port->tkwin;

    port->update = 0;
    port->redraw = 1;
    if (!Tk_IsMapped(tkwin)) {
	return;
    }

#ifdef DEBUG_VIEW
    printf("display viewport widget: size = %dx%d (%dx%d)\n",
	   Tk_Width(tkwin), Tk_Height(tkwin), port->w, port->h);
#endif

    if ((port->w == 0)|| (port->h == 0)) return;

    SmDisplayViewport(NULL, (Tk_Item *) port, port->dpy, Tk_WindowId(tkwin),
		      0, 0, port->w, port->h);
}

/*
 *----------------------------------------------------------------------
 *
 * ViewportDestroy --
 *
 * clean up the internal structure of a viewport at a safe time
 * (when no-one is using it anymore).
 *
 *----------------------------------------------------------------------
 */

static void
ViewportDestroy(blockPtr)
     char *blockPtr;	/* Info about square widget. */
{
    Tcl_HashEntry *t;
    CurrentTexture *curr;
    PortItem *port = (PortItem *) blockPtr;
    CellHeader *header = SmGetCellHeader(port->interp);

    Tcl_DeleteHashTable(&port->visibleHash);

    Tk_FreeOptions(widgetConfigSpecs, (char *) port, port->dpy, 0);

    if (port->ambientLight != NULL) {
	Tk_FreeColor(port->ambientLight);
    }
    if (port->outlineColor != NULL) {
	Tk_FreeColor(port->outlineColor);
    }
    if (port->fillColor != NULL) {
	Tk_FreeColor(port->fillColor);
    }
    if (port->outlineGC != None) {
	Tk_FreeGC(port->dpy, port->outlineGC);
    }
    if (port->cell != NULL) {
	SmRemovePortFromCell(port->cell, port);
    }
    if (port->glxpm != None) {
	glXDestroyGLXPixmap(port->dpy, port->glxpm);
    }
    if (port->glxpmGC != None) {
        t = Tcl_FindHashEntry(&header->glxGC, (char *) port->glxpmGC);
        curr = (CurrentTexture *) Tcl_GetHashValue(t);
        if (--curr->refCount == 0) {
            Tcl_DeleteHashEntry(t);
            (void) ckfree((void *) curr);
        }
	glXDestroyContext(port->dpy, port->glxpmGC);
    }
    if (port->pm != None) {
	XFreePixmap(port->dpy, port->pm);
    }
    if (port->pmGC != None) {
	Tk_FreeGC(port->dpy, port->pmGC);
    }
    if (port->visinfo != NULL) {
	(void) XFree((char *) port->visinfo);
    }
    ckfree((char *) port);
}

/*
 *--------------------------------------------------------------
 *
 * ViewportNavigate --
 *
 * manages navigation in a viewport
 *
 *--------------------------------------------------------------
 */

static void
ViewportNavigate(port, x, y, rx, ry, cx, cy, dimensions, modifier)
     PortItem *port;
     int x;
     int y;
     int rx;
     int ry;
     double cx;
     double cy;
     double dimensions[3];
     int modifier;
{
    int dir;
    double dx, dy;
    double fx, fy;
    double gradient, ratio;

    if (port->invalid) return;
    if ((cx == 0) && (cy == 0)) return;

    dx = x - rx - port->w / 2;
    dy = y - ry - port->h / 2;

    if ((dx == 0) && (dy == 0)) return;

    if (dy < port->h / -10) {
	dir = 0;
    }
    else if (dy < port->h / 10) {
	dir = 3;
    }
    else {
	dir = 6;
    }

    if (dx > port->w / 10) {
	dir += 2;
    }
    else if (dx > port->w / -10) {
	dir += 1;
    }

    if (dir == 4) {
	if (dx == 0) {
	    if (dy > 0) {
		dir = 1;
	    }
	    else {
		dir = 7;
	    }
	}
	else if (dy == 0) {
	    if (dx > 0) {
		dir = 5;
	    }
	    else {
		dir = 3;
	    }
	}
	else {
	    gradient = port->h / port->w;
	    ratio = dy / dx;
	    if ((ratio > gradient) || (ratio < -gradient)) {
		if (dy > 0) {
		    dir = 7;
		}
		else {
		    dir = 1;
		}
	    }
	    else {
		if (dx > 0) {
		    dir = 5;
		}
		else {
		    dir = 3;
		}
	    }
	}
    }

    fx = cx * dx / port->w;
    fy = cy * dy / port->h;

    switch (dir) {

      case 0:
      case 2:
      case 6:
      case 8: 
	switch (modifier) {

	  case MODIFIER_NONE:
	    ViewportRotateEye(port, -fx, 'y');
	    ViewportTranslateEye(port, dimensions[2] * fy, 'z');
	    break;

	  case MODIFIER_SHIFT:
	    ViewportTranslateEye(port, dimensions[0] * fx, 'x');
	    ViewportTranslateEye(port, dimensions[1] * -fy, 'y');
	    break;

	  case MODIFIER_CONTROL:
	    ViewportRotateEye(port, -fx, 'z');
	    ViewportRotateEye(port, -fy, 'x');
	    break;
	}
	break;

      case 1:
      case 7:
	switch (modifier) {

	  case MODIFIER_NONE:
	    ViewportTranslateEye(port, dimensions[2] * fy, 'z');
	    break;

	  case MODIFIER_SHIFT:
	    ViewportTranslateEye(port, dimensions[1] * -fy, 'y');
	    break;

	  case MODIFIER_CONTROL:
	    ViewportRotateEye(port, -fy, 'x');
	    break;
	}
	break;

      case 3:
      case 5:
	switch (modifier) {

	  case MODIFIER_NONE:
	    ViewportRotateEye(port, -fx, 'y');
	    break;

	  case MODIFIER_SHIFT:
	    ViewportTranslateEye(port, dimensions[0] * fx, 'x');
	    break;

	  case MODIFIER_CONTROL:
	    ViewportRotateEye(port, fx, 'z');
	    break;
	}
	break;
    }
    port->redraw = 1;
    port->invalid = SmComputeViewTransform(port);
    if (port->async && !port->update) {
	port->update = 1;
	Tk_DoWhenIdle(SmViewportUpdate, (ClientData) port);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ViewportTranslateEye --
 *
 * translate eye position
 *
 *--------------------------------------------------------------
 */

static void
ViewportTranslateEye(port, f, axis)
     PortItem *port;
     double f;
     int axis;
{
    switch ((char) axis) {

      case 'x':
	port->vrp[0] += f * port->mr[0];
	port->vrp[1] += f * port->mr[4];
	port->vrp[2] += f * port->mr[8];
	break;

      case 'y':
	port->vrp[0] += f * port->mr[1];
	port->vrp[1] += f * port->mr[5];
	port->vrp[2] += f * port->mr[9];
	break;

      case 'z':
	port->vrp[0] += f * port->mr[2];
	port->vrp[1] += f * port->mr[6];
	port->vrp[2] += f * port->mr[10];
	break;
    }
}

/*
 *--------------------------------------------------------------
 *
 * ViewportRotateEye --
 *
 * rotate vpn and vup
 *
 *--------------------------------------------------------------
 */

static void
ViewportRotateEye(port, f, axis)
     PortItem *port;
     double f;
     int axis;
{
    double cosine, sine;
    double ax, ay, az;
    double m[3][3];

    cosine = cos(f);
    sine = sin(f);

    switch ((char) axis) {

      case 'x':
	ax = port->mr[0];
	ay = port->mr[4];
	az = port->mr[8];
	break;

      case 'y':
	ax = port->mr[1];
	ay = port->mr[5];
	az = port->mr[9];
	break;

      case 'z':
	ax = port->mr[2];
	ay = port->mr[6];
	az = port->mr[10];
	break;
    }

    m[0][0] = ax * ax + cosine * (1 - ax * ax);
    m[0][1] = ax * ay * (1 - cosine) - az * sine;
    m[0][2] = az * az * (1 - cosine) + ay * sine;

    m[1][0] = ax * ay * (1 - cosine) + az * sine;
    m[1][1] = ay * ay + cosine * (1 - ay * ay);
    m[1][2] = ay * az * (1 - cosine) - ax * sine;

    m[2][0] = az * ax * (1 - cosine) - ay * sine;
    m[2][1] = ay * az * (1 - cosine) + ax * sine;
    m[2][2] = az * az + cosine * (1 - az * az);

    ax = m[0][0] * port->vpn[0] + m[0][1] * port->vpn[1] + m[0][2] * port->vpn[2];
    ay = m[1][0] * port->vpn[0] + m[1][1] * port->vpn[1] + m[1][2] * port->vpn[2];
    az = m[2][0] * port->vpn[0] + m[2][1] * port->vpn[1] + m[2][2] * port->vpn[2];

    port->vpn[0] = ax;
    port->vpn[1] = ay;
    port->vpn[2] = az;

    if (axis != 'y') {
	ax = m[0][0] * port->vup[0] + m[0][1] * port->vup[1] + m[0][2] * port->vup[2];
	ay = m[1][0] * port->vup[0] + m[1][1] * port->vup[1] + m[1][2] * port->vup[2];
	az = m[2][0] * port->vup[0] + m[2][1] * port->vup[1] + m[2][2] * port->vup[2];

	port->vup[0] = ax;
	port->vup[1] = ay;
	port->vup[2] = az;
    }
}
