/********************************************************************************
 *
 * sphere.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Sphere
 *
 * Read in a Sphere node.
 *
 * fields:	radius		1	SFFloat
 *
 *******************************************************************************/

Node *
Vrml_read_Sphere(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeSphere *sphere;

    sphere = &node->node.sphere;
    sphere->radius = 1;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Sphere", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Sphere", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad Sphere format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {

	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Sphere", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Sphere", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad Sphere format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "radius")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &sphere->radius, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (sphere->radius < 0) {
		Tcl_AppendResult(interp, argv, ": negative Sphere radius", (char *) NULL);
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Sphere field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }

  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Sphere
 *
 * Render a Sphere node.
 *
 * fields:	radius		1	SFFloat
 *
 *******************************************************************************/

#define	SPHERE_SLICES	20
#define	SPHERE_LOOPS	20

int
Vrml_render_Sphere(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, index, comp;

    double sbase, sstep;
    double tbase, tstep;
    double angle1, delta1;
    double angle2, delta2;

    Model *model;
    Vertex *vertices, *v;
    Surface *surfaces, *s;

    GLenum smode, tmode;

    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];
    char double0[TCL_DOUBLE_SPACE];
    char double1[TCL_DOUBLE_SPACE];
    char double2[TCL_DOUBLE_SPACE];
    char double3[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-average",
	"on",
	"-shading",
	"smooth",
	"-materials",
	"",
	"-normals",
	"",
	"-cull",
	"back",
	"-ccw",
	"on",
    };
    NodeSphere *sphere = &node->node.sphere;
    float radius = sphere->radius;
    float r;

    float s0, s1, s2;
    DOUBLE x, y, z;
    SFVec3f pos, fwd, up, scale;
    SurfaceProperties *prop = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];

    argv[3] = ambient;
    argv[5] = diffuse;
    argv[7] = specular;
    argv[9] = emissive;

    /* add ambient */
    if (prop->ambientColor) {
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[2], double2);

	sprintf(ambient, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (prop->diffuseColor) {
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[2], double2);

	sprintf(diffuse, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (prop->specularColor) {
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[2], double2);
	Tcl_PrintDouble(interp, (double) prop->shininess[0], double3);

	sprintf(specular, "%s %s %s %s", double0, double1, double2, double3);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (prop->emissiveColor) {
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[2], double2);

	sprintf(emissive, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* create polygon model */
    if (Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 22, argv) != TCL_OK) {
	return 2;
    }
    if ((model = SmFindModel(state->cell, node->label)) == NULL) {
	return 2;
    }

    if (prop->texture2 && prop->image) {
	comp = prop->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;

	smode = (prop->wrapS == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;
	tmode = (prop->wrapT == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;
    }

    /* add vertices */
    if ((vertices = (Vertex *) ckalloc(sizeof(Vertex) * (2 + SPHERE_SLICES * SPHERE_LOOPS))) == NULL) {
	Tcl_AppendResult(interp, "model \"", node->label,
			 "\" (Sphere): out of memory while allocating vertex array", (char *) NULL);
	return 2;
    }
    v = vertices;
    v->lv[0] = 0;
    v->lv[1] = (DOUBLE) (s1 * radius);
    v->lv[2] = 0;
    v++;
    v->lv[0] = 0;
    v->lv[1] = -vertices->lv[1];
    v->lv[2] = 0;
    v++;

    delta1 = 3.14159265358979323846 / (SPHERE_SLICES + 1);
    delta2 = 2 * 3.14159265358979323846 / SPHERE_LOOPS;

    for (i = 0, angle1 = delta1; i < SPHERE_SLICES; i++, angle1 += delta1) {

	y = (DOUBLE) (s1 * radius * cos(angle1));

	angle2 = 1.57079632679489661923;
	r = (DOUBLE) (radius * sin(angle1));
	x = (DOUBLE) (r * s0);
	z = (DOUBLE) (-r * s2);

	for (j = 0; j < SPHERE_LOOPS; j++, angle2 += delta2, v++) {
	    v->lv[0] = (DOUBLE) (x * cos(angle2));
	    v->lv[1] = y;
	    v->lv[2] = (DOUBLE) (z * sin(angle2));
	}
    }

    /* add surfaces */
    if ((surfaces = (Surface *) ckalloc(sizeof(Surface) * SPHERE_LOOPS * (SPHERE_SLICES + 1))) == NULL) {
	Tcl_AppendResult(interp, "model \"", node->label,
			 "\" (Sphere): out of memory while allocating surface array", (char *) NULL);
	(void) ckfree((void *) vertices);
	return 2;
    }

    /* initialize all surfaces */
    for (s = surfaces, i = 0; i < 2 * SPHERE_LOOPS; s++, i++) {
	s->index = NULL;
	s->texcoords = NULL;
	s->texture = NULL;

	if ((s->index = (int *) ckalloc(sizeof(int) * 3)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (Sphere): out of memory while allocating vertex index array", (char *) NULL);
	    goto err;
	}
	if (prop->texture2 && prop->image) {
	    if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * 3)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Sphere): out of memory while allocating texture coordinates array", (char *) NULL);
		goto err;
	    }
	    s->texture = prop->texture2;
	    s->component = comp;
	    s->func = GL_MODULATE;
	    s->smode = smode;
	    s->tmode = tmode;
	    s->magFilter = GL_NEAREST;
	    s->minFilter = GL_NEAREST;
	    s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
	    s->bdColor[3] = 1;
	    prop->texture2->refCount++;
	}
	s->vcnt = 3;
	s->materials = NULL;
	s->lvn = NULL;
    }

    for (; i < (SPHERE_SLICES + 1) * SPHERE_LOOPS; s++, i++) {
	s->index = NULL;
	s->texcoords = NULL;
	s->texture = NULL;

	if ((s->index = (int *) ckalloc(sizeof(int) * 4)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (Sphere): out of memory while allocating vertex index array", (char *) NULL);
	    goto err;
	}
	if (prop->texture2 && prop->image) {
	    if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * 4)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Sphere): out of memory while allocating texture coordinates array", (char *) NULL);
		goto err;
	    }
	    s->texture = prop->texture2;
	    s->component = comp;
	    s->func = GL_MODULATE;
	    s->smode = smode;
	    s->tmode = tmode;
	    s->magFilter = GL_NEAREST;
	    s->minFilter = GL_NEAREST;
	    s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
	    s->bdColor[3] = 1;
	    prop->texture2->refCount++;
	}
	s->vcnt = 4;
	s->materials = NULL;
	s->lvn = NULL;
    }

    /* slice #1: north polar cap */
    sbase = 0;
    sstep = 1.0 / SPHERE_LOOPS;

    tbase = 1;
    tstep = 1.0 / (SPHERE_SLICES + 1);

    for (s = surfaces, i = 2; i < (const) SPHERE_LOOPS + 1; s++, i++) {
	s->index[0] = 0;
	s->index[1] = i;
	s->index[2] = i + 1;

	if (s->texcoords) {
	    s->texcoords[0].v[0] = sbase + sstep / 2.0;
	    s->texcoords[0].v[1] = 1;

	    s->texcoords[1].v[0] = sbase;
	    s->texcoords[1].v[1] = tbase - tstep;

	    s->texcoords[2].v[0] = sbase + sstep;
	    s->texcoords[2].v[1] = tbase - tstep;

	    sbase += sstep;
	}
    }
    s->index[0] = 0;
    s->index[1] = SPHERE_LOOPS + 1;
    s->index[2] = 2;

    if (s->texcoords) {
	s->texcoords[0].v[0] = sbase + sstep / 2.0;
	s->texcoords[0].v[1] = 1;

	s->texcoords[1].v[0] = sbase;
	s->texcoords[1].v[1] = tbase - tstep;

	s->texcoords[2].v[0] = sbase + sstep;
	s->texcoords[2].v[1] = tbase - tstep;
    }
    s++;

    /* slice #2: south polar cap */
    sbase = 0;
    index = (SPHERE_SLICES - 1) * SPHERE_LOOPS + 2;
    for (i = index; i < (const) index + SPHERE_LOOPS - 1; i++, s++) {
	s->index[0] = 1;
	s->index[1] = i + 1;
	s->index[2] = i;

	if (s->texcoords) {
	    s->texcoords[0].v[0] = sbase + sstep / 2.0;
	    s->texcoords[0].v[1] = 0;

	    s->texcoords[1].v[0] = sbase + sstep;
	    s->texcoords[1].v[1] = tstep;

	    s->texcoords[2].v[0] = sbase;
	    s->texcoords[2].v[1] = tstep;

	    sbase += sstep;
	}
    }
    s->index[0] = 1;
    s->index[1] = index;
    s->index[2] = i;
    if (s->texcoords) {
	s->texcoords[0].v[0] = sbase + sstep / 2.0;
	s->texcoords[0].v[1] = 0;

	s->texcoords[1].v[0] = sbase + sstep;
	s->texcoords[1].v[1] = tstep;

	s->texcoords[2].v[0] = sbase;
	s->texcoords[2].v[1] = tstep;
    }
    s++;

    /* all slices in the middle */
    tbase = 1 - tstep;
    for (i = 0; i < SPHERE_SLICES - 1; i++) {
	index = 2 + i * SPHERE_LOOPS;

	sbase = 0;
	for (j = 0; j < SPHERE_LOOPS - 1; s++, j++) {
	    s->index[0] = index + j;
	    s->index[1] = index + SPHERE_LOOPS + j;
	    s->index[2] = s->index[1] + 1;
	    s->index[3] = s->index[0] + 1;

	    if (s->texcoords) {
		s->texcoords[0].v[0] = sbase;
		s->texcoords[0].v[1] = tbase;

		s->texcoords[1].v[0] = sbase;
		s->texcoords[1].v[1] = tbase - tstep;

		s->texcoords[2].v[0] = sbase + sstep;
		s->texcoords[2].v[1] = s->texcoords[1].v[1];

		s->texcoords[3].v[0] = s->texcoords[2].v[0];
		s->texcoords[3].v[1] = tbase;

		sbase += sstep;
	    }
	}
	s->index[0] = index + j;
	s->index[1] = index + SPHERE_LOOPS + j;
	s->index[2] = index + SPHERE_LOOPS;
	s->index[3] = index;

	if (s->texcoords) {
	    s->texcoords[0].v[0] = sbase;
	    s->texcoords[0].v[1] = tbase;

	    s->texcoords[1].v[0] = sbase;
	    s->texcoords[1].v[1] = tbase - tstep;

	    s->texcoords[2].v[0] = sbase + sstep;
	    s->texcoords[2].v[1] = s->texcoords[1].v[1];

	    s->texcoords[3].v[0] = s->texcoords[2].v[0];
	    s->texcoords[3].v[1] = tbase;
	}
	tbase -= tstep;
	s++;
    }

    model->nv = SPHERE_LOOPS * SPHERE_SLICES + 2;
    model->v = vertices;
    model->ns = SPHERE_LOOPS * (SPHERE_SLICES + 1);
    model->s = surfaces;
    model->useTexture = 1;
    model->materials = NULL;
    model->normals = NULL;
    model->textures = NULL;

    (void) SmComputeVertexNormals(model, 1);
    (void) SmComputeBoundingBox(model);

    return 0;

  err:
    for (j = 0, s = surfaces; j < i; j++, s++) {
	if (s->index) (void) ckfree((void *) s->index);
	if (s->texcoords) (void) ckfree((void *) s->texcoords);
	if (s->texture) s->texture->refCount--;
    }
    (void) ckfree((void *) vertices);
    (void) ckfree((void *) surfaces);
    return 2;
}
