/********************************************************************************
 *
 * switch.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Switch
 *
 * Read in a Switch  node.
 *
 * fields:	whichChild	-1		SFLong
 *
 *******************************************************************************/

Node *
Vrml_read_Switch(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    int dofield, nchild;
    char *field;
    Node *child, *last;
    NodeSwitch *sw;

    sw = &node->node.which;
    sw->whichChild = -1;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Switch", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Switch", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Switch format", (char *) NULL);
	return (Node *) -1;
    }

    /* parse all fields until first child node or close curly bracket */
    dofield = 1;
    while (dofield) {

	switch(Vrml_get_token(channel, &field)) {

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Switch", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Switch", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad Switch format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "whichChild")) {
	    if (Vrml_read_SFLong(interp, channel, argv, &sw->whichChild, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    dofield = 0;
	}
	if (dofield) {
	    Vrml_free_token(field);
	}
    }

    /* parse child nodes */
    last = NULL;
    nchild = 0;
    child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, field);
    Vrml_free_token(field);
    field = NULL;

    while ((child != NULL) && (child != (Node *) -1)) {
	nchild++;
	if (last) {
	    last->sibling = child;
	}
	else {
	    node->child = child;
	}
	last = child;
	child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, NULL);
    }
    if (child == NULL) {
	
	/* make sure field value is legal */
	if ((sw->whichChild != -1) && (sw->whichChild != -3) &&
	    ((sw->whichChild < 0) || (sw->whichChild >= nchild))) {
	    Tcl_AppendResult(interp, argv, ": bad Switch whichChild index value", (char *) NULL);
	    goto err;
	}
	else {
	    return node;
	}
    }
  
  err:
    if (field) Vrml_free_token(field);
    Vrml_free_Switch(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_Switch
 *
 * Free an Switch node.
 *
 * fields:	whichChild	-1	SFLong
 *
 *******************************************************************************/

void
Vrml_free_Switch(node)
     Node *node;
{
    Node *child, *sibling;

    for (child = node->child; child; child = sibling) {
	sibling = child->sibling;
	Vrml_free_node(child);
    }
}


/********************************************************************************
 *
 * Vrml_render_Switch
 *
 * Render a Switch node.
 *
 * fields:	whichChild	-1		SFLong
 *
 *******************************************************************************/

int
Vrml_render_Switch(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    SFVec3f pos, fwd, up, scale;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);

    if ((Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 0, NULL) != TCL_OK) ||
	(Vrml_push_model(interp, state, node) != TCL_OK)) {
	return 2;
    }

    return 0;
}


/********************************************************************************
 *
 * Vrml_finish_Switch
 *
 * Finish rendering a Switch node.
 *
 *******************************************************************************/

int
Vrml_finish_Switch(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    Vrml_pop_model(state);

    return 0;
}
