/********************************************************************************
 *
 * texture2.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Texture2
 *
 * Read in an Texture2 node.
 *
 * fields:	filename	""	SFString
 *		image		0 0 0	SFImage
 *		wrapS		REPEAT	SFEnum
 *		wrapT		REPEAT	SFEnum
 *
 * wrapS, wrapT:	REPEAT, CLAMP
 *
 *******************************************************************************/

Node *
Vrml_read_Texture2(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeTexture2 *tex;

    tex = &node->node.texture2;
    tex->filename = NULL;
    tex->image.width = tex->image.height = tex->image.comp = 0;
    tex->image.data = NULL;
    tex->wrapS = tex->wrapT = ENUM_REPEAT;
    tex->texture = NULL;
    tex->loaded = 0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Texture2", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Texture2", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad Texture2 format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Texture2", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Texture2", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad Texture2 format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "filename")) {
	    if (tex->filename) {
		Vrml_free_token(tex->filename);
		tex->filename = NULL;
	    }
	    if (Vrml_read_SFString(interp, channel, argv, &tex->filename, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "image")) {
	    if (tex->image.data) (void) ckfree((void *) tex->image.data);
	    if (Vrml_read_SFImage(interp, channel, argv, &tex->image) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "wrapS")) {
	    if (Vrml_read_SFEnum(interp, channel, argv, (unsigned long) (ENUM_REPEAT | ENUM_CLAMP), &tex->wrapS) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "wrapT")) {
	    if (Vrml_read_SFEnum(interp, channel, argv, (unsigned long) (ENUM_REPEAT | ENUM_CLAMP), &tex->wrapT) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Texture2 field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    Vrml_free_Texture2(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_Texture2
 *
 * Free a Texture2 node.
 *
 * fields:	filename	""	SFString
 *		image		0 0 0	SFImage
 *		wrapS		REPEAT	SFEnum
 *		wrapT		REPEAT	SFEnum
 *
 * wrapS, wrapT:	REPEAT, CLAMP
 *
 *******************************************************************************/

void
Vrml_free_Texture2(node)
     Node *node;
{
    NodeTexture2 *tex;

    tex = &node->node.texture2;
    if (tex->filename) Vrml_free_token(tex->filename);
    if (tex->image.data) (void) ckfree((void *) tex->image.data);
}


/********************************************************************************
 *
 * Vrml_render_Texture2
 *
 * Render a Texture2 node.
 *
 * fields:	filename	""	SFString
 *		image		0 0 0	SFImage
 *		wrapS		REPEAT	SFEnum
 *		wrapT		REPEAT	SFEnum
 *
 * wrapS, wrapT:	REPEAT, CLAMP
 *
 *******************************************************************************/

int
Vrml_render_Texture2(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    Node *origin;
    NodeTexture2 *tex;
    SFImage *image;
    Texture *texture;

    tex = &node->node.texture2;
    if (!(origin = node->origin)) {
	if ((tex->filename) && !tex->loaded) {
	    Tcl_AppendElement(interp, "Texture2");
	    Tcl_AppendElement(interp, tex->filename);
	    tex->loaded = 1;
	    return 1;
	}

	if (tex->filename && (tex->loaded == 2)) {

	    /* use tex->fileImage */
	    image = &tex->fileImage;
	}
	else {
	    /* use tex->image */
	    image = &tex->image;
	}
	texture = tex->texture = SmNewTextureMapFromSFImage(interp,
							    image->width,
							    image->height,
							    image->comp,
							    image->data);
    }
    else {
	while (origin->origin) {
	    origin = origin->origin;
	}
	tex = &origin->node.texture2;
	if (tex->filename && (tex->loaded == 2)) {
	    image = &tex->fileImage;
	}
	else {
	    image = &tex->image;
	}
	texture = tex->texture;
    }
    state->surface->wrapS = tex->wrapS;
    state->surface->wrapT = tex->wrapT;
    state->surface->image = image;
    state->surface->texture2 = texture;
    return 0;
}
