/********************************************************************************
 *
 * texture2transform.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Texture2Transform
 *
 * Read in a Texture2Transform node.
 *
 * fields:	translation	0 0	SFVec2f
 *		rotation	0	SFFloat
 *		scaleFactor	1 1 	SFVec2f
 *		center		0 0	SFVec2f
 *
 *******************************************************************************/

Node *
Vrml_read_Texture2Transform(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeTexture2Transform *xform;

    xform = &node->node.texture2Transform;
    xform->translation.v[0] = xform->translation.v[1] = 0;
    xform->rotation = 0;
    xform->scaleFactor.v[0] = xform->scaleFactor.v[1] = 1;
    xform->center.v[0] = xform->center.v[1] = 0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Texture2Transform", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Texture2Transform", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Texture2Transform format", (char *) NULL);
	return (Node *) -1;
    }

    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Texture2Transform", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Texture2Transform", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad Texture2Transform format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "translation")) {
	    if (Vrml_read_SFVec2f(interp, channel, argv, &xform->translation, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "rotation")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &xform->rotation, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "scaleFactor")) {
	    if (Vrml_read_SFVec2f(interp, channel, argv, &xform->scaleFactor, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "center")) {
	    if (Vrml_read_SFVec2f(interp, channel, argv, &xform->center, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Texture2Transform field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Texture2Transform
 *
 * Render a Texture2Transform node.
 *
 * fields:	translation	0 0	SFVec2f
 *		rotation	0	SFFloat
 *		scaleFactor	1 1 	SFVec2f
 *		center		0 0	SFVec2f
 *
 *******************************************************************************/

int
Vrml_render_Texture2Transform(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    NodeTexture2Transform *xform = &node->node.texture2Transform;
    SurfaceProperties *s = state->surface;

    memcpy((void *) &s->translation, (void *) &xform->translation, sizeof(SFVec2f));
    memcpy((void *) &s->scaleFactor, (void *) &xform->scaleFactor, sizeof(SFVec2f));
    memcpy((void *) &s->center, (void *) &xform->center, sizeof(SFVec2f));
    s->rotation = xform->rotation;

    return 0;
}
