/********************************************************************************
 *
 * transform.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Transform
 *
 * Read in a Transform node.
 *
 * fields:	translation		0 0 0		SFVec3f
 *		rotation		0 0 1 0		SFRotation
 *		scaleFactor		1 1 1		SFVec3f
 *		scaleOrientation	0 0 1 0		SFRotation
 *		center			0 0 0		SFVec3f
 *
 *******************************************************************************/

Node *
Vrml_read_Transform(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeTransform *xform;

    xform = &node->node.transform;
    xform->translation.v[0] = xform->translation.v[1] = xform->translation.v[2] = 0;
    xform->rotation[0] = xform->rotation[1] = xform->rotation[3] = 0;
    xform->rotation[2] = 1;
    xform->scaleFactor.v[0] = xform->scaleFactor.v[1] = xform->scaleFactor.v[2] = 1;
    xform->scaleOrientation[0] = xform->scaleOrientation[1] = xform->scaleOrientation[3] = 0;
    xform->scaleOrientation[2] = 1;
    xform->center.v[0] = xform->center.v[1] = xform->center.v[2] = 0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Transform", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Transform", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad Transform format", (char *) NULL);
	return (Node *) -1;
    }

    /* parse all fields until close curly bracket */
    while (1) {

	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Transform", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Transform", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_CLOSE_CURLY:
	    return node;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad Transform format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "translation")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &xform->translation, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "rotation")) {
	    if (Vrml_read_SFRotation(interp, channel, argv, xform->rotation) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "scaleFactor")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &xform->scaleFactor, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "scaleOrientation")) {
	    if (Vrml_read_SFRotation(interp, channel, argv, xform->scaleOrientation) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "center")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &xform->center, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Transform field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }

  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Transform
 *
 * Render a Transform node.
 *
 * fields:	translation		0 0 0		SFVec3f
 *		rotation		0 0 1 0		SFRotation
 *		scaleFactor		1 1 1		SFVec3f
 *		scaleOrientation	0 0 1 0		SFRotation
 *		center			0 0 0		SFVec3f
 *
 *******************************************************************************/

int
Vrml_render_Transform(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    NodeTransform *xform = &node->node.transform;
    SFVec3f *center = &xform->center;

    /* order of application:

       translation (translation)
       translation (center)
       rotation (rotation)
       rotation (scaleOrientation)
       scale
       rotation (-scaleOrientation)
       translation (-center)

       */

    Vrml_apply_translation(state, &xform->translation);
    Vrml_apply_translation(state, &xform->center);
    Vrml_apply_rotation(state, xform->rotation);
    Vrml_apply_rotation(state, xform->scaleOrientation);
    Vrml_apply_scale(state, &xform->scaleFactor);

    xform->scaleOrientation[3] = -xform->scaleOrientation[3];
    Vrml_apply_rotation(state, xform->scaleOrientation);
    xform->scaleOrientation[3] = -xform->scaleOrientation[3];

    center->v[0] = -center->v[0];
    center->v[1] = -center->v[1];
    center->v[2] = -center->v[2];
    Vrml_apply_translation(state, center);
    center->v[0] = -center->v[0];
    center->v[1] = -center->v[1];
    center->v[2] = -center->v[2];
    
    return 0;
}
