#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * local data structures
 *
 ********************************************************************************/

static MaskTable maskTable[] = {

  {"ALL",	MASK_ALL},
  {"TOP",	MASK_TOP},
  {"BOTTOM",	MASK_BOTTOM},
  {"SIDES",	MASK_SIDES},
  {"NONE",	MASK_NONE},
  {"BOLD",	MASK_BOLD},
  {"ITALIC",	MASK_ITALIC},
  {NULL,	0},
};


static EnumTable enumTable[] = {

  {"LEFT",		ENUM_LEFT},
  {"CENTER",		ENUM_CENTER},
  {"RIGHT",		ENUM_RIGHT},
  {"AUTO",		ENUM_AUTO},
  {"ON",		ENUM_ON},
  {"OFF",		ENUM_OFF},
  {"UNKNOWN_ORDERING",	ENUM_UNKNOWN_ORDERING},
  {"CLOCKWISE",		ENUM_CLOCKWISE},
  {"COUNTERCLOCKWISE",	ENUM_COUNTERCLOCKWISE},
  {"UNKNOWN_SHAPE_TYPE",ENUM_UNKNOWN_SHAPE_TYPE},
  {"SOLID",		ENUM_SOLID},
  {"UNKNOWN_FACE_TYPE",	ENUM_UNKNOWN_FACE_TYPE},
  {"CONVEX",		ENUM_CONVEX},
  {"REPEAT",		ENUM_REPEAT},
  {"CLAMP",		ENUM_CLAMP},
  {"NONE",		ENUM_NONE},
  {"POINT",		ENUM_POINT},
  {"SERIF",		ENUM_SERIF},
  {"SANS",		ENUM_SANS},
  {"TYPEWRITER",	ENUM_TYPEWRITER},
  {"DEFAULT",		ENUM_DEFAULT},
  {"OVERALL",		ENUM_OVERALL},
  {"PER_PART",		ENUM_PER_PART},
  {"PER_PART_INDEXED",	ENUM_PER_PART_INDEXED},
  {"PER_FACE",		ENUM_PER_FACE},
  {"PER_FACE_INDEXED",	ENUM_PER_FACE_INDEXED},
  {"PER_VERTEX",	ENUM_PER_VERTEX},
  {"PER_VERTEX_INDEXED",ENUM_PER_VERTEX_INDEXED},
  {NULL,		0},
};


/********************************************************************************
 *
 * Vrml_read_SFBitMask
 *
 * Read in a mask of bit flags.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFBitMask(interp, channel, argv, legal, mask)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     unsigned int legal;
     unsigned int *mask;
{
    int single;
    char *token;
    MaskTable *m;
    unsigned int value;
    
    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_OPEN_PAREN:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad bitmask format", (char *) NULL);
	return TCL_ERROR;
    }
    
    value = 0;
    while (1) {
	
	if (!single) {
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_WORD:
		break;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		return TCL_ERROR;
		
	      case TOKEN_END:
	      case TOKEN_EOF:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		return TCL_ERROR;
		
	      case TOKEN_CLOSE_PAREN:
		*mask = value;
		return TCL_OK;
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad bitmask format", (char *) NULL);
		return TCL_ERROR;
	    }
	}
	
	/* parse mask */
	for (m = maskTable; m->str; m++) {
	    if (!strcmp(m->str, token)) {
		if (m->value & legal) {
		    value |= m->value;
		    break;
		}
		else {
		    Tcl_AppendResult(interp, argv, ": illegal bit flag mask \"", token, "\"", (char *) NULL);
		    Vrml_free_token(token);
		    return TCL_ERROR;
		}
	    }
	}
	Vrml_free_token(token);
	if (!m->str) {
	    Tcl_AppendResult(interp, argv, ": unknown bit flag mask \"", token, "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	
	if (!single) {
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_END:
	      case TOKEN_EOF:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		return TCL_ERROR;
		
	      case TOKEN_CLOSE_PAREN:
		*mask = value;
		return TCL_OK;
		
	      case TOKEN_BAR:
		break;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);

	      default:
		Tcl_AppendResult(interp, argv, ": bad bitmask format", (char *) NULL);
		return TCL_ERROR;
	    }
	}
	else {
	    *mask = value;
	    return TCL_OK;
	}
    }
}


/********************************************************************************
 *
 * Vrml_read_SFBool
 *
 * Read in a single boolean value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFBool(interp, channel, argv, bool)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     int *bool;
{
    char *token;

    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_WORD:
	break;
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad boolean format", (char *) NULL);
	return TCL_ERROR;
    }
    
    /* parse boolean value */
    if (!strcmp(token, "TRUE") || !strcmp(token, "1")) {
	*bool = 1;
    }
    else if (!strcmp(token, "FALSE") || !strcmp(token, "0")) {
	*bool = 0;
    }
    else {
	Tcl_AppendResult(interp, argv, ": bad boolean value \"", token, "\"", (char *) NULL);
	Vrml_free_token(token);
	return TCL_ERROR;
    }
    Vrml_free_token(token);
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFColor
 *
 * Read in a single color value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFColor(interp, channel, argv, color, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFColor *color;
     char *save;
     int *result;
{
    int count, code;
    char *token;
    
    count = code = 0;
    if (save) {
	token = save;
    }
    
    while (count < 3) {

	switch(save ? TOKEN_WORD : (code = Vrml_get_token(channel, &token))) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
	    
	  case TOKEN_WORD:
	    if (sscanf(token, "%f", &color->rgb[count]) != 1) {
		Tcl_AppendResult(interp, argv, ": bad floating point number \"", token, "\"", (char *) NULL);
		if (!save) Vrml_free_token(token);
		if (result) *result = code;
		return TCL_ERROR;
	    }
	    if ((color->rgb[count] < (float) 0.0) || (color->rgb[count] > (float) 1.0)) {
		Tcl_AppendResult(interp, argv, ": RGB value out of range (0.0-1.0) \"", token, "\"", (char *) NULL);
		if (!save) Vrml_free_token(token);
		if (result) *result = code;
		return TCL_ERROR;
	    }
	    if (!save) Vrml_free_token(token);
	    save = NULL;
	    count++;
	    break;

	  default:
	    if (result) *result = code;
	    if (!result || (count != 0) || (code != TOKEN_CLOSE_BRACKET)) {
		Tcl_AppendResult(interp, argv, ": bad color format", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFEnum
 *
 * Read in a single enumerated type value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFEnum(interp, channel, argv, legal, value)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     unsigned long legal;
     unsigned long *value;
{
    char *token;
    EnumTable *e;

    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_WORD:
	break;
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad enumerated value", (char *) NULL);
	return TCL_ERROR;
    }
    
    /* parse enumerated type */
    for (e = enumTable; e->str; e++) {
	if (!strcmp(e->str, token)) {
	    if (e->value & legal) {
		*value = e->value;
		Vrml_free_token(token);
		return TCL_OK;
	    }
	    else {
		Tcl_AppendResult(interp, argv, ": illegal enumerated value \"", token, "\"", (char *) NULL);
		Vrml_free_token(token);
		return TCL_ERROR;
	    }
	}
    }
    Tcl_AppendResult(interp, argv, ": bad enumerated value \"", token, "\"", (char *) NULL);
    Vrml_free_token(token);
    return TCL_ERROR;
}


/********************************************************************************
 *
 * Vrml_read_SFFloat
 *
 * Read in a single floating point value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFFloat(interp, channel, argv, value, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     float *value;
     char *save;
     int *result;
{
    int code = 0;
    char *token;
    
    if (save) {
	token = save;
    }
    switch(save ? TOKEN_WORD : (code = Vrml_get_token(channel, &token))) {
	
      case TOKEN_WORD:
	break;
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	if (result) *result = code;
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	if (result) *result = code;
	return TCL_ERROR;
	
      default:
	if (result) *result = code;
	if (!result || (code != TOKEN_CLOSE_BRACKET)) {
	    Tcl_AppendResult(interp, argv, ": bad floating point format", (char *) NULL);
	}
	return TCL_ERROR;
    }
    
    /* parse floating point value */
    if (sscanf(token, "%f", value) != 1) {
	Tcl_AppendResult(interp, argv, ": bad floating point value \"", token, "\"", (char *) NULL);
	if (!save) Vrml_free_token(token);
	if (result) *result = code;
	return TCL_ERROR;
    }
    if (!save) Vrml_free_token(token);
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFImage
 *
 * Read in a single image value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFImage(interp, channel, argv, image)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFImage *image;
{
    int parms[3];
    char *token, buf[128];
    long i, count;
    unsigned long *p, *q;
    
    for (count = 0; count < 3; count++) {
	
	switch(Vrml_get_token(channel, &token)) {
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    return TCL_ERROR;
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return TCL_ERROR;
	    
	  case TOKEN_WORD:
	    if (sscanf(token, "%d", &parms[count]) != 1) {
		Tcl_AppendResult(interp, argv, ": not an integer \"", token, "\"", (char *) NULL);
		Vrml_free_token(token);
		return TCL_ERROR;
	    }
	    Vrml_free_token(token);
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad image format", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    
    /* make sure width, height, and number of components have valid values */
    
    if (parms[0] < 0) {
	sprintf(buf, "%d", parms[0]);
	Tcl_AppendResult(interp, argv, ": image width must be non-negative \"", buf, "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (parms[1] < 0) {
	sprintf(buf, "%d", parms[1]);
	Tcl_AppendResult(interp, argv, ": image height must be non-negative \"", buf, "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if ((parms[2] < 1) || (parms[2] > 4)) {
	sprintf(buf, "%d", parms[2]);
	Tcl_AppendResult(interp, argv, ": image components must be between 1 and 4 \"", buf, "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    count = parms[0] * parms[1];
    image->width = parms[0];
    image->height = parms[1];
    image->comp = parms[2];
    image->data = NULL;
    if (count > 0) {
	if ((p = q = (unsigned long *) ckalloc(sizeof(unsigned long) * count)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": failed to allocate memory for image data", (char *) NULL);
	    return TCL_ERROR;
	}

	/* read in image data */
	for (i = 0; i < (const) count; i++) {

	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_WORD:
		break;
		
	      case TOKEN_OUT_OF_MEMORY:
		(void) ckfree((void *) q);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		return TCL_ERROR;

	      case TOKEN_END:
	      case TOKEN_EOF:
		(void) ckfree((void *) q);
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		return TCL_ERROR;

	      default:
		(void) ckfree((void *) q);
		Tcl_AppendResult(interp, argv, ": bad image data format", (char *) NULL);
		return TCL_ERROR;
	    }
	    
	    /* parse floating point value */
	    if (sscanf(token, "%li", p++) != 1) {
		Tcl_AppendResult(interp, argv, ": bad image data value \"", token, "\"", (char *) NULL);
		Vrml_free_token(token);
		(void) ckfree((void *) q);
		return TCL_ERROR;
	    }
	    Vrml_free_token(token);
	}
	image->data = q;
    }
    else {
	Tcl_AppendResult(interp, argv, ": illegal image size", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFLong
 *
 * Read in a single long integer value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFLong(interp, channel, argv, value, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     long *value;
     char *save;
     int *result;
{
    int code;
    char*token;

    code = 0;
    if (save) {
	token = save;
    }
    switch(save ? TOKEN_WORD : (code = Vrml_get_token(channel, &token))) {

      case TOKEN_WORD:
	break;

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	if (result) *result = code;
	return TCL_ERROR;

      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	if (result) *result = code;
	return TCL_ERROR;

      default:
	if (result) *result = code;
	if (!result || (code != TOKEN_CLOSE_BRACKET)) {
	    Tcl_AppendResult(interp, argv, ": bad long integer format", (char *) NULL);
	}
	return TCL_ERROR;
    }
    
    /* parse long integer value */
    if (sscanf(token, "%li", value) != 1) {
	Tcl_AppendResult(interp, argv, ": bad long integer value \"", token, "\"", (char *) NULL);
	if (!save) Vrml_free_token(token);
	if (result) *result = code;
	return TCL_ERROR;
    }
    if (!save) Vrml_free_token(token);
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFMatrix
 *
 * Read in a single 4x4 matrix.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFMatrix(interp, channel, argv, matrix)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     float matrix[4][4];
{
    int row, col;
    char *token;

    for (col = 0; col < 4; col++) {
	for (row = 0; row < 4; row++) {
	    
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		return TCL_ERROR;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		return TCL_ERROR;

	      case TOKEN_WORD:
		if (sscanf(token, "%f", &matrix[row][col]) != 1) {
		    Tcl_AppendResult(interp, argv, ": bad floating point value \"", token, "\"", (char *) NULL);
		    Vrml_free_token(token);
		    return TCL_ERROR;
		}
		Vrml_free_token(token);
		break;
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad matrix format", (char *) NULL);
		return TCL_ERROR;
	    }
	}
    }

    /* make sure last row of matrix is [0 0 0 1] */
    if ((matrix[3][0] != 0) || (matrix[3][1] != 0) || (matrix[3][2] != 0) || (matrix[3][3] != 1)) {
	Tcl_AppendResult(interp, argv, ": last row of matrix must be [0 0 0 1]", (char *) NULL);
	return TCL_ERROR;
    }
    else {
	return TCL_OK;
    }
}


/********************************************************************************
 *
 * Vrml_read_SFRotation
 *
 * Read in a single rotation value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFRotation(interp, channel, argv, rotation)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     float rotation[4];
{
    int count;
    char *token;
    float d;

    for (count = 0; count < 4; count++) {
	
	switch(Vrml_get_token(channel, &token)) {
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    return TCL_ERROR;
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return TCL_ERROR;
	    
	  case TOKEN_WORD:
	    if (sscanf(token, "%f", &rotation[count]) != 1) {
		Tcl_AppendResult(interp, argv, ": bad floating point value \"", token, "\"", (char *) NULL);
		Vrml_free_token(token);
		return TCL_ERROR;
	    }
	    Vrml_free_token(token);
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad rotation format", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    d = sqrt(rotation[0] * rotation[0] + rotation[1] * rotation[1] + rotation[2] * rotation[2]);
    if (d > 0) {
	d = 1.0 / d;
	rotation[0] *= d;
	rotation[1] *= d;
	rotation[2] *= d;
	return TCL_OK;
    }
    else {
	Tcl_AppendResult(interp, argv, ": bad rotation axis", (char *) NULL);
	return TCL_ERROR;
    }
}


/********************************************************************************
 *
 * Vrml_read_SFString
 *
 * Read in a single string value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFString(interp, channel, argv, str, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     char **str;
     char *save;
     int *result;
{
    int code = 0;

    if (save) {
	*str = save;
    }
    else {
	switch(code = Vrml_get_token(channel, str)) {

	  case TOKEN_WORD:
	    break;

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
      
	  case TOKEN_END:
	  case TOKEN_EOF:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
      
	  default:
	    if (result) *result = code;
	    if (!result || (code != TOKEN_CLOSE_BRACKET)) {
		Tcl_AppendResult(interp, argv, ": bad string format", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFVec2f
 *
 * Read in a single two-dimensional vector value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFVec2f(interp, channel, argv, vec2f, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFVec2f *vec2f;
     char *save;
     int *result;
{
    int count, code;
    char *token;

    if (save) {
	token = save;
	code = 0;
    }

    for (count = 0; count < 2; count++) {

	switch(save ? TOKEN_WORD : (code = Vrml_get_token(channel, &token))) {

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;
	    
	  case TOKEN_WORD:
	    if (sscanf(token, "%f", &vec2f->v[count]) != 1) {
		Tcl_AppendResult(interp, argv, ": bad floating point value \"", token, "\"", (char *) NULL);
		if (!save) Vrml_free_token(token);
		if (result) *result = code;
		return TCL_ERROR;
	    }
	    if (!save) Vrml_free_token(token);
	    save = NULL;
	    break;

	  default:
	    if (result) *result = code;
	    if (!result || (count != 0) || (code != TOKEN_CLOSE_BRACKET)) {
		Tcl_AppendResult(interp, argv, ": bad two-dimensional vector format", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_SFVec3f
 *
 * Read in a single three-dimensional vector value.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_SFVec3f(interp, channel, argv, vec3f, save, result)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFVec3f *vec3f;
     char *save;
     int *result;
{
    int count, code;
    char *token;

    if (save) {
	token = save;
	code = TCL_OK;
    }

    for (count = 0; count < 3; count++) {

	switch(save ? TOKEN_WORD : (code = Vrml_get_token(channel, &token))) {

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    if (result) *result = code;
	    return TCL_ERROR;

	  case TOKEN_WORD:
	    if (sscanf(token, "%f", &vec3f->v[count]) != 1) {
		Tcl_AppendResult(interp, argv, ": bad floating point value \"", token, "\"", (char *) NULL);
		if (!save) Vrml_free_token(token);
		if (result) *result = code;
		return TCL_ERROR;
	    }
	    if (!save) Vrml_free_token(token);
	    save = NULL;
	    break;

	  default:
	    if (result) *result = code;
	    if (!result || (count != 0) || (code != TOKEN_CLOSE_BRACKET)) {
		Tcl_AppendResult(interp, argv, ": bad three-dimensional vector format", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_MFColor
 *
 * Read in one or more color values.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFColor(interp, channel, argv, color, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFColor **color;
     int *len;
{
    int count, size, single, code;
    char *token;

    *color = NULL;
    switch(Vrml_get_token(channel, &token)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad color format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;
    
    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else {
	    if ((*color = (SFColor *) ckalloc(sizeof(SFColor))) == NULL) {
		Vrml_free_token(token);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		return TCL_ERROR;
	    }
	    else {
		if (Vrml_read_SFColor(interp, channel, argv, *color, token, NULL) != TCL_OK) {
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    *len = 1;
		    Vrml_free_token(token);
		    return TCL_OK;
		}
	    }
	}
    }
    else {
	
	/* handle multiple-valued case */
	
	count = 0;
	size = 10;
	if ((*color = (SFColor *) ckalloc(sizeof(SFColor) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	
	/* read in SFColor value */
	if (Vrml_read_SFColor(interp, channel, argv, (*color) + count++, NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *color);
		*color = NULL;
		return TCL_OK;
	    }
	    goto err;
	}

	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;

	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad color format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad color format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		size += 10;
		if ((*color = (SFColor *) ckrealloc((char *) *color, sizeof(SFColor) * size)) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
	    }

	    /* read in SFColor value */
	    if (Vrml_read_SFColor(interp, channel, argv, (*color) + count++, token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	    Vrml_free_token(token);
	}
    }
  err:
    if (*color) {
	(void) ckfree((void *) *color);
	*color = NULL;
    }
    *len = 0;
    return TCL_ERROR;
}


/********************************************************************************
 *
 * Vrml_read_MFFloat
 *
 * Read in one or more floating point values.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFFloat(interp, channel, argv, num, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     float **num;
     int *len;
{
    int count, size, single, code;
    char *token;

    *num = NULL;
    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad floating point format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;
    
    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else {
	    if ((*num = (float *) ckalloc(sizeof(float))) == NULL) {
		Vrml_free_token(token);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
	    }
	    else {
		if (Vrml_read_SFFloat(interp, channel, argv, *num, token, NULL) != TCL_OK) {
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    *len = 1;
		    Vrml_free_token(token);
		    return TCL_OK;
		}
	    }
	}
    }
    else {
	
	/* handle multiple-valued case */
	
	count = 0;
	size = 10;

	if ((*num = (float *) ckalloc(sizeof(float) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	
	/* read in SFFloat value */
	if (Vrml_read_SFFloat(interp, channel, argv, (*num) + count++, NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *num);
		*num = NULL;
		return TCL_OK;
	    }
	    goto err;
	}

	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
		
	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad floating point format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad floating point format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		size += 10;
		if ((*num = (float *) ckrealloc((char *) *num, sizeof(float) * size)) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
	    }

	    /* read in SFFloat value */
	    if (Vrml_read_SFFloat(interp, channel, argv, (*num) + count++, token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	    Vrml_free_token(token);
	}
    }
  err:
    if (*num) {
	(void) ckfree((void *) *num);
	*num = NULL;
    }
    *len = 0;
    return TCL_ERROR;
}


/********************************************************************************
 *
 * Vrml_read_MFLong
 *
 * Read in one or more long integers.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFLong(interp, channel, argv, num, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     long **num;
     int *len;
{
    int count, size, single, code;
    char *token;

    *num = NULL;
    switch(Vrml_get_token(channel, &token)) {
      
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;

      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad long integer format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;
    
    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else {
	    if ((*num = (long *) ckalloc(sizeof(long))) == NULL) {
		Vrml_free_token(token);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
	    }
	    else {
		if (Vrml_read_SFLong(interp, channel, argv, *num, token, NULL) != TCL_OK) {
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    *len = 1;
		    Vrml_free_token(token);
		    return TCL_OK;
		}
	    }
	}
    }
    else {
	
	/* handle multiple-valued case */
	
	count = 0;
	size = 10;

	if ((*num = (long *) ckalloc(sizeof(long) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	
	/* read in SFLong value */
	if (Vrml_read_SFLong(interp, channel, argv, (*num) + count++, NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *num);
		*num = NULL;
		return TCL_OK;
	    }
	    goto err;
	}

	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
		
	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad long integer format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad long integer format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		size += 10;
		if ((*num = (long *) ckrealloc((char *) *num, sizeof(long) * size)) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
	    }

	    /* read in SFLong value */
	    if (Vrml_read_SFLong(interp, channel, argv, (*num) + count++, token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	    Vrml_free_token(token);
	}
    }
  err:
    if (*num) {
	(void) ckfree((void *) *num);
	*num = NULL;
    }
    *len = 0;
    return TCL_ERROR;
}


/********************************************************************************
 *
 * Vrml_read_MFString
 *
 * Read in one or more strings.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFString(interp, channel, argv, str, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     char ***str;
     int *len;
{
    int i, count, size, single, code;
    char *token;

    *str = NULL;
    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad string format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;
    
    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else if ((*str = (char **) ckalloc(sizeof(char *))) == NULL) {
	    Vrml_free_token(token);
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	else {
	    **str = token;
	    *len = 1;
	    return TCL_OK;
	}
    }
    else {
	
    /* handle multiple-valued case */
	
	count = 0;
	size = 10;

	if ((*str = (char **) ckalloc(sizeof(char *) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	for (i = 0; i < size; i++) {
	    (*str)[i] = NULL;
	}
	
	/* read in SFString value */
	if (Vrml_read_SFString(interp, channel, argv, (char **) ((*str) + count), NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *str);
		*str = NULL;
		return TCL_OK;
	    }
	    goto err;
	}
	count = 1;

	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
		
	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad string format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad string format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		if ((*str = (char **) ckrealloc((char *) *str, sizeof(char *) * (size + 10))) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    size += 10;
		    for (i = count; i < size; i++) {
			(*str)[i] = NULL;
		    }
		}
	    }

	    /* read in SFString value */
	    if (Vrml_read_SFString(interp, channel, argv, (char **) ((*str) + count++), token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	}
    }
  err:
    if (*str) {
	for (i = 0; i < size; i++) {
	    if (*str[i]) Vrml_free_token((*str)[i]);
	}
	(void) ckfree((void *) *str);
	*str = NULL;
    }
    *len = 0;
    return TCL_ERROR;
}


/********************************************************************************
 *
 * Vrml_read_MFVec2f
 *
 * Read in one or more Vec2f values.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFVec2f(interp, channel, argv, vector, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFVec2f **vector;
     int *len;
{
    int count, size, single, code;
    char *token;

    *vector = NULL;
    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad two-dimensional vector format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;
    
    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else {
	    if ((*vector = (SFVec2f *) ckalloc(sizeof(SFVec2f))) == NULL) {
		Vrml_free_token(token);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
	    }
	    else {
		if (Vrml_read_SFVec2f(interp, channel, argv, *vector, token, NULL) == -1) {
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    *len = 1;
		    Vrml_free_token(token);
		    return TCL_OK;
		}
	    }
	}
    }
    else {
	
    /* handle multiple-valued case */
	
	count = 0;
	size = 10;

	if ((*vector = (SFVec2f *) ckalloc(sizeof(SFVec2f) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    return TCL_ERROR;
	}

	/* read in SFVec2f value */
	if (Vrml_read_SFVec2f(interp, channel, argv, (*vector) + count++, NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *vector);
		*vector = NULL;
		return TCL_OK;
	    }
	    goto err;
	}

	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;

	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad two-dimensional vector format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad two-dimensional vector format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		size += 10;
		if ((*vector = (SFVec2f *) ckrealloc((char *) *vector, sizeof(SFVec2f) * size)) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
	    }

	    /* read in SFVec2f value */
	    if (Vrml_read_SFVec2f(interp, channel, argv, (*vector) + count++, token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	    Vrml_free_token(token);
	}
    }
  err:
    if (*vector) {
	(void) ckfree((void *) *vector);
	*vector = NULL;
    }
    *len = 0;
    return TCL_ERROR;

}


/********************************************************************************
 *
 * Vrml_read_MFVec3f
 *
 * Read in one or more Vec3f values.
 *
 * Return:		TCL_OK if successful, or
 *			TCL_ERROR upon error.
 *
 *******************************************************************************/

int
Vrml_read_MFVec3f(interp, channel, argv, vector, len)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char* argv;
     SFVec3f **vector;
     int *len;
{
    int count, size, single, code;
    char *token;

    *vector = NULL;
    switch(Vrml_get_token(channel, &token)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_END:
      case TOKEN_EOF:
	Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
	return TCL_ERROR;
	
      case TOKEN_WORD:
	single = 1;
	break;
	
      case TOKEN_OPEN_BRACKET:
	single = 0;
	break;
	
      default:
	Tcl_AppendResult(interp, argv, ": bad three-dimensional vector format", (char *) NULL);
	return TCL_ERROR;
    }
    *len = 0;

    /* handle single-valued case */
    if (single) {
	if (!strcmp(token, "~")) {
	    Vrml_free_token(token);
	    return TCL_OK;
	}
	else {
	    if ((*vector = (SFVec3f *) ckalloc(sizeof(SFVec3f))) == NULL) {
		Vrml_free_token(token);
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;
	    }
	    else {
		if (Vrml_read_SFVec3f(interp, channel, argv, *vector, token, NULL) == -1) {
		    Vrml_free_token(token);
		    goto err;
		}
		else {
		    *len = 1;
		    Vrml_free_token(token);
		    return TCL_OK;
		}
	    }
	}
    }
    else {
	
	/* handle multiple-valued case */
	
	count = 0;
	size = 10;

	if ((*vector = (SFVec3f *) ckalloc(sizeof(SFVec3f) * size)) == NULL) {
	    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
	    goto err;
	}
	
	/* read in SFVec3f value */
	if (Vrml_read_SFVec3f(interp, channel, argv, (*vector) + count++, NULL, &code) != TCL_OK) {
	    if (code == TOKEN_CLOSE_BRACKET) {
		(void) ckfree((void *) *vector);
		*vector = NULL;
		return TCL_OK;
	    }
	    goto err;
	}
	    
	while (1) {
	    
	    /* look for delimiter ',' or ']' */
	    switch(Vrml_get_token(channel, &token)) {
		
	      case TOKEN_COMMA:
		break;
		
	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;
		
	      case TOKEN_OUT_OF_MEMORY:
		Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		goto err;

	      case TOKEN_EOF:
	      case TOKEN_END:
		Tcl_AppendResult(interp, argv, ": unexpected end of input", (char *) NULL);
		goto err;
		
	      case TOKEN_WORD:
		Vrml_free_token(token);
		
	      default:
		Tcl_AppendResult(interp, argv, ": bad three-dimensional vector format", (char *) NULL);
		goto err;
	    }
	    
	    switch(Vrml_get_token(channel, &token)) {

	      case TOKEN_CLOSE_BRACKET:
		*len = count;
		return TCL_OK;

	      case TOKEN_WORD:
		break;

	      default:
		Tcl_AppendResult(interp, argv, ": bad three-dimensional vector format", (char *) NULL);
		goto err;
	    }

	    /* allocate more memory if necessary */
	    if (count == size) {
		size += 10;
		if ((*vector = (SFVec3f *) ckrealloc((char *) *vector, sizeof(SFVec3f) * size)) == NULL) {
		    Tcl_AppendResult(interp, argv, ": out of memory", (char *) NULL);
		    Vrml_free_token(token);
		    goto err;
		}
	    }

	    /* read in SFVec3f value */
	    if (Vrml_read_SFVec3f(interp, channel, argv, (*vector) + count++, token, NULL) != TCL_OK) {
		Vrml_free_token(token);
		goto err;
	    }
	    Vrml_free_token(token);
	}
    }
  err:
    if (*vector) {
	(void) ckfree((void *) *vector);
	*vector = NULL;
    }
    *len = 0;
    return TCL_ERROR;
}
