/********************************************************************************
 *
 * parse.h
 *
 ********************************************************************************/

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <smInt.h>

#ifndef USE_TCL75
	#define		Tcl_Channel		int
	#define		Tcl_Read(f, b, n)	read(f, b, n)
#endif

#define	BUFSIZE			1024

#define	TOKEN_WORD		0
#define	TOKEN_OPEN_PAREN	1
#define	TOKEN_CLOSE_PAREN	2
#define	TOKEN_OPEN_CURLY	3
#define	TOKEN_CLOSE_CURLY	4
#define	TOKEN_OPEN_BRACKET	5
#define	TOKEN_CLOSE_BRACKET	6
#define	TOKEN_COMMA		7
#define	TOKEN_BAR		8
#define	TOKEN_OUT_OF_MEMORY	9
#define	TOKEN_EOF		10
#define	TOKEN_END		11

#define	NODE_AsciiText		1
#define	NODE_Cone		2
#define	NODE_Cube		3
#define	NODE_Cylinder		4
#define	NODE_IndexedFaceSet	5
#define	NODE_IndexedLineSet	6
#define	NODE_PointSet		7
#define	NODE_Sphere		8

#define	NODE_Coordinate3	10
#define	NODE_FontStyle		11
#define	NODE_Info		12
#define	NODE_LOD		13
#define	NODE_Material		14
#define	NODE_MaterialBinding	15
#define	NODE_Normal		16
#define	NODE_NormalBinding	17
#define	NODE_Texture2		18
#define	NODE_Texture2Transform	19
#define	NODE_TextureCoordinate2	20
#define	NODE_ShapeHints		21

#define	NODE_MatrixTransform	22
#define	NODE_Rotation		23
#define	NODE_Scale		24
#define	NODE_Transform		25
#define	NODE_Translation	26

#define	NODE_OrthographicCamera	27
#define	NODE_PerspectiveCamera	28
#define	NODE_DirectionalLight	29
#define	NODE_PointLight		30
#define	NODE_SpotLight		31

#define	NODE_Group		40
#define	NODE_Separator		41
#define	NODE_Switch		42
#define	NODE_TransformSeparator	43
#define	NODE_WWWAnchor		44

#define	NODE_Script		45

#define	NODE_EndLOD			113
#define	NODE_EndGroup			140
#define	NODE_EndSeparator		141
#define	NODE_EndSwitch			142
#define	NODE_EndTransformSeparator	143
#define	NODE_EndWWWAnchor		144

#define	NODE_WWWInline		50
#define	NODE_USE		51


/********************************************************************************
 *
 * elemental data types
 *
 ********************************************************************************/

typedef struct SFColor {
    float	rgb[3];
} SFColor;

typedef struct SFVec3f {
    float	 v[3];
} SFVec3f;

typedef struct SFVec2f {
    float	v[2];
} SFVec2f;

typedef struct SFImage {
    long	width;
    long	height;
    long	comp;
    unsigned long	*data;
} SFImage;

typedef struct MaskTable {
    char		*str;
    unsigned int	value;
} MaskTable;

typedef struct EnumTable {
    char		*str;
    unsigned int	value;
} EnumTable;


typedef struct TransformProperties {

    float	global[4][4];
    float	local[4][4];
    struct TransformProperties *next;

} TransformProperties;


typedef struct SurfaceProperties {

    /* Coordinate3 */
    struct SFVec3f	*coord3;
    int			coord3len;

    /* Material */
    SFColor		*ambientColor;
    int			ambientlen;
    SFColor		*diffuseColor;
    int			diffuselen;
    SFColor		*specularColor;
    int			specularlen;
    SFColor		*emissiveColor;
    int			emissivelen;
    float		*shininess;
    int			shininesslen;
    float		*transparency;
    int			transparencylen;

    /* MaterialBinding */
    unsigned long	materialBinding;

    /* Normal */
    SFVec3f		*normal;
    int			normallen;

    /* NormalBinding */
    unsigned long	normalBinding;

    /* ShapeHints */

    unsigned long	vertexOrdering;
    unsigned long	shapeType;
    unsigned long	faceType;

    /* Texture2 */
    SFImage		*image;
    unsigned long	wrapS;
    unsigned long	wrapT;
    Texture		*texture2;

    /* Texture2Transform */
    SFVec2f		translation;
    float		rotation;
    SFVec2f		scaleFactor;
    SFVec2f		center;

    /* TextureCoordinate2 */
    SFVec2f		*texcoord2;
    int			texcoord2len;

    struct SurfaceProperties *next;

} SurfaceProperties;

typedef struct CurrentModel {
    struct Node		*node;
    struct CurrentModel	*next;

} CurrentModel;

typedef struct State {

    SurfaceProperties	*surface;
    TransformProperties	*transform;
    CurrentModel	*model;

    int			id;
    char		*prefix;
    Cell		*cell;
    struct Node		*curr;
    struct Node		*list;
    struct Node		*names;

    int			nameCount;

    struct Node		*infos;
    struct Node		*scripts;
    struct Node		*objects;
    struct Node		*lights;
    struct Node		*cameras;
    struct Node		*anchors;
    struct Node		*lods;

} State;

typedef struct DecodeTable {
    char		*type;
    int			code;
    struct Node *	(*new)(Tcl_Interp *, Tcl_Channel, char *, struct Node *, struct Node **, struct Node **, struct Node **);
    void		(*free)(struct Node *);
    int			(*render)(Tcl_Interp *interp, struct Node *, struct State *);
} DecodeTable;


/********************************************************************************
 *
 * nodes
 *
 ********************************************************************************/

/******************************
 * AsciiText
 ******************************/

#define	ENUM_LEFT	0x00000001
#define	ENUM_CENTER	0x00000002
#define	ENUM_RIGHT	0x00000004

typedef struct NodeAsciiText {
    char		**string;
    int			stringlen;
    float		spacing;
    float		*width;
    int			widthlen;
    unsigned long	justification;
} NodeAsciiText;


/******************************
 * Cone
 ******************************/

#define	MASK_ALL	0x0007
#define	MASK_BOTTOM	0x0002
#define	MASK_SIDES	0x0004

typedef struct NodeCone {
    unsigned int	parts;
    float		bottomRadius;
    float		height;
} NodeCone;


/******************************
 * Coordinate3
 ******************************/

typedef struct NodeCoordinate3 {
    SFVec3f	*point;
    int		pointlen;
} NodeCoordinate3;


/******************************
 * Cube
 ******************************/

typedef struct NodeCube {
    float	width;
    float	height;
    float	depth;
} NodeCube;


/******************************
 * Cylinder
 ******************************/

#define	MASK_TOP	0x0001

typedef struct NodeCylinder {
    unsigned int	parts;
    float		radius;
    float		height;
} NodeCylinder;


/******************************
 * DiectionalLight
 ******************************/

typedef struct NodeDirectionalLight {
    int		on;
    float	intensity;
    SFColor	color;
    SFVec3f	direction;
    SFVec3f	wdirection;
    struct Node	*model;
} NodeDirectionalLight;


/******************************
 * FontStyle
 ******************************/

#define	ENUM_SERIF	0x00020000
#define	ENUM_SANS	0x00040000
#define	ENUM_TYPEWRITER	0x00080000

#define	MASK_NONE	0x0000
#define	MASK_BOLD	0x0001
#define	MASK_ITALIC	0x0002

typedef struct NodeFontStyle {
    float		size;
    unsigned long	family;
    unsigned int	style;
} NodeFontStyle;


/******************************
 * Group
 ******************************/

typedef struct NodeGroup {
    int		dummy;
} NodeGroup;


/******************************
 * IndexedFaceSet
 ******************************/

typedef struct NodeIndexedFaceSet {
    long	*coordIndex;
    int		coordlen;
    long	*materialIndex;
    int		materiallen;
    long	*normalIndex;
    int		normallen;
    long	*textureCoordIndex;
    int		texturelen;
} NodeIndexedFaceSet;


/******************************
 * IndexedLineSet
 ******************************/

typedef struct NodeIndexedLineSet {
    long	*coordIndex;
    int		coordlen;
    long	*materialIndex;
    int		materiallen;
    long	*normalIndex;
    int		normallen;
    long	*textureCoordIndex;
    int		texturelen;
} NodeIndexedLineSet;


/******************************
 * Info
 ******************************/

typedef struct NodeInfo {
    char	**string;
    int		stringlen;
} NodeInfo;


/******************************
 * LOD
 ******************************/

typedef struct NodeLOD {
    float	*range;
    int		rangelen;
    SFVec3f	center;
    SFVec3f	wcenter;
} NodeLOD;


/******************************
 * Material
 ******************************/

typedef struct NodeMaterial {
    SFColor	*ambientColor;
    int		ambientlen;
    SFColor	*diffuseColor;
    int		diffuselen;
    SFColor	*specularColor;
    int		specularlen;
    SFColor	*emissiveColor;
    int		emissivelen;
    float	*shininess;
    int		shininesslen;
    float	*transparency;
    int		transparencylen;
} NodeMaterial;


/******************************
 * MaterialBinding
 ******************************/

#define	ENUM_DEFAULT		0x00020000
#define	ENUM_OVERALL		0x00040000
#define	ENUM_PER_PART		0x00080000
#define	ENUM_PER_PART_INDEXED	0x00100000
#define	ENUM_PER_FACE		0x00200000
#define	ENUM_PER_FACE_INDEXED	0x00400000
#define	ENUM_PER_VERTEX		0x00800000
#define	ENUM_PER_VERTEX_INDEXED	0x01000000

typedef struct NodeMaterialBinding {
    unsigned long	value;
} NodeMaterialBinding;


/******************************
 * MatrixTransform
 ******************************/

typedef struct NodeMatrixTransform {
    float matrix[4][4];
} NodeMatrixTransform;


/******************************
 * Normal
 ******************************/

typedef struct NodeNormal {
    SFVec3f	*vector;
    int		vectorlen;
} NodeNormal;


/******************************
 * NormalBinding
 ******************************/

typedef struct NodeNormalBinding {
    unsigned long	value;
} NodeNormalBinding;


/******************************
 * OrthographicCamera
 ******************************/

typedef struct NodeOrthographicCamera {
    SFVec3f	position;
    float	orientation[4];
    float	nearDistance;
    float	farDistance;
    float	focalDistance;
    float	focalLength;
    float	height;
    float	aspectRatio;

    SFVec3f	vrp;
    SFVec3f	vpn;
    SFVec3f	vup;

} NodeOrthographicCamera;


/******************************
 * PerspectiveCamera
 ******************************/

typedef struct NodePerspectiveCamera {
    SFVec3f	position;
    float	orientation[4];
    float	nearDistance;
    float	farDistance;
    float	focalDistance;
    float	focalLength;
    float	heightAngle;
    float	aspectRatio;

    SFVec3f	vrp;
    SFVec3f	vpn;
    SFVec3f	vup;

} NodePerspectiveCamera;


/******************************
 * PointLight
 ******************************/

typedef struct NodePointLight {
    int		on;
    float	intensity;
    SFColor	color;
    SFVec3f	location;
    SFVec3f	wlocation;
    struct Node	*model;
} NodePointLight;


/******************************
 * PointSet
 ******************************/

typedef struct NodePointSet {
    long	startIndex;
    long	numPoints;
} NodePointSet;


/******************************
 * Rotation
 ******************************/

typedef struct NodeRotation {
    float	rotation[4];
} NodeRotation;


/******************************
 * Scale
 ******************************/

typedef struct NodeScale {
    SFVec3f	scaleFactor;
} NodeScale;


/******************************
 * Separator
 ******************************/

#define	ENUM_AUTO	0x00000008
#define	ENUM_ON		0x00000010
#define	ENUM_OFF	0x00000020

typedef struct NodeSeparator {
    unsigned long	culling;
} NodeSeparator;


/******************************
 * ShapeHints
 ******************************/

#define ENUM_UNKNOWN_ORDERING	0x00000040
#define	ENUM_CLOCKWISE		0x00000080
#define	ENUM_COUNTERCLOCKWISE	0x00000100

#define	ENUM_UNKNOWN_SHAPE_TYPE	0x00000200
#define	ENUM_SOLID		0x00000400

#define	ENUM_UNKNOWN_FACE_TYPE	0x00000800
#define	ENUM_CONVEX		0x00001000

typedef struct NodeShapeHints {
    unsigned long	vertexOrdering;
    unsigned long	shapeType;
    unsigned long	faceType;
    float		creaseAngle;
} NodeShapeHints;


/******************************
 * Sphere
 ******************************/

typedef struct NodeSphere {
    float	radius;
} NodeSphere;


/******************************
 * SpotLight
 ******************************/

typedef struct NodeSpotLight {
    int		on;
    float	intensity;
    SFColor	color;
    SFVec3f	location;
    SFVec3f	wlocation;
    SFVec3f	direction;
    SFVec3f	wdirection;
    float	dropOffRate;
    float	cutOffAngle;
    struct Node	*model;
} NodeSpotLight;


/******************************
 * Switch
 ******************************/

typedef struct NodeSwitch {
    long	whichChild;
} NodeSwitch;


/******************************
 * Texture2
 ******************************/

#define	ENUM_REPEAT	0x00002000
#define	ENUM_CLAMP	0x00004000

typedef struct NodeTexture2 {
    int			loaded;
    char		*filename;
    SFImage		image;
    SFImage		fileImage;
    Texture		*texture;
    unsigned long	wrapS;
    unsigned long	wrapT;
} NodeTexture2;


/******************************
 * Texture2Transform
 ******************************/

typedef struct NodeTexture2Transform {
    SFVec2f	translation;
    float	rotation;
    SFVec2f	scaleFactor;
    SFVec2f	center;
} NodeTexture2Transform;


/******************************
 * TextureCoordinate2
 ******************************/

typedef struct NodeTextureCoordiante2 {
    int		pointlen;
    SFVec2f	*point;
} NodeTextureCoordinate2;


/******************************
 * Transform
 ******************************/

typedef struct NodeTransform {
    SFVec3f	translation;
    float	rotation[4];
    SFVec3f	scaleFactor;
    float	scaleOrientation[4];
    SFVec3f	center;
} NodeTransform;


/******************************
 * TransformSeparator
 ******************************/

typedef struct NodeTransformSeparator {
    int		dummy;
} NodeTransformSeparator;


/******************************
 * Translation
 ******************************/

typedef struct NodeTranslation {
    SFVec3f	translation;
} NodeTranslation;


/******************************
 * WWWAnchor
 ******************************/

#define	ENUM_NONE	0x00008000
#define	ENUM_POINT	0x00010000

typedef struct NodeWWWAnchor {
    char		*name;
    char		*description;
    unsigned long	map;
} NodeWWWAnchor;


/******************************
 * WWWInline
 ******************************/

typedef struct NodeWWWInline {
    int		expanded;
    char	*name;
    SFVec3f	bboxSize;
    SFVec3f	bboxCenter;
} NodeWWWInline;


/******************************
 * Script
 ******************************/

typedef struct NodeScript {
    char	*language;
    char	*script;
} NodeScript;


/******************************
 * Node
 ******************************/

typedef struct Node {
    int		code;
    char	*name;
    char	*label;

    struct Node	*namelist;
    struct Node *inlinelist;
    struct Node	*texturelist;

    struct Node	*parent;		/* used by LOD nodes only */
    struct Node	*child;
    struct Node	*sibling;
    struct Node	*next;
    struct Node	*toplevel;
    struct Node *origin;

    struct Node		*infos;
    struct Node		*scripts;
    struct Node		*objects;
    struct Node		*lights;
    struct Node		*cameras;
    struct Node		*anchors;
    struct Node		*lods;
    struct Node		*map;

    union {
	NodeAsciiText		asciiText;
	NodeCone		cone;
	NodeCoordinate3		coordinate3;
	NodeCube		cube;
	NodeDirectionalLight	directionalLight;
	NodeCylinder		cylinder;
	NodeFontStyle		fontStyle;
	NodeGroup		group;
	NodeIndexedFaceSet	indexedFaceSet;
	NodeIndexedLineSet	indexedLineSet;
	NodeInfo		info;
	NodeLOD			LOD;
	NodeMaterial		material;
	NodeMaterialBinding	materialBinding;
	NodeMatrixTransform	matrixTransform;
	NodeNormal		normal;
	NodeNormalBinding	normalBinding;
	NodeOrthographicCamera	orthographicCamera;
	NodePerspectiveCamera	perspectiveCamera;
	NodePointLight		pointLight;
	NodePointSet		pointSet;
	NodeRotation		rotation;
	NodeScale		scale;
	NodeSeparator		separator;
	NodeShapeHints		shapeHints;
	NodeSphere		sphere;
	NodeSpotLight		spotLight;
	NodeSwitch		which;
	NodeTexture2		texture2;
	NodeTexture2Transform	texture2Transform;
	NodeTextureCoordinate2	textureCoordinate2;
	NodeTransform		transform;
	NodeTransformSeparator	transformSeparator;
	NodeTranslation		translation;
	NodeWWWAnchor		wwwAnchor;
	NodeWWWInline		wwwInline;
	NodeScript		script;
    } node;
} Node;


/********************************************************************************
 *
 * functions for parsing fields
 *
 ********************************************************************************/

int	Vrml_get_token _ANSI_ARGS_ ((Tcl_Channel channel, char **token));
void	Vrml_free_token _ANSI_ARGS_ ((char *token));
Node *	Vrml_read_node _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node **names, Node **inlines, Node **textures,
				     int recursive, char *save));
void	Vrml_free_node _ANSI_ARGS_ ((Node *node));

int	Vrml_read_SFBitMask _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, unsigned int legal, unsigned int *mask));
int	Vrml_read_SFBool _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, int *bool));
int	Vrml_read_SFColor _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFColor *color, char *save, int *result));
int	Vrml_read_SFEnum _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, unsigned long legal, unsigned long *value));
int	Vrml_read_SFFloat _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, float *value, char *save, int *result));
int	Vrml_read_SFImage _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFImage *image));
int	Vrml_read_SFLong _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, long *value, char *save, int *result));
int	Vrml_read_SFMatrix _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, float matrix[4][4]));
int	Vrml_read_SFRotation _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, float rotation[]));
int	Vrml_read_SFString _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, char **str, char *save, int *result));
int	Vrml_read_SFVec2f _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFVec2f *vec2f, char *save, int *result));
int	Vrml_read_SFVec3f _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFVec3f *vec3f, char *save, int *result));

int	Vrml_read_MFColor _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFColor **color, int *len));
int	Vrml_read_MFFloat _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, float **num, int *len));
int	Vrml_read_MFLong _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, long **num, int *len));
int	Vrml_read_MFString _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, char ***str, int *len));
int	Vrml_read_MFVec2f _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFVec2f **vector, int *len));
int	Vrml_read_MFVec3f _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, SFVec3f **vector, int *len));


/********************************************************************************
 *
 * functions for parsing nodes
 *
 ********************************************************************************/

Node *	Vrml_read_AsciiText _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Cone _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Cube _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Cylinder _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_IndexedFaceSet _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_IndexedLineSet _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_PointSet _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Sphere _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));

Node *	Vrml_read_Coordinate3 _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_FontStyle _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Info _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_LOD _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Material _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_MaterialBinding _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Normal _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_NormalBinding _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Texture2 _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Texture2Transform _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_TextureCoordinate2 _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_ShapeHints _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));

Node *	Vrml_read_MatrixTransform _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Rotation _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Scale _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Transform _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Translation _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));

Node *	Vrml_read_OrthographicCamera _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_PerspectiveCamera _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_DirectionalLight _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_PointLight _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_SpotLight _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));

Node *	Vrml_read_Group _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Separator _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_Switch _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_TransformSeparator _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_WWWAnchor _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));
Node *	Vrml_read_WWWInline _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));

Node *	Vrml_read_Script _ANSI_ARGS_ ((Tcl_Interp *interp, Tcl_Channel channel, char *argv, Node *node, Node **names, Node **inlines, Node **textures));


/********************************************************************************
 *
 * functions for cleaning up nodes
 *
 ********************************************************************************/

void	Vrml_free_AsciiText _ANSI_ARGS_ ((Node *node));
void	Vrml_free_IndexedFaceSet _ANSI_ARGS_ ((Node *node));
void	Vrml_free_IndexedLineSet _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Coordinate3 _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Group _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Info _ANSI_ARGS_ ((Node *node));
void	Vrml_free_LOD _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Material _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Normal _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Separator _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Switch _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Texture2 _ANSI_ARGS_ ((Node *node));
void	Vrml_free_TextureCoordinate2 _ANSI_ARGS_ ((Node *node));
void	Vrml_free_TransformSeparator _ANSI_ARGS_ ((Node *node));
void	Vrml_free_WWWAnchor _ANSI_ARGS_ ((Node *node));
void	Vrml_free_WWWInline _ANSI_ARGS_ ((Node *node));
void	Vrml_free_Script _ANSI_ARGS_ ((Node *node));


/********************************************************************************
 *
 * functions for rendering nodes
 *
 ********************************************************************************/

int	Vrml_render_AsciiText _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Cone _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Cube _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Cylinder _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_IndexedFaceSet _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_IndexedLineSet _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_PointSet _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Sphere _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));

int	Vrml_render_Coordinate3 _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_FontStyle _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Info _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_LOD _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Material _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_MaterialBinding _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Normal _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_NormalBinding _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Texture2 _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Texture2Transform _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_TextureCoordinate2 _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_ShapeHints _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));

int	Vrml_render_MatrixTransform _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Rotation _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Scale _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Transform _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Translation _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_OrthographicCamera _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_PerspectiveCamera _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_DirectionalLight _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_PointLight _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_SpotLight _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));

int	Vrml_render_Group _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Separator _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_Switch _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_TransformSeparator _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_WWWAnchor _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));
int	Vrml_render_WWWInline _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));

int	Vrml_render_Script _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state));

/********************************************************************************
 *
 * functions for finishing rendering group nodes
 *
 ********************************************************************************/

int	Vrml_finish_Group _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 
int	Vrml_finish_Separator _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 
int	Vrml_finish_Switch _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 
int	Vrml_finish_TransformSeparator _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 
int	Vrml_finish_WWWAnchor _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 
int	Vrml_finish_LOD _ANSI_ARGS_ ((Tcl_Interp *interp, Node *node, State *state)); 

/********************************************************************************
 *
 * functions for rendering
 *
 ********************************************************************************/

int	Vrml_push_model _ANSI_ARGS_ ((Tcl_Interp *interp, State *state, Node *node));
int	Vrml_push_transform _ANSI_ARGS_ ((Tcl_Interp *interp, State *state));
int	Vrml_push_surface _ANSI_ARGS_ ((Tcl_Interp *interp, State *state));

void	Vrml_pop_model _ANSI_ARGS_ ((State *state));
void	Vrml_pop_transform _ANSI_ARGS_ ((State *state));
void	Vrml_pop_surface _ANSI_ARGS_ ((State *state));

int	Vrml_new_object _ANSI_ARGS_ ((Tcl_Interp *interp, State *state, Node *node, char *type,
				      SFVec3f *pos, SFVec3f *fwd, SFVec3f *up,
				      int argc, char **argv));
void	Vrml_collect_URLs _ANSI_ARGS_ ((Tcl_Interp *interp, Node *inlines, Node *textures));
void	Vrml_collect_results _ANSI_ARGS_ ((Tcl_Interp *interp, State *state));
void	Vrml_output_SFVec3f _ANSI_ARGS_ ((Tcl_Interp *interp, char *buffer, SFVec3f *vec));

void	Vrml_get_materials _ANSI_ARGS_ ((Tcl_Interp *interp, State *state, int *indices,
					 char *ambient, char *diffuse, char *specular, char *emissive, char *shininess));
void	Vrml_get_material_values _ANSI_ARGS_ ((Tcl_Interp *interp, State *state, int *indices, Material *mat));
void	Vrml_initialize_material_indices _ANSI_ARGS_ ((State *state, int indices[], int start));

void	Vrml_get_normal _ANSI_ARGS_ ((Tcl_Interp *interp, State *state, int *indices, SFVec3f *scale, char *vector));
void	Vrml_initialize_normal_indices _ANSI_ARGS_ ((State *state, int indices[], int start));

void	Vrml_rotate_vector _ANSI_ARGS_ ((float rotation[], float x, float y, float z, float *vout));
void	Vrml_coord3_transform _ANSI_ARGS_ ((State *state, SFVec3f *in, SFVec3f *out));
void	Vrml_vector_transform _ANSI_ARGS_ ((State *state, SFVec3f *in, SFVec3f *out));
void	Vrml_texcoord2_transform _ANSI_ARGS_ ((State *state, SFVec2f *in, SFVec2f *out));
void	Vrml_object_state _ANSI_ARGS_ ((State *state, SFVec3f *pos, SFVec3f *fwd, SFVec3f *up, SFVec3f *scale));
void	Vrml_apply_rotation _ANSI_ARGS_ ((State *state, float *rotation));
void	Vrml_apply_scale _ANSI_ARGS_ ((State *state, SFVec3f *scale));
void	Vrml_apply_translation _ANSI_ARGS_ ((State *state, SFVec3f *translation));

int	SmHandleNewModel _ANSI_ARGS_ ((Cell *cell, int argc, char **argv, Model **model));
int     SmComputeVertexNormals _ANSI_ARGS_ ((Model *model, int reset));
int     SmComputeBoundingBox _ANSI_ARGS_ ((Model *model));
int	SmNormalizeVector3D _ANSI_ARGS_ ((DOUBLE vin[], DOUBLE vout[], double epsilon));

Cell	*SmFindCell _ANSI_ARGS_ ((Tcl_Interp *interp, char *label));
Model	*SmFindModel _ANSI_ARGS_ ((Cell *cell, char *label));
Texture *SmNewTextureMapFromSFImage _ANSI_ARGS_ ((Tcl_Interp *interp, long width, long height,
						  long comp, unsigned long *data));
