#####
# Globales
#
set UNDO_BUFFER_SIZE 64



#####
# undo_init
#
# Initialise the undo and redo buffer. 
# The procedure is invoked from typewriter.create.
#
proc undo_init {win} {
	global undo_buffer

	set undo_buffer($win-pos) 0
	set undo_buffer($win-count) 0
	set undo_buffer($win-undo_count) 0
	set undo_buffer($win-prev_cmd) ""
	set undo_buffer($win-prev_index) 0.0
}



#####
# undo_term
#
# Unset the undo and redo buffer for the specified window. 
# The procedure is invoked from typewriter.destroy.
#
proc undo_term {win} {
	global undo_buffer

	foreach buf [array names undo_buffer $win*] {
		unset undo_buffer($buf)
	}
}



#####
# undobuf_push {win cmd sel index data} 
#
# This is the procedure which buffers the undo information.
# It is called up from the text widget wrapper while an insert or delete cmd.
# The sequence of delete and insert at the same index will be recognised as
# replace. The replace is done in one undo. The replace command is realised
# as a list of commands and datas in one undo buffer slot.
#
proc undo_push {win cmd sel index data} {
	global undo_buffer UNDO_BUFFER_SIZE

	# Apply replace?
	if {$index == $undo_buffer($win-prev_index) && \
	   $cmd == "insert" && $undo_buffer($win-prev_cmd) == "replace"} {
		# Set pointer to the undo_buffer
		set p [expr ($undo_buffer($win-pos) - 1) % $UNDO_BUFFER_SIZE]
	
		# Append the new undo information into the undo ring buffer
		lappend undo_buffer($win-cmd-$p) $cmd
		lappend undo_buffer($win-data-$p) $data

	} else {
		# Set pointer to the undo_buffer
		set p $undo_buffer($win-pos)
	
		# Insert the new undo information into the undo ring buffer
		set undo_buffer($win-cmd-$p)   $cmd
		set undo_buffer($win-index-$p) $index
		set undo_buffer($win-data-$p)  [list $data]

		# Set the new buffer position
		set undo_buffer($win-pos) [expr [incr p] % $UNDO_BUFFER_SIZE]

		# Increment undo buffer cmd count
		if {$undo_buffer($win-count) < $UNDO_BUFFER_SIZE} {
			incr undo_buffer($win-count)
		}
	}

	# Reset undo counter and menu entry
	if {$undo_buffer($win-undo_count)} {
		set undo_buffer($win-undo_count) 0

		# Update menu entry Redo
		.mbar.edit entryconfigure Redo -state disabled
	}

	# Save the cmd and index
	if {$sel == 1 && $cmd == "delete"} {
		set undo_buffer($win-prev_cmd) replace
	} else {
		set undo_buffer($win-prev_cmd) $cmd
	}
	set undo_buffer($win-prev_index) $index
}



#####
# undo {$win}
#
# This is the main undo procedure.
# If no win is given, the topWindow will be taken.
# The undo will be done direct into the renamed (wrapped) widget (.$win.text).
# - Refer to typewriter.create -
#
proc undo {{win ""}} {
	global undo_buffer UNDO_BUFFER_SIZE winInfo

	set win [window_check $win]

	# If nothing is to do, return
	if {!$undo_buffer($win-count)} {
		return
	}

	# Set pointer to the undo buffer.
	set p [expr ($undo_buffer($win-pos) - 1) % $UNDO_BUFFER_SIZE]

	# Get the undo information from the undo buffer
	set index $undo_buffer($win-index-$p)
	set n [llength $undo_buffer($win-cmd-$p)]

	while {$n} {
		set n [expr $n - 1]
		set cmd [lindex $undo_buffer($win-cmd-$p) $n]
		set data [lindex $undo_buffer($win-data-$p) $n]

		# Set the cursor to the indexed position
		uplevel [list .$win.text mark set insert $index]

		# Forward the undo command direct into the text widget (.$win.text)
		if {$cmd == "insert"} {
			uplevel [list .$win.text delete insert "$index + [string length $data] c"]
		} else {
			uplevel [list .$win.text insert insert $data]
		}
	}

	# Correct the undo buffer counters and pointer
	set undo_buffer($win-pos) $p
	set undo_buffer($win-count) [expr $undo_buffer($win-count) - 1]
	incr undo_buffer($win-undo_count)

	# Update menu entry Redo
	.mbar.edit entryconfigure Redo -state normal

	# Force view at insert position
	uplevel [list .$win.text see insert]

	# Set changed flag, and window title
	if {!$winInfo($win-dirty)} {
		set winInfo($win-dirty) 1
		wm title $win "* [wm title $win]"
		wm iconname $win [wm title $win]
	}
}



#####
# redo {$win}
#
# This is the main redo procedure.
# If no win is given, the topWindow will be taken.
# The redo will be done direct into the renamed (wrapped) widget (.$win.text).
# - Refer to typewriter.create -
#
proc redo {{win ""}} {
	global undo_buffer UNDO_BUFFER_SIZE winInfo

	set win [window_check $win]

	# If nothing is to do, return
	if {!$undo_buffer($win-undo_count)} {
		return
	}

	# Set pointer to the undo buffer
	set p $undo_buffer($win-pos)

	# Get the redo information from the undo buffer
	set index $undo_buffer($win-index-$p)
	set n [llength $undo_buffer($win-cmd-$p)]

	for {set i 0} {$i < $n} {incr i} {
		set cmd [lindex $undo_buffer($win-cmd-$p) $i]
		set data [lindex $undo_buffer($win-data-$p) $i]
		
		# Set the cursor to the indexed position
		uplevel [list .$win.text mark set insert $index]

		# Forward the undo command direct into the text widget (.$win.text)
		if {$cmd == "insert"} {
			uplevel [list .$win.text insert insert $data]
		} else {
			uplevel [list .$win.text delete insert "$index + [string length $data] c"] 
		}
	}

	# Correct the undo buffer counter and pointer
	set undo_buffer($win-pos) [expr [incr p] % $UNDO_BUFFER_SIZE]
	set undo_buffer($win-undo_count) [expr $undo_buffer($win-undo_count) - 1]

	# Increment undo buffer cmd count
	if {$undo_buffer($win-count) < $UNDO_BUFFER_SIZE} {
		incr undo_buffer($win-count)
	}

	# Update menu entry Redo
	if {!$undo_buffer($win-undo_count)} {
		.mbar.edit entryconfigure Redo -state disabled
	}

	# Force view at insert position
	uplevel [list .$win.text see insert]
}
