#####
# Globales
#
# The counters are used from the proc typewriter.open to give 
# each typewriter window a unique name.
#
set counter(new) 0
set counter(open) 0



#####
# win typewriter.create {win}
#
# The procedure creates a new toplevel window with horizontal and vertical scrollbars, 
# maps the default bindings to the text widget, and installs the wm protocolls.
#
# Parameters:
#	win  : The name of the window
#	title: The title of the window
#
proc typewriter.create {win {title ""}} {
    global winInfo topWindow

	toplevel $win -class Typewriter
	wm protocol $win WM_DELETE_WINDOW "typewriter.close $win"
	wm group $win .

	if {$title != ""} {
		wm title $win $title
		wm iconname $win $title
	}


	text $win.text -xscrollcommand "$win.xsbar set" -yscrollcommand "$win.ysbar set"
	scrollbar $win.xsbar -orient horizontal -takefocus 0 -command "$win.text xview"
	scrollbar $win.ysbar -orient vertical -takefocus 0 -command "$win.text yview"

	grid $win.text $win.ysbar -sticky nsew
	grid $win.xsbar -sticky nsew
	grid columnconfigure $win 0 -weight 1
	grid rowconfigure $win 0 -weight 1

	# Set tabulator positions 
	set em [font measure [$win.text cget -font] m]
	for {set i 1} {$i < 50} {incr i} {
		lappend tabs [expr $i * $em * 4]
	}
	$win.text configure -tabs $tabs -highlightcolor [$win.text cget -highlightbackground]

	set winInfo($win-filename) ""
	set winInfo($win-dirty) 0
	undo_init $win

	# Map functions to control keys, and set bindings
	functions.bind $win.text
	bind $win.text <Destroy> "typewriter.destroy $win"
	bind $win.text <FocusIn> "set topWindow $win"

	# Wrap the text widget
	rename $win.text .$win.text

	proc $win.text {args} {
		global winInfo

		# Get the name of the widget
		set win .[lindex [split [lindex [info level [info level]] 0] "."] 1]

		# Set the dirty flag while insert or delete if not already set, and
		# push data into the undo buffer
		if [regexp {^(ins|del).*} [lindex $args 0]] {
	
			if {!$winInfo($win-dirty)} {
				set winInfo($win-dirty) 1
				wm title $win "* [wm title $win]"
				wm iconname $win [wm title $win]
			}

			# Get undo information
			set cmd [lindex $args 0]
			set index [lindex $args 1]
			set sel [expr ![catch {uplevel .$win.text index sel.first}]]

			# Index end is set to set to the beginning of the next line.
			if {$index == "end"} {
				set index [uplevel [list .$win.text index "end-1l lineend"]]
			} else {
				set index [uplevel [list .$win.text index $index]]
			}	

			if {$cmd == "delete"} {
				# "Delete until end of line" sends "insert lineend" as index2
				if {[set index2 [lindex $args 2]] != ""} {
					set index2 [uplevel [list .$win.text index $index2]]
				}
				set data [uplevel [list .$win.text get $index] $index2]
			} else {
				set data [lindex $args 2]
			}
		
			# Forward the undo information to the undo buffer
			after idle [list undo_push $win $cmd $sel $index $data]
		}

		# Forward the args to the text widget
		# Do not change the position of this line !
		uplevel .$win.text $args
	}

	# Set the focus into the text widget, and return
	focus $win.text

	return $win
}



#####
# typewriter.destroy {win}
#
# The procedure is called up from the event <Destroy>, and 
# deletes all window specific variables.
#
proc typewriter.destroy {win} {
    global winInfo undo_buffer
    
    unset winInfo($win-filename)
    unset winInfo($win-dirty)
	undo_term $win

	# Remove wrapper
	rename $win.text {}
}



#####
# win typewriter.open {args}
#
# args: win or -file filename
#
# The procedure accepts a window name or the option -file with a filename.
# If a filename is given, the procedure is searching whithin the window list, whether 
# the file is already loaded. If yes, the window will be raised. Otherwise, the file
# will be loaded.
# 
proc typewriter.open {args} {
    global counter winInfo

	# File or window ?
	if {[lindex $args 0] == "-file"} {
		set filename [lindex $args 1]
		set win [typewriter.find $filename]
	} else {
		set filename ""
		set win [lindex $args 0]
	}

	if {$win != ""} {
		window_raise $win
	} else {
		if {$filename == ""} {
			set win [typewriter.create ".n[incr counter(new)]" "New $counter(new)"]
		} else {
			if {[file exists $filename] && ![file readable $filename]} {
				set msg "Can not read file $filename. \nPermission denied. "
				tk_messageBox -title "Typewriter: Open file" -default ok -message $msg -icon error -type ok
				return
			}
			set win [typewriter.create ".o[incr counter(open)]" "# [file tail $filename]"]
 	  	 	set winInfo($win-filename) $filename
			set winInfo($win-dirty) 2

			# Open the file, and read in chunks of 1kB direct into the text widget (.$win.text)
			if {[file exists $filename]} {
				set fid [open $filename r]
				.$win.text insert end [read $fid 1024]
				.$win.text mark set insert 1.0

				# The catch within the while is to avoid errors, if a big file
				# read and the typewriter is closed while reading the file.
				while {![eof $fid]} {
					if {[catch {.$win.text insert end [read $fid 1024]}]} {
						close $fid
						return 
					}
					update
				}
				close $fid
		
				# Delete the newline 
				.$win.text delete "end - 1c"
			}

			set winInfo($win-dirty) 0
			wm title $win [file tail $filename]
			wm iconname $win [file tail $filename]
		}
	}
	return $win
}



#####
# typewriter.save
#
# Saves the file. If no filename was given, the procedure ask for a new filename.
# In mode -as the filedialog will be opened.
#
# returns: 0	- File not saved
#          1 - File saved
#
proc typewriter.save {{win ""} {mod ""}} {
    global winInfo fileTypeList fileExtension

	set win [window_check $win]
	if {$win == ""} {
		return 1
	}

	# Avoid saving while loading
	if {$winInfo($win-dirty) == 2} {
		return 1
	}

	if {$winInfo($win-filename) == "" || $mod == "as"} {
		set initFile [file tail $winInfo($win-filename)]

		if {[set filename [tk_getSaveFile -filetypes $fileTypeList -initialfile $initFile]] == ""} {
			return 0
		}
		set winInfo($win-filename) $filename
    }
    
	$win configure -cursor "exchange"
	update

    if {[catch {set fid [open $winInfo($win-filename) w]}]} {
		set msg "Cannot save file. \nPermission denied."
		tk_messageBox -title "Typewriter: Save file" -default ok -message $msg -icon error -type ok
		return
	}

    puts $fid [.$win.text get 1.0 "end - 1 char"]
    close $fid

	set winInfo($win-dirty) 0
	wm title $win [file tail $winInfo($win-filename)]
	wm iconname $win [file tail $winInfo($win-filename)]

	$win configure -cursor ""
	update

	return 1
}



#####
# typewriter.close {win} 
#
# The procedure destroys the window. 
# If no win is given, the topWindow will be destroyed
#
proc typewriter.close {{win ""}} { 
	global winInfo topWindow
	
	set win [window_check $win]

	if {$win == ""} {
		return 1
	}

	# Save file dialog ?
	if {$winInfo($win-dirty) == 1} {
		set msg "Save file"

		if {$winInfo($win-filename) == ""} {
			set msg $msg?
		} else {
			set msg "$msg \"[file tail $winInfo($win-filename)]\"?"
		}
		window_raise $win
		set answer [tk_messageBox -title "Typewriter: Save?" -default yes -message $msg -icon question -type yesnocancel]

		switch $answer {
			yes {
				if {[typewriter.save $win]} {
					destroy $win
				} else {
					return 0
				}
			}
			no	{ destroy $win }
			cancel { return 0 }
		}
	} else {
	    destroy $win
	}

	set topWindow ""

    return 1
}



#####
# window_raise {win}
#
proc window_raise {win} {
	global topWindow

	set win [winfo toplevel $win]

	wm deiconify $win
	raise $win
}



#####
# typewriter.find {filename}
#
# Find the window name for a filename.
#
proc typewriter.find {filename} {
	global winInfo
	set win ""

	foreach i [array names winInfo *filename] {
		if {$winInfo($i) == $filename} {
			set win [lindex [split $i -] 0]
			break;
		}
	}
	return $win
}



#####
# typewriter.toggle_wrap_mode {}
#
proc typewriter.toggle_wrap_mode {} {
	global topWindow topWindowWrapMode

	if {$topWindow == ""} {	
		return
	}

	$topWindow.text configure -wrap $topWindowWrapMode
}



#####
# window_check {win}
#
proc window_check {win} {
	global topWindow

	if {$win == ""} {
		return $topWindow
	}
	return $win
}
