#ifndef narrayInt_h
#define narrayInt_h

/* $Id: narrayInt.h,v 1.1 1998/08/20 16:54:44 nickm Exp nickm $ */

/*
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#if STDC_HEADERS || HAVE_STRING_H
#  include <string.h>
#  if !STDC_HEADERS && HAVE_MEMORY_H
#    include <memory.h>
#  endif
#else
#  include <strings.h>
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#  define memcmp(s1, s2, n) bcmp ((s1), (s2), (n))
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#else
#error "need stdarg.h"
#endif

#ifdef HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif

/* compiled code is basically a sequence of unsigned longs.
 * the top 5 bits are opcode, the bottom 27 are data.
 * to keep track of Code* sequences, we prepend a CodeInfo
 * struct.  Each Code* sequence will have an OP_INFO that
 * points to this CodeInfo struct.
 */

typedef unsigned long Code;

typedef struct {
    int len;
    int alloced;
} CodeInfo;

#define GET_CODE_INFO(info, code) \
    assert(((*code) & OPCODE_MASK) == OP_INFO); \
    info = (CodeInfo*) code[1]

#define OPCODE_SHIFT 27
#define OPCODE_MASK  (~0 << OPCODE_SHIFT)
#define OPERAND_MASK ~OPCODE_MASK
#define OP_PUSH_MAXINT (1 << (OPCODE_SHIFT - 2))
#define DECODE_INT(code, n) \
    n = *code & OPERAND_MASK; \
    if (n > OP_PUSH_MAXINT) n |= OPCODE_MASK

#define OP_PUSH  (1 << OPCODE_SHIFT) /* push an integer (< OP_PUSH_MAXINT) */
#define OP_PUSHD (2 << OPCODE_SHIFT) /* push a NArrayFloat, operand is index
				      * index closure->double_table
				      */
#define OP_ADD   (3 << OPCODE_SHIFT) /* add */
#define OP_SUB   (4 << OPCODE_SHIFT) /* subtract */
#define OP_MUL   (5 << OPCODE_SHIFT) /* multiply */
#define OP_DIV   (6 << OPCODE_SHIFT) /* divide */
#define OP_FN    (7 << OPCODE_SHIFT) /* call function -- next word is
				      * address of function, operand
				      * is # of arguments
				      */
#define OP_REF   (8 << OPCODE_SHIFT) /* push the value of the top of the
				      * location stack
				      */
#define OP_SET   (9 << OPCODE_SHIFT) /* set the value of the top of the
				      * location to the top of the operand
				      * stack
				      */
#define OP_NEG   (10 << OPCODE_SHIFT) /* negate */ 
#define OP_NOT   (11 << OPCODE_SHIFT) /* 0 -> 1, anything else -> 0 */
#define OP_EQ    (12 << OPCODE_SHIFT) /* 1 if equal */
#define OP_LE    (13 << OPCODE_SHIFT) /* less than or equal */
#define OP_LT    (14 << OPCODE_SHIFT) /* less than */
#define OP_GE    (15 << OPCODE_SHIFT) /* greater than or equal */
#define OP_GT    (16 << OPCODE_SHIFT) /* greater than */
#define OP_OR    (17 << OPCODE_SHIFT) /* (logical) or */
#define OP_AND   (18 << OPCODE_SHIFT) /* (logical) and */
#define OP_JUMPT (19 << OPCODE_SHIFT) /* jump forward operand instructions if
				       * stack top != 0
				       */
#define OP_JUMP  (20 << OPCODE_SHIFT) /* jump forward operand intructions */
#define OP_INFO  (21 << OPCODE_SHIFT) /* next word points to CodeInfo */
#define OP_ID    (22 << OPCODE_SHIFT) /* push the address of a variable
				       * onto the location stack -- also
				       * returned by the scanner when
				       * it scans an indentifer
				       */
#define OP_MARK  (23 << OPCODE_SHIFT) /* push a mark onto the mark stack */
#define OP_THIS  (24 << OPCODE_SHIFT) /* reference the current element
				       * of the array
				       */
#define OP_ARRAY (25 << OPCODE_SHIFT) /* reference the current element
				       * of an array in closure->array_table
				       */
#define OP_DUPLV (26 << OPCODE_SHIFT) /* duplicate the top of the location
				       * stack
				       */
#define OP_IREF  (27 << OPCODE_SHIFT) /* push the value of a index */
#define OP_STRING (28 << OPCODE_SHIFT) /* push a string onto the string
					* stack
					*/
#define OP_END   0

/* the stack machine maintains 3 stacks:  the operand stack, the mark
 * stack, and the location stack.  The mark stack is used in referencing
 * array elements, the location stack holds addresses of things that
 * can be OP_REF'ed or OP_SET.
 */

#define MAX_OPERAND_STACK 64
#define MAX_LOCATION_STACK 16
#define MAX_MARK_STACK 16

typedef enum LocationType {
    LOCATION_TYPE_VAR, LOCATION_TYPE_ELEMENT
} LocationType;
    
typedef struct {
    NArrayFloat* ptr;
} LocationStack[MAX_LOCATION_STACK];

typedef NArrayOperand OperandStack[MAX_OPERAND_STACK];

typedef int MarkStack[MAX_MARK_STACK];

#define DEBUG_PARSE 1
#define DEBUG_TRACE 2
#define DEBUG_DUMP  4

/* stuff for the parser/scanner */

typedef unsigned long* CodePtr;
#define YYSTYPE CodePtr

#undef YY_INPUT
#define YY_INPUT(buf, result, max_size) \
    result = NArray_yyinput(buf, max_size)

#undef yyerror
#define yyerror(s) NArray_yyerror(s)

#define YYERROR_VERBOSE 1	/* for bison */

#define MAX_ARITY_STACK 10

/* the compiler keeps state in a (global) CompileState struct.
 */
typedef struct {
    char* expression;
    int expr_ndx;
    int expr_length;
    Code* code;
    int ok;
    NArray_Closure* closure;
    int use_error_msg;		/* if 1, then use error_msg otherwise
				 * use whatever yyerror said
				 */
    char error_msg[64];
    int arity_stack[MAX_ARITY_STACK]; /* holds # of arguments to a fn call */
    int arity_sp;
} CompileState;

extern CompileState* narray_compile_state;

/* function prototypes */

extern int yyparse(void);
extern Code* NArray_Compile(NArray* array, char* expression);
extern int NArray_ApplyCode(NArray* array, Code* code);
extern void NArray_PrintCode(NArray* array, Code* code);
extern int NArray_CodeLength(Code* code);
extern void NArray_FreeCode(Code* code);
extern Code* NArray_VarMakeCode(Code first, ...);
extern Code* NArray_AppendCodeAndFree(Code* a, Code* b);
extern Code* NArray_AppendOps(Code* a, Code first, ...);
extern Code NArray_LookupFn(int fn);
extern Code* NArray_AppendDouble(Code* code, double d);
extern Code* NArray_AppendId(Code* code, char* id);
extern Code* NArray_AppendString(Code* code, char* string);
extern int NArray_CreateClosureIdSlot(NArray_Closure* closure, char* id);
extern void NArray_ExpandClosureIds(NArray_Closure* closure, char* id);
extern int NArray_yyinput(char* buf, int max_size);
extern void NArray_yyerror(char* s);
extern void NArray_UninternId(int id);
extern int NArray_CodeInit(Tcl_Interp* interp);
extern int NArray_FunctionsInit(Tcl_Interp* interp);

#endif
