/* 
 * hash.c --
 *
 *	This file contains code for the "hash" package. It provides
 *	a Tcl command for creating and accessing hash tables.
 *
 * Copyright (c) 2000, Neil D. McKay
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 *  $Header: /home/neil/src/tcl/hash/RCS/hash.c,v 1.4 2000/10/18 04:26:49 neil Exp $
 */

#include <tcl.h>

/*
 * The HashDescrip struct contains the data used by each
 * hash table object. A pointer to one of these is passed
 * to the object command associated with the hash table.
 * Keys for entries in the hash table are character strings;
 * values are Tcl_Obj pointers.
 */

typedef struct {
    Tcl_HashTable hashTable;	/* Hash table				*/
    int writable;		/* Boolean: entries may be added
				 * if true, entries are fixed if false	*/
    Tcl_Obj *missingEntryProc;	/* Name of command to execute if
				 * a non-existent entry is referenced.	*/
    Tcl_Obj *deleteProc;	/* Name of command to execute when the
				 * hash table is deleted.		*/
    Tcl_Interp *interp;		/* Interpreter in which associated
				 * object command resides		*/
    Tcl_Command hashCommand;	/* Command token of object command	*/
    int flags;			/* Misc. flags.				*/
} HashDescrip;

/*
 * Flags for the hash object.
 */

#define HASH_BEING_DELETED	0x1	/* Indicates the hash is in the
					 * process of being deleted. (Used
					 * to keep object cleanup from
					 * being done multiple times.)	*/

/*
 * The next few procs implement the various hash table object
 * commands: setting and getting values, deleting entries, etc.
 */

/*
 *----------------------------------------------------------------------
 *
 * HashCgetProc --
 *
 *	Implements the "cget" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashCgetProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    HashDescrip *hashPtr;
    char *optName;

    hashPtr = (HashDescrip *) clientData;

    /* Check args. */

    if (objc != 1) {
	Tcl_SetResult(interp, "Usage: <hashObjCmd> cget <optionName>",
			TCL_STATIC);
	return(TCL_ERROR);
    }

    /* Get and return argument value. */

    optName = Tcl_GetStringFromObj(objv[0], NULL);
    if(strcmp(optName, "-allowchanges") == 0) {
	Tcl_SetObjResult(interp, Tcl_NewBooleanObj(hashPtr->writable));
    } else if (strcmp(optName, "-missingentryproc") == 0) {
	if (hashPtr->missingEntryProc != NULL) {
	    Tcl_SetObjResult(interp, hashPtr->missingEntryProc);
	}
    } else if (strcmp(optName, "-deleteproc") == 0) {
	if (hashPtr->deleteProc != NULL) {
	    Tcl_SetObjResult(interp, hashPtr->deleteProc);
	}
    } else {
	Tcl_AppendResult(interp, "Unknown option name \"",
		optName, "\"; should be -allowchanges, -missingentryproc or -deleteproc",
		NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashConfigureProc --
 *
 *	Implements the "configure" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashConfigureProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    HashDescrip *hashPtr;
    Tcl_Obj *resultObj;
    int i, optLen, optNameLen;
    char *optVal, *optName;

    /* If only one argument, treat it like a "cget". */

    if(objc == 1) {
	return(HashCgetProc(clientData, interp, objc, objv));
    }

    /* Better have an even number of arguments... */

    if ((objc % 2) != 0) {
	Tcl_SetResult(interp,
		"Usage:\n\t<hashCmd> configure ?<option>?\nor\n\t<hashCmd> configure ?<option> <value>? ...",
		TCL_STATIC);
	return(TCL_ERROR);
    }

    hashPtr = (HashDescrip *) clientData;

    /* If no arguments, spit out all the option-value pairs. */

    if (objc == 0) {
	/* Return a list of option-value pairs. */

	resultObj = Tcl_NewListObj(0, NULL);
	Tcl_ListObjAppendElement(NULL, resultObj,
			Tcl_NewStringObj("-allowchanges", -1));
	Tcl_ListObjAppendElement(NULL, resultObj,
			Tcl_NewBooleanObj(hashPtr->writable));
	Tcl_ListObjAppendElement(NULL, resultObj,
			Tcl_NewStringObj("-missingentryproc", -1));
	if (hashPtr->missingEntryProc != NULL) {
	    Tcl_ListObjAppendElement(NULL, resultObj,
				hashPtr->missingEntryProc);
	} else {
	    Tcl_ListObjAppendElement(NULL, resultObj, Tcl_NewObj());
	}
	Tcl_ListObjAppendElement(NULL, resultObj,
			Tcl_NewStringObj("-deleteproc", -1));
	if (hashPtr->deleteProc != NULL) {
	    Tcl_ListObjAppendElement(NULL, resultObj,
				hashPtr->deleteProc);
	} else {
	    Tcl_ListObjAppendElement(NULL, resultObj, Tcl_NewObj());
	}

	Tcl_SetObjResult(interp, resultObj);
	return TCL_OK;
    }

    /* Options are specified: set them. */

    for(i = 0; i < objc; i += 2) {
	optName = Tcl_GetStringFromObj(objv[i], &optNameLen);
	if (strncmp(optName, "-allowchanges", optNameLen) == 0) {
	    if (Tcl_GetBooleanFromObj(interp, objv[i + 1],
				&(hashPtr->writable)) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else if (strncmp(optName, "-missingentryproc", optNameLen) == 0) {
	    if (hashPtr->missingEntryProc != NULL) {
		Tcl_DecrRefCount(hashPtr->missingEntryProc);
	    }
	    optVal = Tcl_GetStringFromObj(objv[i + 1], &optLen);
	    if (optLen <= 0) {
		hashPtr->missingEntryProc = NULL;
	    } else {
		hashPtr->missingEntryProc = objv[i + 1];
		Tcl_IncrRefCount(objv[i + 1]);
	    }
	} else if (strncmp(optName, "-deleteproc", optNameLen) == 0) {
	    if (hashPtr->deleteProc != NULL) {
		Tcl_DecrRefCount(hashPtr->deleteProc);
	    }
	    optVal = Tcl_GetStringFromObj(objv[i + 1], &optLen);
	    if (optLen <= 0) {
		hashPtr->deleteProc = NULL;
	    } else {
		hashPtr->deleteProc = objv[i + 1];
		Tcl_IncrRefCount(objv[i + 1]);
	    }
	} else {
	    Tcl_AppendResult(interp,
		"Unknown option \"", optName, "\"; should be -allowchanges, -missingentryproc, or -deleteproc",
		NULL);
	    return TCL_ERROR;
	}
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashContentsProc --
 *
 *	Implements the "contents" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashContentsProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    HashDescrip *hashPtr;
    Tcl_Obj *resultObj;
    int i, cObjc, isNew;
    Tcl_Obj **cObjv;

    /* Check args. */

    if (objc > 1) {
	Tcl_SetResult(interp,
		"Usage: <hashCmd> contents ?contentsList?", TCL_STATIC);
	return TCL_ERROR;
    }

    hashPtr = (HashDescrip *) clientData;

    if (objc == 0) {
	resultObj = Tcl_NewListObj(0, NULL);
	entryPtr = Tcl_FirstHashEntry(&(hashPtr->hashTable), &search);
	while(entryPtr != NULL) {
	    Tcl_ListObjAppendElement(interp, resultObj,
		Tcl_NewStringObj(Tcl_GetHashKey(&(hashPtr->hashTable),
							entryPtr), -1));
	    Tcl_ListObjAppendElement(interp, resultObj,
			(Tcl_Obj *) Tcl_GetHashValue(entryPtr));
	    entryPtr = Tcl_NextHashEntry(&search);
	}

	Tcl_SetObjResult(interp, resultObj);
	return TCL_OK;
    }

    /* One argument; must be "contents" list. */

    if (Tcl_ListObjGetElements(interp, objv[0], &cObjc, &cObjv) != TCL_OK) {
	return TCL_ERROR;
    }

    if ((cObjc % 2) != 0) {
	Tcl_SetResult(interp,
		"hash contents list has an odd number of elements", TCL_STATIC);
	return TCL_ERROR;
    }

    if (hashPtr->writable) {
	for(i = 0; i < cObjc; i += 2) {
	    entryPtr = Tcl_CreateHashEntry(&(hashPtr->hashTable),
			Tcl_GetStringFromObj(cObjv[i], NULL), &isNew);
	    if (!isNew) {
		Tcl_DecrRefCount((Tcl_Obj *) Tcl_GetHashValue(entryPtr));
	    }
	    Tcl_SetHashValue(entryPtr, cObjv[i + 1]);
	    Tcl_IncrRefCount(cObjv[i + 1]);
	}
    } else {
	for(i = 0; i < cObjc; i += 2) {
	    entryPtr = Tcl_FindHashEntry(&(hashPtr->hashTable),
			Tcl_GetStringFromObj(cObjv[i], NULL));
	    if (entryPtr == NULL) {
		Tcl_AppendResult(interp, "hash entry \"",
			Tcl_GetStringFromObj(cObjv[i], NULL),
			"\" does not exist, and hash is not writable",
			NULL);
		return TCL_ERROR;
	    }
	    Tcl_DecrRefCount((Tcl_Obj *) Tcl_GetHashValue(entryPtr));
	    Tcl_SetHashValue(entryPtr, cObjv[i + 1]);
	    Tcl_IncrRefCount(cObjv[i + 1]);
	}
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashDeleteProc --
 *
 *	Implements the "delete" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashDeleteProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    Tcl_HashEntry *entryPtr;
    HashDescrip *hashPtr;

    /* Check args. */

    if (objc != 1) {
	Tcl_SetResult(interp,
		"Usage: <hashName> delete <entryName>", TCL_STATIC);
	return TCL_ERROR;
    }

    hashPtr = (HashDescrip *) clientData;

    /* Crap out if the hash table's not modifiable. */

    if(!(hashPtr->writable)) {
	Tcl_SetResult(interp, "Hash table is not modifiable", TCL_STATIC);
	return TCL_ERROR;
    }

    /* Find the appropriate entry. */

    entryPtr = Tcl_FindHashEntry(&(hashPtr->hashTable),
				Tcl_GetStringFromObj(objv[0], NULL));
    if (entryPtr == NULL) {
	Tcl_AppendResult(interp, "Hash entry \"",
			Tcl_GetStringFromObj(objv[0], NULL),
			"\" doesn't exist", NULL);
	return TCL_ERROR;
    }

    /* Delete it. */

    Tcl_DecrRefCount((Tcl_Obj *) Tcl_GetHashValue(entryPtr));
    Tcl_DeleteHashEntry(entryPtr);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashNamesProc --
 *
 *	Implements the "names" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashNamesProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    Tcl_Obj *resultObj;
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    HashDescrip *hashPtr;
    char *key, *pattern;

    /* Check args. */

    if (objc > 1) {
	Tcl_SetResult(interp, "Usage: <hashCmd> names ?pattern?", TCL_STATIC);
	return TCL_ERROR;
    }

    hashPtr = (HashDescrip *) clientData;

    /* Set the pattern, if there is one. */

    if (objc > 0) {
	pattern = Tcl_GetStringFromObj(objv[0], NULL);
    }

    /* Check each key. */

    resultObj = Tcl_NewListObj(0, NULL);
    entryPtr = Tcl_FirstHashEntry(&(hashPtr->hashTable), &search);
    while(entryPtr != NULL) {
	key = Tcl_GetHashKey(&(hashPtr->hashTable), entryPtr);
	if (objc <= 0 || Tcl_StringMatch(key, pattern)) {
	    Tcl_ListObjAppendElement(interp, resultObj,
		    Tcl_NewStringObj(key, -1));
	}
	entryPtr = Tcl_NextHashEntry(&search);
    }

    Tcl_SetObjResult(interp, resultObj);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashSetProc --
 *
 *	Implements the "set" hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashSetProc(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    HashDescrip *hashPtr;
    char *entryName;
    Tcl_HashEntry *entryPtr;
    int isNew;
    Tcl_Obj *missingObjv[3];

    /* Check args. */

    if(objc > 3 || objc < 1) {
	Tcl_SetResult(interp,
		"Usage: <hashCmd> set <entryName> ?=? ?value?", TCL_STATIC);
	return TCL_ERROR;
    }

    hashPtr = (HashDescrip *) clientData;
    entryName = Tcl_GetStringFromObj(objv[0], NULL);

    /* First we'll handle retrieval of a value. */

    if (objc == 1) {
	entryPtr = Tcl_FindHashEntry(&(hashPtr->hashTable), entryName);
	if (entryPtr != NULL) {
	    Tcl_SetObjResult(interp, (Tcl_Obj *) Tcl_GetHashValue(entryPtr));
	    return TCL_OK;
	}

	/*
	 * Entry doesn't exist. If there's no procedure for
	 * generating values for non-existent entries, return
	 * an error.
	 */

	if (hashPtr->missingEntryProc == NULL) {
	    Tcl_AppendResult(interp, "non-existent hash table entry \"",
			entryName, "\"", NULL);
	    return TCL_ERROR;
	}

	/*
	 * Invoke the "missing entry" proc. The arguments are
	 * the hash table name and the entry name, so we have
	 * to figure out the hash table name. We'll do this
	 * by mapping the command token back to a command name.
	 */

	missingObjv[0] = hashPtr->missingEntryProc;
	missingObjv[1] = Tcl_NewStringObj(Tcl_GetCommandName(interp,
						hashPtr->hashCommand), -1);
	missingObjv[2] = objv[0];
	return(Tcl_EvalObjv(interp, 3, missingObjv, TCL_EVAL_GLOBAL));
    }

    /* We're setting a value. Check for the optional "=" sign. */

    if (objc == 3) {
	if (strcmp(Tcl_GetStringFromObj(objv[1], NULL), "=") != 0) {
	    Tcl_AppendResult(interp,
		"argument \"", Tcl_GetStringFromObj(objv[1], NULL),
		"\" should have been \"=\"", NULL);
	    return TCL_ERROR;
	}
    }

    /*
     * Find the entry to be changed, and take appropriate
     * action if it's missing. (Gotta update Tcl_Obj
     * reference counts appropriately too.)
     */

    if (hashPtr->writable) {
	entryPtr = Tcl_CreateHashEntry(&(hashPtr->hashTable),
							entryName, &isNew);
	if (!isNew) {
	    Tcl_DecrRefCount((Tcl_Obj *) Tcl_GetHashValue(entryPtr));
	}
    } else {
	entryPtr = Tcl_FindHashEntry(&(hashPtr->hashTable), entryName);
	if(entryPtr == NULL) {
	    Tcl_SetResult(interp, "hash table is not modifiable", TCL_STATIC);
	    return TCL_ERROR;
	}
    }

    /* Whew! Now we can (finally) modify the hash table value. */

    Tcl_SetHashValue(entryPtr, objv[objc - 1]);
    Tcl_IncrRefCount(objv[objc - 1]);

    /* Return the new entry value, just like the "set" command. */

    Tcl_SetObjResult(interp, objv[objc - 1]);
    return TCL_OK;
}

/*
 * Type HashSubCmdEntry describes an entry in a table of
 * subcommands of the hash object command. This is just
 * a mapping from a name to a proc.
 */

typedef struct {
    char *name;
    Tcl_ObjCmdProc *proc;
} HashSubCmdEntry;

/* List of subcommands for the hash object command. */

HashSubCmdEntry subCmdTab[] = {
    "cget",		HashCgetProc,
    "configure",	HashConfigureProc,
    "contents",		HashContentsProc,
    "delete",		HashDeleteProc,
    "unset",		HashDeleteProc,	/* unset is an alias for delete */
    "names",		HashNamesProc,
    "set",		HashSetProc,
    "->",		HashSetProc,	/* "->" is an alias for "set" */
    NULL, NULL
};

/*
 *----------------------------------------------------------------------
 *
 * HashObjCmd --
 *
 *	Implements the hash object command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashObjCmd(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    HashDescrip *hashPtr;
    char *cmdName, *prefixStr;
    int i, cmdLen;

    /* Check arguments */

    if (objc < 2) {
	Tcl_SetResult(interp,"Usage: <hashName> hashCommand ?args?",
			TCL_STATIC);
	return TCL_ERROR;
    }

    hashPtr = (HashDescrip *) clientData;

    /* Execute the appropriate subcommand. */

    cmdName = Tcl_GetStringFromObj(objv[1], NULL);
    cmdLen = strlen(cmdName);
    for(i = 0; subCmdTab[i].name != NULL; i++) {
	if (strncmp(cmdName, subCmdTab[i].name, cmdLen) == 0) {
	    return((*( subCmdTab[i].proc))(clientData, interp, objc-2, objv+2));
	}
    }

    /* Subcommand not found. */

    Tcl_AppendResult(interp, "Unknown hash command \"",
		cmdName, "\"; should be one of: ", NULL);
    prefixStr = "";
    for(i = 0; subCmdTab[i].name != NULL; i++) {
	Tcl_AppendResult(interp, prefixStr, subCmdTab[i].name, NULL);
	prefixStr = ", ";
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * HashDeleteCmd --
 *
 *	This is called whenever a hash object command is deleted.
 *	It frees storage, and does any other necessary cleanup.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Storage is deallocated.
 *
 *----------------------------------------------------------------------
 */

static void
HashDeleteCmd(clientData)
    ClientData clientData;
{
    HashDescrip *hashPtr;
    Tcl_HashSearch search;
    Tcl_HashEntry *entryPtr;
    Tcl_Obj *objPtr;
    Tcl_Obj *objv[2];

    hashPtr = (HashDescrip *) clientData;

    /* If this hash is already being deleted, we can skip out. */

    if (hashPtr->flags & HASH_BEING_DELETED) {
	return;
    }

    /* Shouldn't need this, but just in case... */

    Tcl_Preserve(clientData);

    /*
     * Indicate that this hash is being deleted. We have to
     * do this in case the deleteProc tries to delete the
     * hash again; we don't want to do everything twice.
     */

    hashPtr->flags |= HASH_BEING_DELETED;

    /*
     * Execute the "deleteproc" command, if there is one. We
     * do this first, so that the hash table is still intact
     * when we execute the command.
     */

    if (hashPtr->deleteProc != NULL) {
	objv[0] = hashPtr->deleteProc;
	objv[1] = Tcl_NewStringObj(Tcl_GetCommandName(hashPtr->interp,
						hashPtr->hashCommand), -1);
	Tcl_EvalObjv(hashPtr->interp, 2, objv, TCL_EVAL_GLOBAL);
    }

    /* Decrement the reference count for each stored object. */

    entryPtr = Tcl_FirstHashEntry(&(hashPtr->hashTable), &search);
    while(entryPtr != NULL) {
	objPtr = (Tcl_Obj *) Tcl_GetHashValue(entryPtr);
	Tcl_DecrRefCount(objPtr);
	entryPtr = Tcl_NextHashEntry(&search);
    }

    /* Free the hash table's storage. */

    Tcl_DeleteHashTable(&(hashPtr->hashTable));

    /* Free the storage used for the "missing entry" procedure. */

    if(hashPtr->missingEntryProc != NULL) {
	Tcl_DecrRefCount(hashPtr->missingEntryProc);
    }
    /* Free the storage used for the "delete" procedure. */

    if(hashPtr->deleteProc != NULL) {
	Tcl_DecrRefCount(hashPtr->deleteProc);
    }

    Tcl_Release(clientData);
    Tcl_EventuallyFree(clientData, Tcl_Free);
}

/*
 *----------------------------------------------------------------------
 *
 * HashCreateProc --
 *
 *	Implements the "hash::create" command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	A hash table object is created, along with its
 *	associated object command.
 *
 *----------------------------------------------------------------------
 */

static int
HashCreateCommand(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    int i;
    char *optName;
    HashDescrip *hashPtr;

    /* Check arguments. */

    if (objc < 2) {
	Tcl_SetResult(interp, "Usage: ", TCL_STATIC);
	Tcl_AppendResult(interp, Tcl_GetStringFromObj(objv[0], NULL),
		" hashName ?options?", NULL);
	return TCL_ERROR;
    }

    if (objc % 2 != 0) {
	Tcl_SetResult(interp,
	    "Odd number of optional arguments; must be option-value pairs",
								TCL_STATIC);
	return TCL_ERROR;
    }

    /* Allocate table descriptor. */

    hashPtr = (HashDescrip *) Tcl_Alloc(sizeof(HashDescrip));
    hashPtr->writable = 1;
    hashPtr->missingEntryProc = NULL;
    hashPtr->deleteProc = NULL;
    hashPtr->flags = 0;

    /* Parse options */

    for(i = 2; i < objc; i += 2) {
	optName = Tcl_GetStringFromObj(objv[i], NULL);
	if (strcmp(optName, "-allowchanges") == 0) {
	    if (Tcl_GetBooleanFromObj(interp, objv[i + 1],
					&(hashPtr->writable)) != TCL_OK) {
		Tcl_Free((ClientData) hashPtr);
		return TCL_ERROR;
	    }
	} else if (strcmp(optName, "-missingentryproc") == 0) {
	    hashPtr->missingEntryProc = objv[i + 1];
	    Tcl_IncrRefCount(hashPtr->missingEntryProc);
	} else if (strcmp(optName, "-deleteproc") == 0) {
	    hashPtr->deleteProc = objv[i + 1];
	    Tcl_IncrRefCount(hashPtr->deleteProc);
	} else {
	    Tcl_AppendResult(interp, "Unknown option \"", optName,
			"\"; should be -allowchanges, -missingentryproc, or -deleteproc",
			NULL);
	    Tcl_Free((ClientData) hashPtr);
	    return TCL_ERROR;
	}
    }

    /* Initialize hash table */

    Tcl_InitHashTable(&(hashPtr->hashTable), TCL_STRING_KEYS);

    /* Create hash object command. */

    hashPtr->interp = interp;
    hashPtr->hashCommand = Tcl_CreateObjCommand(interp,
		Tcl_GetStringFromObj(objv[1], NULL),
		HashObjCmd, (ClientData) hashPtr, HashDeleteCmd);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * HashExistsProc --
 *
 *	Implements the "hash::exists" command.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashExistsCommand(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int objc;
    Tcl_Obj *CONST objv[];
{
    Tcl_CmdInfo info;

    /* Check args. */

    if (objc != 2) {
	Tcl_AppendResult(interp, "Usage: ",
		Tcl_GetStringFromObj(objv[0], NULL),
		" <hashName>", NULL);
	return TCL_ERROR;
    }

    /* See if a command with this name exists. */

    if (Tcl_GetCommandInfo(interp,
		Tcl_GetStringFromObj(objv[1], NULL), &info)) {
	/*
	 * This should be an object command, and its
	 * implementation proc should be HashObjCmd.
	 */

	if (info.isNativeObjectProc
		&& (info.objProc == HashObjCmd)) {
	    Tcl_SetObjResult(interp, Tcl_NewBooleanObj(1));
	    return TCL_OK;
	}
    }

    /* No command found; return 0. */

    Tcl_SetObjResult(interp, Tcl_NewBooleanObj(0));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Hash_Init --
 *
 *	Creates the commands in the "Hash" package, and
 *	informs the package system that the package has
 *	been loaded.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * Side effects:
 *	Commands are created; the "hash" namespace is created;
 *	the package system is informed that the "Hash" package
 *	has been created.
 *
 *----------------------------------------------------------------------
 */

int
Hash_Init(interp)
    Tcl_Interp *interp;
{
    if (Tcl_InitStubs(interp, "8.0", 0) == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_PkgRequire(interp, "Tcl", TCL_VERSION, 0) == NULL) {
	if (TCL_VERSION[0] == '7') {
	    if (Tcl_PkgRequire(interp, "Tcl", "8.0", 0) == NULL) {
		return TCL_ERROR;
	    }
	}
    }

    if (Tcl_PkgProvide(interp, "Hash", VERSION) != TCL_OK) {
	return TCL_ERROR;
    }

    Tcl_CreateObjCommand(interp, "hash::create", HashCreateCommand,
			NULL, NULL);
    Tcl_CreateObjCommand(interp, "hash::exists", HashExistsCommand,
			NULL, NULL);
    return TCL_OK;
}

