###############################################################################
###############################################################################
#                               Getleft
###############################################################################
###############################################################################
# copyright (C) 1999-2000 Andres Garcia Garcia - andresgarci@retemail.es
# Distributed under the terms of the GPL
###############################################################################
###############################################################################
# Version 0.8.3
###############################################################################
###############################################################################
# This program recursively downloads Web sites, as it goes, it changes absolute
# paths in the links to relative ones, and links to active content pages get
# changed to the resulting pages.
###############################################################################
#
# The following lines look for the most recent wish version in your path,
# keep in mind this program only works with version 8.1 or newer
#
# \
if which wish8.3&>/dev/null; then
# \
    exec wish8.3 -f $0
# \
elif which wish8.2&>/dev/null; then
# \
    exec wish8.2 -f $0
# \
elif which wish8.1&>/dev/null; then
# \
    exec wish8.1 -f $0
# \
else
# \
    exec wish -f $0
# \
fi

wm withdraw .
if {$tcl_version<8.1} {
    tk_messageBox -icon error -type ok -message "This program only works with\n\
            Tcl/Tk v 8.1 or newer, yours is:\n\t $tcl_version" -title Error
    exit
}

# Okay, we made it this far, now we have to check whether the programs Getleft
# depends on are actually within reach: curl, cat, touch and sed.

set dependencies 0
catch {exec curl}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No curl" \
     -message "Getleft depends on program\n\tcurl\nPlease check the docs"
    set dependencies 1
}
catch {exec sed}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No sed" \
     -message "Getleft depends on the program sed,\nPlease check the docs"
    set dependencies 1
}
catch {exec touch}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No touch" \
     -message "Getleft depends on the program touch,\nPlease check the docs"
    set dependencies 1
}
catch {exec cat noSuchFileWithThisFilename}
if {[lindex $errorCode 1]=="ENOENT"} {
    tk_messageBox -type ok -icon error -title "No cat" \
     -message "Getleft depends on the program cat,\nPlease check the docs"
    set dependencies 1
}

if {$dependencies==1} {
    exit
}

# All is well, so we really start

wm deiconify .

###############################################################################
# InitializeDownload
#    Takes care of downloading the Url entered by the user starting the
#    download.
#
# Parameter:
#    url: The requested Url.
#
# Returns:
#    The requested url, completed, if necessary with protocol or changed
#    if there was a redirection.
#    An empty string in case of error.
###############################################################################
proc InitializeDownload {url} {
    global directories siteUrl options relocated
    global labelTitles labelMessages

    if {![regexp -nocase {([^:/])(/)} $url]} {
        set url $url/
    }

    set parsedUrl [HtmlParser::ParseUrl $url]
    if {$parsedUrl!=1} {
        set siteUrl(prot) [lindex $parsedUrl 0]
        set siteUrl(www)  [lindex $parsedUrl 1]
        set siteUrl(dir)  [lindex $parsedUrl 2]
        set fileName      [lindex $parsedUrl 3]
        if {$fileName==""} {
            set isHtml 1
        } else {
            set isHtml [regexp -nocase {\.html?$} $fileName]
        }
        if {$siteUrl(prot)==""} {
            set siteUrl(prot) http
            set url http://$url
        } else {
            set siteUrl(prot) [string tolower $siteUrl(prot)]
            if {$siteUrl(prot)=="ftp"} {
                set options(checkSize) 0
            }
        }
    } else {
        tk_messageBox -type ok -title $labelTitles(initError) -icon error \
            -message $labelMessages(formatError)
        return
    }

    set directories(base) [Dialogos::SelectDirectory]
    if {![string compare $directories(base) ""]} {
        return
    }
    set directories(local) $directories(base)
    set siteUrl(base) $siteUrl(dir)

    set result [Download $url "-" $isHtml]
    catch {wm withdraw .bajando}

    if {$result==2} {
        tk_messageBox -icon error -type ok -title $labelTitles(error) \
                 -message "$Ventana::Rizo::errorMessage"
        return
    }

    if {($result==1)&&($isHtml==0)} {
        set localFileName [UrlToFile $url]
        if {![file exists $localFileName.html]} {
            file rename -force $localFileName $localFileName.html
            exec touch $localFileName
        }
        return $url
    }

    if {($result==3)||($result==4)} {
        set parsedUrl [HtmlParser::ParseUrl $relocated(url)]
        if {$parsedUrl==1} {
            return
        }
        set siteUrl(prot) [lindex $parsedUrl 0]
        set siteUrl(www)  [lindex $parsedUrl 1]
        set siteUrl(dir)  [lindex $parsedUrl 2]
        set fileName      [lindex $parsedUrl 3]
        if {$siteUrl(prot)=="ftp"} {
            set options(checkSize) 0
        }
        set siteUrl(base) $siteUrl(dir)
        set url $relocated(url)
    }

    return $url
}

###############################################################################
# RelativePath
#    The function returns the relative path from the current directory to the
#    root directory of the download.
#
# Parameter:
#    The directory of the file which is going to be processed for links.
#
# Returns:
#    The relative path.
###############################################################################
proc RelativePath {path_absoluto} {
    global directories

    set patron (?:$directories(base)/)(.*)
    regexp $patron $path_absoluto nada path
    if {![info exists path]} return

    set cuenta [regsub -all {/} $path {} ignorar]

    for {set i 0;set rel ""} {$i<$cuenta} {incr i} {
        append rel ../
    }
    return $rel
}

###############################################################################
# Preprocessing
#    Reads the Web page passed as a parameter and proccess it to extract
#    all links to local files and images it has
#
# Parameters:
#    url: url of the page where are going to process for links
###############################################################################
proc Preprocessing {url} {
    global directories filesDone getleftState

    if [string match $url ""] return

    set file [UrlToFile $url]

    if {$::DEBUG==1} {
#        puts "Fichero a preprocesar: $file"
#        puts "La pagina de referencia: $url"
    }

    set filesDone($file) 1
    set directories(local) [file dirname $file]

    if {$getleftState(filesChosen)==1} {
        Ventana::ShowWindow  process $url
        Ventana::FileStrings $url
    }

    if {[file exists $file.orig]} {
        set fileName $file.orig
    } else {
        set fileName $file
    }

    HtmlParser::Parsing $fileName
    HtmlParser::FilterLinks $url

    if {![file exists $file.orig]} {
        ChangePage $url
    }

    HtmlParser::AbsoluteLinks $url

    return
}

###############################################################################
# TidyNames
#    Removes from the name and path of files things like '?' '~' '+' '-'
#
# Returns
#    The filename without those characters.
###############################################################################
proc TidyNames {nombre} {

    regsub -all {~}  $nombre {} nombre
    regsub -all {\*} $nombre {} nombre
    if {[regexp {(?:^.:)(.*)} $nombre nada filename]} {
        regsub -all {:}  $filename {} filename
	  set nombre $filename
    } else {
        regsub -all {:} $nombre {} nombre
    }
    if {[regexp {([^\?]+)(?:\?)(.*)} $nombre nada uno dos]} {
        regsub -all {\?} $dos {} dos
        regsub -all {\+} $dos {} dos
        regsub -all {/}  $dos {} dos
        regsub -all {\\} $dos {} dos
        set nombre $uno$dos
    }
    return $nombre
}

###############################################################################
# Chosen
#    Selects and deselects the chosen files.
#
# Parameters:
#    patron: regular expresion that identifies the files to (de)select.
#    cual: '1' to select, '0' to deselect.
#    map: 1 in case we are mapping the site.
###############################################################################
proc Chosen {patron cual {map 0} } {
    global linksOk siteMap

    for {set i 1} {$i<$HtmlParser::nLinks} {incr i} {
        if {$map==0} {
            if {[regexp -nocase $patron $HtmlParser::linkFile($i)]} {
                set linksOk($i) $cual
            }
        } else {
            if {[regexp -nocase $patron $siteMap($i,1)]} {
                set linksOk($i) $cual
            }
        }
    }
    return
}

###############################################################################
# ChooseGroup
#    Activates or deactivates the links that are either before or after the
#    link where the user right-clicked.
#
# Parameters:
#    which: 'pre' the links before the click.
#           'fol' the links after.
#    action: '0' deactivates the links
#            '1' activates them.
#    path: Path of the text widget where the links are.
###############################################################################
proc ChooseGroup {which action path} {

    regexp {(.*)(\.)} [$path index insert] nada line
    if {$line>=$HtmlParser::nLinks} return

    if {$which=="pre"} {
        set first 1
        set last  $line
    } else {
        set first $line
        set last  [expr $HtmlParser::nLinks-1]
    }
    for {set i $first} {$i<=$last} {incr i} {

        if {$action==0} {
            $path.enlaces$i deselect
        } else {
            $path.enlaces$i select
        }
    }

    return
}

###############################################################################
# ChangeLinksToChoose
#    If the user changes the downloading options in the 'Choose Files' dialog
#    this procedure takes care of displaying the new links to choose from.
#
# Parameter:
#    referer: url of the page from wherer the links where taken.
#    text: The path of the text widget
#    map:  '1' if we are getting the site map
###############################################################################
proc ChangeLinksToChoose {referer text map} {
    global filesDone

    $text configure -cursor watch
    $text configure -state  normal
    $text delete 1.0 end

    set file [lindex [array get filesDone] 0]
    catch {unset filesDone}
    set refererFile [UrlToFile $referer]
    catch {file rename -force $refererFile.orig $refererFile}
    Preprocessing $referer
    LinksToChoose $text $map

    $text configure -state  disabled
    $text configure -cursor left_ptr

    return
}

###############################################################################
# LinksToChoose
#    Inserts the links in the 'Choose files' directory.
#
# Parameter
#    text: path of the text widget.
#    map: '1' if we are getting the site map
###############################################################################
proc LinksToChoose {text map} {
    global siteIndex siteMap linksOk

    set color [$text cget -background]

    if {$map==1} {
        set HtmlParser::nLinks $siteIndex
    }
puts "$HtmlParser::nLinks"
    for {set i 1;set pre ""} {$i<$HtmlParser::nLinks} {incr i} {
        checkbutton $text.enlaces$i -variable linksOk($i) -background $color
        set linksOk($i) 1

        $text window create $i.0 -window $text.enlaces$i
        if {$map==0} {
            set description [HtmlParser::TidyDescription \
                    $HtmlParser::linkDescription($i)]
        } else {
            for {set j 0 ; set spc "   " ; set pre ""} {$j<$siteMap($i,0)} \
                    {incr j} {
                append pre $spc
            }
            set description [HtmlParser::TidyDescription $siteMap($i,2)]
        }
        if {[regexp {^[\ \t]*$} $description]} {
            if {$map==0} {
                set description $HtmlParser::linkFile($i)
            } else {
                set description $siteMap($i,1)
            }
        }
        $text.enlaces$i configure -text $pre$description

        $text insert insert \n
        bind $text.enlaces$i <Button-3> "
            regexp {(?:enlaces)(.*)} %W nada line
            $text mark set insert \$line.0
            tk_popup .elegir.menu %X %Y
        "
    }

    return
}

###############################################################################
# ChooseDownloads
#    Shows a dialog with all the links in the entered Url so you can choose
#    which ones will be followed and which will be ignored.
#
# Parameter
#    referer: url of the page from which the links where taken.
#    map: '1' if we are getting the site map, defaults to '0'
#
###############################################################################
proc ChooseDownloads {referer {map 0} } {
    global labelButtons labelTitles labelDialogs labelMenus
    global dirGetleft options optionsTmp done

    if {$HtmlParser::nLinks==1} "return 0"
    set elegir  [toplevel .elegir]
    set marco   [frame $elegir.marco]
    set checks  [frame $elegir.checks]
    set botones [frame $elegir.botones]

    wm title $elegir $labelTitles(choose)

    set texto [text $marco.texto -yscrollcommand [list $marco.yscroll set]    \
            -xscrollcommand [list $marco.xscroll set] -wrap none              \
            -cursor left_ptr]
    set aceptar [button $botones.aceptar   -textvariable labelButtons(ok)     \
            -width 9 -command "set done 1"]
    set cancelar [button $botones.cancelar -textvariable labelButtons(cancel) \
            -width 9 -command "set done 0" ]

    set mas   [image create bitmap  -file [file join $dirGetleft(images) mas]]
    set menos [image create bitmap  -file [file join $dirGetleft(images) menos]]
    set lhtml [label  $checks.lhtml -text "  $labelDialogs(html)"]
    set htmla [button $checks.htmla -image $mas   -width 8 -height 8 \
            -command "Chosen {(html$)|(htm$)|(/$)} 1 $map"]
    set htmle [button $checks.htmle -image $menos -width 8 -height 8 \
            -command "Chosen {(html$)|(htm$)|(/$)} 0 $map"]
    set lima  [label $checks.lima   -text "  $labelDialogs(images)"]
    set imaa  [button $checks.imaa  -image $mas   -width 8 -height 8 \
            -command "Chosen {(gif$)|(jpg$)|(bmp$)|(xbm$)|(tiff$)|(png$)} 1 $map"]
    set imae  [button $checks.imae  -image $menos -width 8 -height 8 \
            -command "Chosen {(gif$)|(jpg$)|(bmp$)|(xbm$)|(tiff$)} 0 $map"]
    set lcom  [label $checks.lcom   -text "  $labelDialogs(archives)"]
    set coma  [button $checks.cona  -image $mas   -width 8 -height 8 \
            -command "Chosen {(tar$)|(gz$)|(z$)|(zip$)} 1 $map"]
    set come  [button $checks.cone  -image $menos -width 8 -height 8 \
            -command "Chosen {(tar$)|(gz$)|(z$)|(zip$)} 0 $map"]
    set ltodo [label $checks.ltodo  -text "  $labelDialogs(all)"]
    set todoa [button $checks.todoa -image $mas   -width 8 -height 8 \
            -command "Chosen {.} 1 $map"]
    set todoe [button $checks.todoe -image $menos -width 8 -height 8 \
            -command "Chosen {.} 0 $map"]

    LinksToChoose $texto $map

    set barray [scrollbar $marco.yscroll -orient vertical \
        -command [list $texto yview]]
    set barrax [scrollbar $marco.xscroll -orient horizontal  \
        -command [list $texto xview]]

    set menu [menu $elegir.menu -tearoff 0]
    $menu add cascade -label "$labelDialogs(html)"      -menu $menu.html
    $menu add separator
    $menu add cascade -label "$labelDialogs(images)"    -menu $menu.images
    $menu add separator
    $menu add cascade -label "$labelDialogs(archives)"  -menu $menu.archives
    $menu add separator
    $menu add cascade -label "$labelDialogs(all)" -menu $menu.all
    $menu add separator
    $menu add cascade -label "$labelDialogs(previous)"  -menu $menu.previous
    $menu add cascade -label "$labelDialogs(following)" -menu $menu.following
    $menu add separator
    $menu add cascade -label "$labelMenus(options)" -menu  $menu.options

    menu $menu.html      -tearoff 0
    menu $menu.images    -tearoff 0
    menu $menu.archives  -tearoff 0
    menu $menu.all       -tearoff 0
    menu $menu.previous  -tearoff 0
    menu $menu.following -tearoff 0
    menu $menu.options   -tearoff 0

    $menu.html      add command -label "$labelDialogs(activate)"   \
            -command "$htmla invoke"
    $menu.html      add command -label "$labelDialogs(deactivate)" \
            -command "$htmle invoke"
    $menu.images    add command -label "$labelDialogs(activate)"   \
            -command "$imaa  invoke"
    $menu.images    add command -label "$labelDialogs(deactivate)" \
            -command "$imae  invoke"
    $menu.archives  add command -label "$labelDialogs(activate)"   \
            -command "$coma  invoke"
    $menu.archives  add command -label "$labelDialogs(deactivate)" \
            -command "$come  invoke"
    $menu.all       add command -label "$labelDialogs(activate)"   \
            -command "$todoa invoke"
    $menu.all       add command -label "$labelDialogs(deactivate)" \
            -command "$todoe invoke"
    $menu.previous  add command -label "$labelDialogs(activate)"   \
            -command "ChooseGroup pre 1 $texto"
    $menu.previous  add command -label "$labelDialogs(deactivate)" \
            -command "ChooseGroup pre 0 $texto"
    $menu.following add command -label "$labelDialogs(activate)"   \
            -command "ChooseGroup fol 1 $texto"
    $menu.following add command -label "$labelDialogs(deactivate)" \
            -command "ChooseGroup fol 0 $texto"

    $menu.options add cascade -menu $menu.options.directorios   \
            -label $labelMenus(uplinks)
    $menu.options add cascade -menu $menu.options.levels        \
            -label $labelMenus(levels)
    $menu.options add cascade -menu $menu.options.exLevels        \
            -label $labelMenus(external)
    $menu.options add cascade -menu $menu.options.filters       \
            -label $labelMenus(fileFilter)
    $menu.options add check   -variable options(update)         \
            -label $labelMenus(update)   -onvalue 1 -offvalue 0
    $menu.options add check   -variable options(cgi)            \
            -label $labelMenus(cgi)      -onvalue 1 -offvalue 0 \
            -command "ChangeLinksToChoose $referer $texto $map"

    set menuDir [menu $menu.options.directorios -tearoff 0     \
            -postcommand {set optionsTmp(dir) $options(dir)}]
    $menuDir add radio -variable options(dir)  -value 1        \
           -label $labelMenus(linksFollow) -command "
                if {\$optionsTmp(dir)!=1} {
                    ChangeLinksToChoose $referer $texto $map
                }
            "
    $menuDir add radio -variable options(dir)  -value 0 \
            -label $labelMenus(linksIgnore) -command "
                if {\$optionsTmp(dir)!=0} {
                    ChangeLinksToChoose $referer $texto $map
                }
            "
    set menuLev [menu $menu.options.levels -tearoff 0]
    $menuLev add radio -variable options(levels) -value -1 \
            -label $labelMenus(noLimit)
    for {set i 0} {$i<6} {incr i} {
        $menuLev add radio -label $i -variable options(levels) -value $i
    }
    set menuEx [menu $menu.options.exLevels -tearoff 0]
    $menuEx add radio -variable options(exLevels)    \
            -label $labelMenus(linksIgnore) -value 0 \
            -command "
                set options(external) 0
                ChangeLinksToChoose $referer $texto $map
            "
    for {set i 1} {$i<3} {incr i} {
        $menuEx add radio -label $i -variable options(exLevels) \
            -value $i -command "
                set options(external) 1
                ChangeLinksToChoose $referer $texto $map
            "
    }


    set menuFil [menu $menu.options.filters -tearoff 0]
    $menuFil add check -variable options(onlyHtml) -onvalue 1 -offvalue 0 \
            -label $labelMenus(onlyHtml)
    $menuFil add cascade -menu $menuFil.images \
            -label $labelMenus(images)
    $menuFil add command -command Herramientas::FilterFiles \
            -label $labelMenus(chooseFilter)

    set menuFilImage [menu $menuFil.images -tearoff 0]
    $menuFilImage add check -variable options(onlyThumb) -onvalue 1 -offvalue 0 \
            -label $labelMenus(onlyThumb) -command "
                set options(noThumb) 0
                ChangeLinksToChoose $referer $texto $map
            "
    $menuFilImage add check -variable options(noThumb)   -onvalue 1 -offvalue 0 \
            -label $labelMenus(noThumb) -command "
                set options(onlyThumb) 0
                ChangeLinksToChoose $referer $texto $map
            "

    bind $elegir   <KeyPress-Prior> "$texto yview scroll -1 pages"
    bind $elegir   <KeyPress-Next>  "$texto yview scroll  1 pages"
    bind $elegir   <Key-Up>         "$texto yview scroll -1 units"
    bind $elegir   <Key-Down>       "$texto yview scroll  1 units"
    bind $elegir   <Escape>         "$cancelar invoke"
    bind $texto    <Button-3>       "
        tkTextButton1 %W %x %y
        tk_popup $menu %X %Y
    "
    bind $texto    <Button-1>       {
        %W tag remove sel 0.0 end
    }

    grid $marco -sticky news
    grid $texto $barray -sticky news
    grid $barrax -sticky ew
    grid $checks -sticky ew
    grid $lhtml $htmla $htmle $lima $imaa $imae $lcom $coma $come \
        $ltodo $todoa $todoe
    grid $botones -sticky ew -ipady 4
    grid $aceptar $cancelar

    grid rowconfigure    $elegir 0 -weight 1
    grid columnconfigure $elegir 0 -weight 1
    grid rowconfigure    $marco  0 -weight 1
    grid columnconfigure $marco  0 -weight 1

    $texto configure -state disabled

    wm protocol $elegir WM_DELETE_WINDOW "
            set done 0
            destroy $elegir"

    tkwait visibility $elegir

    grab $elegir
    tkwait variable done
    grab release $elegir
    destroy $elegir

    return $done
}
###############################################################################
# UrlToFile {url} {
#    Given an Url this procedure will return the file in which it will be
#    saved.
#
# Parameters
#    url: the url to process.
#
# Returns:
#    The file in which it will be saved complete with full path.
###############################################################################
proc UrlToFile {url} {
    global directories siteUrl

    set parsedUrl [HtmlParser::ParseUrl $url]
    set prot      [lindex $parsedUrl 0]
    set domain    [lindex $parsedUrl 1]
    set dir       [lindex $parsedUrl 2]
    set file      [lindex $parsedUrl 3]

    if {$file==""} {
        if {$prot=="ftp"} {
            set file index.txt
        } else {
            set file index.html
        }
    }
    if {$domain!=$siteUrl(www)} {
        set fileName /${domain}$dir/$file
    } else {
        set fileName $dir/$file
    }
    set fileName [TidyNames $fileName]

    if [regexp {(?:^/)(.*)} $fileName nada fileName] {
        set directoryRoot $directories(base)
    } else {
        set directoryRoot $directories(local)
    }

    set fileName [file join $directoryRoot $fileName]
    if {[file exists $fileName]} {
        if {([file size $fileName]==0)&&([file exists $fileName.html])} {
            set fileName $fileName.html
        }
    }
    return $fileName
}

###############################################################################
# CheckDir
#    Creates, if needed, the directory in which the link will be saved.
#
# Parameter:
#    fileName: file in which the url is going to be downloaded.
#
# Returns:
#    1: In case of error
###############################################################################
proc CheckDir {fileName} {
    global labelTitles labelMessages

    set newDirectory [file dirname $fileName]
    if [file exists $newDirectory] {
        if {![file isdirectory $newDirectory]} {
            tk_messageBox -type ok -icon error -title $lableTitles(error) \
                    -message "File names are screwed: $fileName"
            return 1
        }
     } else {
        if [catch {file mkdir $newDirectory} error] {
            tk_messageBox -type ok -title $labelTitles(error) -icon error \
                    -message $error
            return 1
        }
     }

     return
}

###############################################################################
# Download
#    Downloads a link
#
# Parameter:
#    enlace:   Link to download
#    madre:    referrer page
#    sufix:    Whether the file ends in 'htm?' (1) or not (0)
#
# Returns:
#    - '4' in case of a relocation and a HTML page
#    - '3' in case of a relocation and not a HTML page
#    - '2' in case of error
#    - '1' if it is a HTML page
#    - '0' if it is not
###############################################################################
proc Download {enlace madre sufix} {
    global siteUrl options errorCode getleftState
    global Ventana::Rizo::meta
    global directories relocated
    global labelMessages

    set fileName [UrlToFile $enlace]
    if {([file exists $fileName])&&($options(update)==0)} {
        if {$sufix==0} {
            if {([file isdirectory $fileName])&&(![regexp {/$} $fileName])} {
                if [file exists $fileName/index.html] {
                    set relocated(url) $enlace/index.html
                    return 4
                }
            } elseif [file exists $fileName.html] {
                return 1
            } else {
                return 0
            }
        } else {
            return 1
        }
    }
    for {set relocated(ok) 0} {1==1} {} {
        Ventana::HeadDownloading $enlace $madre
        if {$::Ventana::Rizo::curlError!=0} {
            ::Herramientas::ErrorLog $fileName $enlace
            return 2
        }
        if {$meta(relocate)!=""} {
            if {$::DEBUG==1} {
                puts "La relocalizacin: $meta(relocate)"
            }
            set relocated(url) $meta(relocate)
            if {([regexp {\?} $meta(relocate)])} {
                return 0
            }
            set parsedUrl [HtmlParser::ParseUrl $meta(relocate)]
            if {$parsedUrl==1} {
                return 2
            }
            set direccionWWW [lindex $parsedUrl 1]
            set direccionDir [lindex $parsedUrl 2]
            set newFileName  [lindex $parsedUrl 3]
            if {$direccionWWW!=$siteUrl(www)} {
                set Ventana::Rizo::errorMessage "Relocated link\n$meta(relocate)"
                return 2
            }
            if {$newFileName==""} {
                set newFileName index.html
            }
            set newFileName [TidyNames [file join $direccionDir $newFileName]]
            set fileName [UrlToFile $meta(relocate)]
            set enlace   $meta(relocate)
            set relocated(ok) 1
            if {[file exists $fileName]} {
                if {([regexp -nocase {\.html?$} $fileName])||([file exists $fileName.html])} {
                    set isHtml 1
                } else {
                    set isHtml 0
                }
                return [expr 3 + $isHtml]
            }
        } else {
            break
        }
    }
    if {$getleftState(filesChosen)==0} {
        if {$meta(content)!="text/html"} {
            set fileName $directories(base)/[file tail $fileName]
        }
    }

    set returnCode [CheckDir $fileName]
    if {$returnCode==1} {
        return 2
    }
    regexp {(.*)(://)} $enlace nada prot
    if {($prot!="ftp")&&([info exists meta(totalBytes)])\
            &&($meta(totalBytes)==0)} {
        catch {error "Sin contenido" GETLEFT \
                "Getleft \"$labelDialogs(noContent)\" 002"}
        ::Herramientas::ErrorLog $fileName $enlace
        return 2
    }
    if {($options(update)==1)&&([file exists $fileName])} {
        if [file exists $fileName.orig] {
            file stat $fileName.orig oldFile
        } else {
            file stat $fileName oldFile
        }
        if {[catch "set meta(lastModified)"]} {
            set lastChange 0
        } else {
            set lastChange [clock scan $meta(lastModified)]
        }
        if {$lastChange<$oldFile(mtime)} {
            if {$sufix==0} {
                if [file exists $fileName.html] {
                   return 1
                } else {
                   return 0
                }
            }
            return 1
        }
    }

    if [string match $Ventana::Rizo::meta(content) "text/html"] {
        set isHtml 1
    } else {
        set isHtml 0
        if {$options(onlyHtml)==1} {
            return 0
        }
    }

    set Ventana::Rizo::curlReport(parar)    0
    set Ventana::Rizo::curlReport(nextFile) 0
    Ventana::FileDownloading $fileName.$labelMessages(downloading) \
            $enlace $madre
    tkwait variable Ventana::Rizo::curlReport(nextFile)
    if {$errorCode!=""} {
global errorInfo
puts "Hubo un error: $errorCode: $errorInfo"
        ::Herramientas::ErrorLog $fileName $enlace
        return 2
    }

    catch {file rename -force $fileName.$labelMessages(downloading) $fileName}
    if {$isHtml==1} {
        file delete $fileName.orig
    }

    if {$relocated(ok)==1} {
        return [expr 3 + $isHtml ]
    }

    return $isHtml
}

###############################################################################
# ChangePage
#    Changes a html page, so that there is consistency with the local
#    directories. After this procedure is run through a page all it's links
#    should be between double qoutes (") and be relative to the the current
#    directory.
#
# Parameters
#    url: The url of the page about to be changed.
###############################################################################
proc ChangePage {url} {
    global siteUrl
    global directories

    set fileName [UrlToFile $url]
    set relativePath [RelativePath $fileName]
    if {$relativePath==""} {
        set relativePath ./
    }

    set script [open "$directories(base)/script.sed" w+]

    for {set i 1} {$i<$HtmlParser::nLinks} {incr i} {
        set link $HtmlParser::linkFile($i)
        set newLink [HtmlParser::CompleteUrl $link $url]

        # First we prepare the link taken from the page
        regsub {^\.\.}    $link {\\.\\.} link
        regsub {^\.}      $link {\\.}    link
        regsub -all {\*}  $link {\\*}    link

        # Now deal with the complete link to get what will bve put in the
        # changed page

        regexp {(?::/)(.*)} $newLink nada newLink
        regexp "(?:/$siteUrl(www))(.*)" $newLink nada newLink
        regsub -all {~|\?|\*} $newLink {} newLink
        regsub -all {&} $newLink {\\\&} newLink

        puts $script "s#=\\(\'\\|\\\"\\)\\($link\\)\\(\\\"\\|\'\\)#=\"$newLink\"#g"
        puts $script "s#=\\($link\\)\\( \\|>\\)#=\"$newLink\"\\2#g"
    }

    puts $script "s#=\"/#=\"$relativePath#g"
    close $script

    file copy $fileName $fileName.orig
    InvokeSed $fileName

    return
}

###############################################################################
# InvokeSed
#    Calls 'sed' to do the dirty job of changing the web pages.
#
# Parameters:
#    pagina: file containing the page to be changed.
#    expr:   (opcional), regular expression to be passed to 'sed'. If not given
#            'sed' will use the file 'script.sed'.
###############################################################################
proc InvokeSed {pagina {expr {}}} {
    global directories
    global labelTitles

    set sedScript [file join $directories(base) script.sed]
    set sedTemp   [file join $directories(base) sedTemp]
    if {$expr!=""} {
        set file [open "$sedScript" w+]
        puts $file $expr
        close $file
    } else {
        if {![file exists "$sedScript"]} {
            return
        }
    }

    if {[file exists $pagina.html]} {
        set pagina $pagina.html
    }
    if [catch {exec sed -f "$sedScript"  $pagina > "$sedTemp"} error] {
        tk_messageBox -title $labelTitles(error) -message $error -type ok \
                -icon error
    } else {
        file rename -force "$sedTemp" $pagina
    }
    file delete "$sedScript"

    return
}

###############################################################################
# ChangeLinkList
#    Changes or removes one link form the list that contains all the links of
#    one page
#
# Parameters:
#    linkList: list with the links
#    oldLink: link to change or remove
#    newLink: the new links, defaults to none to remove the old one.
#
# Returns
#    The new list
###############################################################################
proc ChangeLinkList {linkList oldLink {newLink {} } } {

    set index   [lsearch -exact $linkList $oldLink]
    if {$newLink!=""} {
        set linkList [lreplace $linkList $index $index $newLink]
    } else {
        set linkList [lreplace $linkList $index $index]
    }

    return $linkList
}

###############################################################################
# SaveSedExp
#    Saves one line of the sed script that will change the links to active
#    pages to the resulting page
#
# Parameter
#    exp: sed expresion to be saved
###############################################################################
proc SaveSedExp {exp} {
    global directories

    set handle [open [file join $directories(base) script.sed] a+]
    puts $handle $exp
    close $handle

    return
}

###############################################################################
# ControlDownloading
#    Send 'Downloading' all the links one by one.
#
# Parameters:
#    urlList: list with the links to download.
#    referer: html page from which the links come from.
#    level: current level of recursion for internal links.
#    externalLevel: current level of recursion for external link, defaults to 0
###############################################################################
proc ControlDownloading {urlList referer level {externalLevel 0}} {
    global siteUrl directories relocated
    global options filesDone labelTitles labelMessages

    if {$::DEBUG==1} {
#        puts "En control"
#        puts "ListaEnlaces: $urlList"
#        puts "Pagina madre: $referer"
    }

    if {[llength $urlList]==0} return

    set cosa ""
    set invokeSed 0

    set j 0
    foreach urlToDownload $urlList {
        update
        incr j
        if {$options(onlyHtml)==1} {
            if [regexp -nocase {(jpg)|(jpeg)|(gif)|(gz)|(tar)|(zip)|(exe)|(ps)|(pdf)|(txt)|(text)|(avi)|(ram)|(wav)} \
                    [file extension $urlToDownload] ] {
                set isHtml($j) 0
                continue
            }
        }
        if {($options(filter)!="")&&([regexp -nocase "$options(filter)" \
                $urlToDownload])} {
            set isHtml($j) 0
            continue
        }

        set fileName [UrlToFile $urlToDownload]

        if [regexp {/$} $urlToDownload] {
            set isHtml($j) 1
            set urlList [ChangeLinkList $urlList $urlToDownload \
                    ${urlToDownload}index.html]
            regexp "(?:$directories(base))(.*/)(index.html)" \
                    $fileName nada tmpFileName
            regsub -all {&} $tmpFileName {\\\&} tmpAmper
            SaveSedExp "s#\\($tmpFileName\\)\\\"#${tmpAmper}index.html\"#g"
            set invokeSed 1
        }

        if [regexp -nocase {\.html?} [file extension $fileName]] {
            set sufijo 1
        } else {
            set sufijo 0
        }

        set isHtml($j) [Download $urlToDownload $referer $sufijo]

        switch -regexp -- $isHtml($j) {
            1 {
                if ($sufijo==0) {
                    catch {file rename $fileName $fileName.html}
                    exec touch $fileName
                    set urlList [ChangeLinkList $urlList \
                            $urlToDownload $urlToDownload.html]
                    set urlToDownload [file tail [TidyNames $urlToDownload]]
                    regsub -all {&} $urlToDownload {\\\&} tmp
                    SaveSedExp "s#\\($urlToDownload\\)\\\"#$tmp.html\"#g"
                    set invokeSed 1
                }
            }
            2 {
                if {[regexp {/$} $urlToDownload]} {
                    set urlList [ChangeLinkList $urlList \
                            ${urlToDownload}index.html]
                } else {
                    set urlList [ChangeLinkList $urlList $urlToDownload]
                }
                incr j -1
            }
            {3|4} {
                set oldLink [UrlToFile $urlToDownload]
                set newLink [UrlToFile $relocated(url)]
                set pattern "(?:$directories(base))(.*)"
                regexp $pattern $oldLink nada oldLink
                regexp $pattern $newLink nada newLink
                set sufix [regexp -nocase {\.html?} \
                        [file extension $newLink]]
                if {$isHtml($j)==4} {
                    set isHtml($j) 1
                    if {$sufix==0} {
                        set newLink $newLink.html
                    }
                }
                regsub -all {&} $newLink {\\\&} newLink
                SaveSedExp "s#\\($oldLink\\)\\\"#$newLink\"#g"
                set invokeSed 1
                set urlList [ChangeLinkList $urlList \
                        $urlToDownload $relocated(url)]
            }
        }
        if {$options(stopFile)==1} {
            break
        }
        if {$options(pauseFile)==1} {
            Ventana::Pause file
        }
    }
    if {$options(pausePage)==1} {
        Ventana::Pause page
    }

    if {$invokeSed==1} {
        InvokeSed [UrlToFile $referer]
    }

    if {$options(stopFile)==1} {
        return
    }

    set j 1
    foreach url $urlList {
        if {$isHtml($j)==1} {
            set fileToProcess   [UrlToFile $url]
            set parsedUrl       [HtmlParser::ParseUrl $url]
            if {![info exists filesDone($fileToProcess)]} {
                set tmpUrl        $siteUrl(dir)
                set tmpLocal      $directories(local)
                set tmpprot       $siteUrl(prot)
                set domain        [lindex $parsedUrl 1]
                set siteUrl(dir)  [lindex $parsedUrl 2]
                set siteUrl(prot) [lindex $parsedUrl 0]
                if {$domain!=$siteUrl(www)} {
                    set nextExternalLevel [expr $externalLevel + 1]
                } else {
                    set nextExternalLevel 0
                }
                if {($level<$options(levels))||($options(levels)==-1)} {
                    if {$nextExternalLevel<=$options(exLevels)} {
                        Preprocessing $url
                        if {$HtmlParser::nLinks!=1} {
                            set nextLevel [expr $level +1]
                            ControlDownloading [PrepareDownloading $nextLevel] \
                                    $url $nextLevel $nextExternalLevel
                            if {($options(stopPage)==1)||($options(stopFile)==1)} {
                                break
                            }
                        }
                    }
                }
                set siteUrl(prot)      $tmpprot
                set siteUrl(dir)       $tmpUrl
                set directories(local) $tmpLocal
            }
        }
        incr j
    }
    return
}

###############################################################################
# PrepareDownloading
#    Prepares the list of files to download
#
# Parameter
#    nivelActual: level of recursion
###############################################################################
proc PrepareDownloading {nivelActual} {
    global linksOk siteIndex siteMap options getleftState

    for {set i 1 ; set urlList ""} {$i<$HtmlParser::nLinks} {incr i} {
        if {($getleftState(filesChosen)==1)||($linksOk($i)==1)} {
            lappend urlList $HtmlParser::linkFile($i)
        }
    }

    if {$options(map)==1} {
        for {set i 1} {$i<$HtmlParser::nLinks} {incr i} {
            if {![info exists siteMap($HtmlParser::linkFile($i))]} {
                set siteMap($siteIndex,0) $nivelActual
                set siteMap($siteIndex,1) $HtmlParser::linkFile($i)
                set siteMap($siteIndex,2) $HtmlParser::linkDescription($i)
                set siteMap($HtmlParser::linkFile($i)) 1
                incr siteIndex
            }
        }
    }
    if {$getleftState(filesChosen)==0} {
        set getleftState(filesChosen) 1
    }

    return $urlList
}

###############################################################################
# ReadSel
#    Reads the selection from the clipboard
#
# Returns
#    Whatever was in the clipboard
###############################################################################
proc ReadSel {} {
    if {[catch {selection get -selection CLIPBOARD} cosa]} {
        if {[catch {selection get} cosa]} return
    }
    return $cosa
}

###############################################################################
# EnterUrl
#    Creates a dialog box where you can write or paste the desired url.
#
# Returns:
#    The url entered by the user.
###############################################################################
proc EnterUrl {} {
    global filesDone options urlHistory done
    global directories urlIndex getleftState
    global labelButtons labelTitles labelMessages

    if [winfo exists .introducir] {
        return
    }

    set coord(x) [winfo rootx .]
    set coord(y) [winfo rooty .]

    set intro [toplevel .introducir]
    wm title $intro $labelTitles(enterUrl)
    wm resizable $intro 0 0
    wm geometry $intro +[expr $coord(x)+100]+[expr $coord(y)+15]

    set done                      0
    set options(stopPage)         0
    set options(stopFile)         0
    set options(pausePage)        0
    set options(pauseFile)        0
    set getleftState(filesChosen) 0
    set getleftState(errorFound)  0

    set marcoEx [frame $intro.marcoEx]
    set marco   [frame $marcoEx.marco -bd 2 -relief groove]

    if {[catch {llength $urlHistory} listItems]} {
        set listItems 0
	set urlHistory ""
    }

    Ccombinado::ComboBox $marco.pagina $urlHistory -default [ReadSel] \
            -erasable 1

    set pagina $marco.pagina
    set botones  [frame $marcoEx.botones]
    set empezar  [button $intro.empezar  -textvariable labelButtons(start) \
            -width 8 -command {set done 1}]
    set cancelar [button $intro.cancelar -textvariable labelButtons(cancel) \
            -width 8 -command {set done 0}]

    bind $pagina.entry <Return>   "$empezar invoke"
    bind $pagina.entry <KP_Enter> "$empezar invoke"
    bind $pagina.entry <Button-3> "
        $pagina.entry insert insert \[ReadSel\]
    "
    bind $intro    <Escape>   "$cancelar invoke"

    pack $pagina   -padx 20 -pady 20
    pack $marcoEx  -ipadx 10 -ipady 5
    pack $marco    -side bottom
    pack $cancelar -side right -padx 9 -pady 5
    pack $empezar  -side right -pady 5
    pack $botones  -side right

    focus $pagina.entry
    tkwait variable done

    if {$done==0} {
        destroy $intro
        return
    }

    set newListItems [llength $Ccombinado::cbArgs(items)]
    if {$listItems!=$newListItems} {
        set urlHistory $Ccombinado::cbArgs(items)
        set urlIndex   $newListItems
        SaveConfig
    }

    set requestedUrl [$pagina.entry get]
    destroy $intro

    if {$requestedUrl==""} {
        return
    }
    if {![EnHistorial $requestedUrl]} {
        if {[llength $urlHistory]<10} {
            lappend urlHistory $requestedUrl
        } else {
            set urlHistory [lreplace $urlHistory $urlIndex $urlIndex \
                    $requestedUrl]
        }
        if {$urlIndex==9} {
            set urlIndex 0
        } else {
            incr urlIndex
        }
        SaveConfig
    }
    catch "unset filesDone"
    set getleftState(downloading) 1

    set requestedUrl [InitializeDownload $requestedUrl]
    if {$requestedUrl==""} {
        set getleftState(downloading) 0
        return
    }
    set localFileName [UrlToFile $requestedUrl]
    if {($options(levels)==0)||(![regexp -nocase {\.html?$} $localFileName])} {
        tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                -message $labelMessages(theEnd)
        set getleftState(downloading) 0
        return $requestedUrl
    }

    Preprocessing $requestedUrl
    if {$HtmlParser::nLinks==1} {
        tk_messageBox -icon info -title $labelTitles(noLinks) -type ok \
                -message $labelMessages(noLinks)
        set getleftState(downloading) 0
        return
    }
    set chosen [ChooseDownloads $requestedUrl]
    if {$chosen==1} {
        ControlDownloading [PrepareDownloading 0] $requestedUrl 1

        if [catch {wm withdraw .bajando} error] {
            if {$::DEBUG==1} {
                tk_messageBox -title Error -icon error -message $error -type ok
            }
        }
        update
        if {$options(map)==0} {
            tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                    -message $labelMessages(theEnd)
            if {$getleftState(errorFound)==1} {
                ::Herramientas::ShowErrorLog
            }
        } else {
            return $requestedUrl
        }
        set options(stopFile)           0
        set options(stopPage)           0
        set options(pauseFile)          0
        set options(pausePage)          0
        set getleftState(filesChosen)   0
    }
    catch "unset filesDone"
    set getleftState(downloading) 0

    return $requestedUrl
}

###############################################################################
# GetSiteMap
#    Downloads the html pages in a site, to extract the site map from them
###############################################################################
proc GetSiteMap {} {
    global siteIndex siteMap options getleftState
    global siteUrl linksOk labelMessages labelTitles

    set options(map)       1
    set options(onlyHtml)  1
    set siteIndex          1
    catch {unset siteMap}

    set requestedUrl [EnterUrl]

    if {$requestedUrl==""} {
        return
    }

    set options(onlyHtml)  0
    set options(map)       0

    if {$siteIndex==1} {
        return
    }

    set done [ChooseDownloads $requestedUrl 1]
    set tmp $options(levels)
    set $options(levels) 0

    if {$done==1} {
        for {set i 1} {$i<$siteIndex} {incr i} {
            if {$linksOk($i)==1} {
                set enlace $siteMap($i,1)
                set filename [UrlToFile $enlace]
                if {[file exists $filename]} {
                    continue
                } else {
                    set isHtml [regexp -nocase {\.html?} $filename]
                    Download $enlace "-" $isHtml
                }
                if {$options(stopFile)==1} {
                    break
                }
                if {$options(pauseFile)==1} {
                    Ventana::Pause file
                }
            }
        }
        tk_messageBox -icon info -title $labelTitles(theEnd) -type ok \
                -message $labelMessages(theEnd)
    }
    unset siteMap
    unset linksOk
    unset HtmlParser::linkFile
    unset HtmlParser::linkDescription
    unset HtmlParser::linkType

    if {$getleftState(errorFound)==1} {
        ::Herramientas::ShowErrorLog
    }
    set options(levels)             $tmp
    set options(stopFile)           0
    set options(stopPage)           0
    set options(pauseFile)          0
    set options(pausePage)          0
    set getleftState(filesChosen)   0

    return
}

###############################################################################
# ReadConfig
#    Reads the configuration files
#
# Parameter:
#    Configuration file to read, if not present, the user configuration  file
#    will be read.
###############################################################################
proc ReadConfig { {confFile {} } } {
    global urlHistory urlIndex dirGetleft options
    global env tcl_platform


    if {$confFile!=""} {
        set ficherorc $confFile
    } else {
        if [regexp {Windows} $tcl_platform(os)] {
            set ficherorc [file join "$dirGetleft(main)" getleft.ini]
        } else {
            set ficherorc [file join $env(HOME) .Getleft]
        }
    }

    if [catch {open $ficherorc r} handle] {
        exec touch $ficherorc
        set options(lang) en
        set urlIndex 0
        return 0
    }

    for {} {![eof $handle]} {} {
        set line [gets $handle]
        if [regexp {^#|^ |^$} $line] {
            continue
        }
        if [regexp {(?:\[)(.*)(?:\])} $line nada active] {
            continue
        }
        switch -regexp -- $active {
            historial|urlHistory {
                 lappend urlHistory $line
            }
            urlIndex|hisIndex {
                set urlIndex $line
            }
            proxy {
                set options(dirProxy) $line
                set options(proxy) 1
            }
            ^language$ {
                set options(lang) $line
            }
            scripts {
                set dirGetleft(scripts) $line
            }
            images {
                set dirGetleft(images) $line
            }
            doc {
                set dirGetleft(doc) $line
            }
            languages {
                set dirGetleft(languages) $line
            }
        }
    }

    close $handle

    if {![info exists options(lang)]} {
        set options(lang) en
    }
    if {![info exists urlIndex]} {
        set urlIndex 0
    }

    if [catch {return [llength $urlHistory]} ] {
        return 0
    }
}

###############################################################################
# SaveConfig
#   Saves the configuration
###############################################################################
proc SaveConfig {} {
    global urlHistory urlIndex dirGetleft options
    global env tcl_platform
    global labelMessages labelTitles

    if [regexp {Windows} $tcl_platform(os)] {
        set ficherorc [file join "$dirGetleft(main)" getleft.ini]
    } else {
        set ficherorc [file join $env(HOME) .Getleft]
    }

    if [catch {open  $ficherorc w+} fichero] {
        tk_messageBox -title $labelTitles(error) -icon Error \
            -message $labelMessages(saveConfig)
        return
    }

    puts $fichero "\[urlHistory\]\n"
    if {![catch {llength $urlHistory} len]} {
        for {set i 0} {$i<$len} {incr i} {
            puts $fichero [lindex $urlHistory $i]
        }
    }

    puts $fichero "\n\[urlIndex\]\n"
    puts $fichero $urlIndex

    if [info exists options(dirProxy)] {
        puts $fichero "\n\[proxy\]\n"
        puts $fichero "$options(dirProxy)"
    }

    puts $fichero "\n\[language\]\n"
    puts $fichero "$options(lang)"

    close $fichero

    return
}

###############################################################################
# MostrarHistorial
#    Shows the url history
###############################################################################
proc MostrarHistorial {} {
    global urlHistory urlIndex

    set len [llength $urlHistory]
    for {set i 0} {$i<$len} {incr i} {
        puts "$i: [lindex $urlHistory $i]"
    }
    return
}

###############################################################################
# EnHistorial
#    Checks whether the given url is already in the url history
#
# Parameter:
#    pagina: given url
#
# Returns:
#    1 if it is.
#    0 if it is not
###############################################################################
proc EnHistorial {pagina} {
    global urlHistory

    set len [llength $urlHistory]
    for {set i 0} {$i<$len} {incr i} {
        if [catch {regsub {(http://)||(ftp://)} \
                [lindex $urlHistory $i] {} una}] {
            return 0
        }
        regsub {(http://)||(ftp://)} $pagina {} dos

        if [string match $una $dos] {
            return 1
        }
    }
    return 0
}

###############################################################################
# ChangeLanguage
#    Changes the default language for the program
#
# Parameters:
#    lang: language to be used
###############################################################################
proc ChangeLanguage {lang} {
    global options

    set options(lang) $lang
    MenusLabels $lang

    SaveConfig

    return
}

###############################################################################
# ReadSupportedLang
#    Reads the 'supported' file to know which languages this version supports
#    and puts then in supportedLang global variable
###############################################################################
proc ReadSupportedLang {} {
    global dirGetleft supportedLang
    global labelTitles

    if [catch {open [file join $dirGetleft(languages) supported] r} handle] {
        tk_messageBox -type ok -icon error -title Error -message $handle
        return
    }

    for {} {![eof $handle]} {} {
        set line [gets $handle]
        if [regexp {^#|^ |^$} $line] {
            continue
        }
        set language [lindex $line 1]
        set key      [lindex $line 0]
        set encoding [lindex $line 2]

        set supportedLang($language) $key
        set supportedLang($key)      $encoding
    }
    return
}

###############################################################################
# ReadLangFile
#    Reads one of the files with translations
#
# Parameter
#    The two letters code that identifies one language, i.e., 'es', 'en'
#
# Returns
#    - '0' all went well
#    - '1' there was an error
###############################################################################
proc ReadLangFile {sufix} {
    global dirGetleft supportedLang
    global labelMenus labelButtons  labelTitles labelMessages
    global labelDialogs labelMonths indexMenus

    set languageFile [file join $dirGetleft(languages) menus.$sufix]

    if [catch {open $languageFile r} handle] {
        tk_messageBox -icon error -type ok -title Error \
                -message "File not found"
        return 1
    }
    fconfigure $handle -encoding $supportedLang($sufix)

    for {set active ""} {![eof $handle]} {} {
        set line [gets $handle]
        if [regexp {^#|^ |^$} $line] {
            continue
        }
        if {![string compare $line "End"]} {
            set active ""
            continue
        }
        set key   [lindex $line 0]
        set value [lindex $line 1]

        switch -exact -- $active {
            Menus {
                set indexMenus($key) [string first & $value]
                if {$indexMenus($key)!=-1} {
                    regsub {&} $value {} value
                }
                set labelMenus($key) $value
            }
            Buttons {
                set labelButtons($key) $value
            }
            Titles {
                set labelTitles($key) $value
            }
            Messages {
                set labelMessages($key) $value
            }
            Dialogs {
                set labelDialogs($key) $value
            }
            Months {
                set labelMonths($key) $value
            }
            "" {
                set active $line
            }
        }
    }
    return 0
}

###############################################################################
# MenusLabels
#    Puts the labels into the menus.
#
# Parameters:
#    labFile: file with the labels in the, hopefully, desired language
###############################################################################
proc MenusLabels {{labFile ""}} {
    global menus dirGetleft
    global labelButtons labelTitles labelMessages labelDialogs labelMonths
    global labelMenus indexMenus

    if {$labFile==""} {
        set labFile en
    }

    ReadLangFile $labFile

    .menus      entryconfigure 1 -label $labelMenus(file)           \
            -underline $indexMenus(file)
    $menus(1)   entryconfigure 0 -label $labelMenus(enterUrl)       \
            -underline $indexMenus(enterUrl)
    $menus(1)   entryconfigure 1 -label $labelMenus(siteMap)        \
            -underline $indexMenus(siteMap)
    $menus(1)   entryconfigure 3 -label $labelMenus(stop)           \
            -underline $indexMenus(stop)
    $menus(1,3) entryconfigure 0 -label $labelMenus(stopPage)       \
            -underline $indexMenus(stopPage)
    $menus(1,3) entryconfigure 1 -label $labelMenus(stopFile)       \
            -underline $indexMenus(stopFile)
    $menus(1,3) entryconfigure 2 -label $labelMenus(stopNow)        \
            -underline $indexMenus(stopNow)
    $menus(1)   entryconfigure 4 -label $labelMenus(pause)          \
            -underline $indexMenus(pause)
    $menus(1,4) entryconfigure 0 -label $labelMenus(stopPage)       \
            -underline $indexMenus(stopPage)
    $menus(1,4) entryconfigure 1 -label $labelMenus(stopFile)       \
            -underline $indexMenus(stopFile)
    $menus(1)   entryconfigure 6 -label $labelMenus(exit)           \
            -underline $indexMenus(exit)

    .menus      entryconfigure 2 -label $labelMenus(options)        \
            -underline $indexMenus(options)
    $menus(2)   entryconfigure 0 -label $labelMenus(uplinks)        \
            -underline $indexMenus(uplinks)
    $menus(2,0) entryconfigure 0 -label $labelMenus(linksFollow)    \
            -underline $indexMenus(linksFollow)
    $menus(2,0) entryconfigure 1 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 1 -label $labelMenus(levels)         \
            -underline $indexMenus(levels)
    $menus(2,0) entryconfigure 2 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2,1) entryconfigure 0 -label $labelMenus(noLimit)        \
            -underline $indexMenus(noLimit)
    $menus(2)   entryconfigure 2 -label $labelMenus(external)       \
            -underline $indexMenus(external)
    $menus(2,2) entryconfigure 0 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 3 -label $labelMenus(fileFilter)     \
            -underline $indexMenus(fileFilter)
    $menus(2,3) entryconfigure 0 -label $labelMenus(onlyHtml)       \
            -underline $indexMenus(onlyHtml)
    $menus(2,3) entryconfigure 1 -label $labelMenus(images)         \
            -underline $indexMenus(images)
    $menus(2,3,1) entryconfigure 0 -label $labelMenus(onlyThumb)    \
            -underline $indexMenus(onlyThumb)
    $menus(2,3,1) entryconfigure 1 -label $labelMenus(noThumb)      \
            -underline $indexMenus(noThumb)
    $menus(2,3) entryconfigure 2 -label $labelMenus(chooseFilter)   \
            -underline $indexMenus(chooseFilter)
    $menus(2)   entryconfigure 4 -label $labelMenus(resume)         \
            -underline $indexMenus(resume)
    $menus(2)   entryconfigure 5 -label $labelMenus(update)         \
            -underline $indexMenus(update)
    $menus(2)   entryconfigure 6 -label $labelMenus(cgi)            \
            -underline $indexMenus(cgi)
    $menus(2)   entryconfigure 7 -label $labelMenus(useProxy)       \
            -underline $indexMenus(useProxy)
    $menus(2)   entryconfigure 8 -label $labelMenus(checkSize)      \
            -underline $indexMenus(checkSize)

    .menus      entryconfigure 3 -label $labelMenus(tools)          \
            -underline $indexMenus(tools)
    $menus(3)   entryconfigure 0 -label $labelMenus(purgeFiles)     \
            -underline $indexMenus(purgeFiles)
    $menus(3)   entryconfigure 1 -label $labelMenus(restoreOrig)    \
            -underline $indexMenus(restoreOrig)
    $menus(3)   entryconfigure 2 -label $labelMenus(configureProxy) \
            -underline $indexMenus(configureProxy)
    $menus(3)   entryconfigure 3 -label $labelMenus(languages)      \
            -underline $indexMenus(languages)
    if {$::DEBUG==1} {
        $menus(3)   entryconfigure 4 -label $labelMenus(resource)   \
                -underline $indexMenus(resource)
    }

    .menus      entryconfigure 4 -label $labelMenus(help)           \
            -underline $indexMenus(help)
    $menus(4)   entryconfigure 0 -label $labelMenus(manual)         \
            -underline $indexMenus(manual)
    $menus(4)   entryconfigure 2 -label $labelMenus(changes)        \
            -underline $indexMenus(changes)
    $menus(4)   entryconfigure 3 -label $labelMenus(license)        \
            -underline $indexMenus(license)
    $menus(4)   entryconfigure 5 -label $labelMenus(about)          \
            -underline $indexMenus(about)

    return
}

###############################################################################
# ShowMenu1
#    Before posting the file menu, this procedure decides which entries
#    should be active, and which will have to wait
###############################################################################
proc ShowMenu1 {} {
    global getleftState

    if {$getleftState(downloading)==0} {
        .menus.fichero entryconfigure 0 -state normal
        .menus.fichero entryconfigure 1 -state normal
        .menus.fichero entryconfigure 3 -state disabled
        .menus.fichero entryconfigure 4 -state disabled
    } else {
        .menus.fichero entryconfigure 0 -state disabled
        .menus.fichero entryconfigure 1 -state disabled
        .menus.fichero entryconfigure 3 -state normal
        .menus.fichero entryconfigure 4 -state normal
    }
    return
}

###############################################################################
# CreateMenus
#   Creates the menus, without putting the labels
###############################################################################
proc CreateMenus {} {
    global options dirGetleft menus supportedLang

    menu .menus -relief flat

    set menus(1) [menu .menus.fichero      -tearoff 0 -postcommand ShowMenu1]
    .menus add cascade -menu $menus(1)
    set menus(2) [menu .menus.editar       -tearoff 0]
    .menus add cascade -menu $menus(2)
    set menus(3) [menu .menus.herramientas -tearoff 0]
    .menus add cascade -menu $menus(3)
    set menus(4) [menu .menus.help         -tearoff 0]
    .menus add cascade -menu $menus(4)

    $menus(1) add command -command EnterUrl
    $menus(1) add command -command GetSiteMap
    $menus(1) add separator
    $menus(1) add cascade -menu $menus(1).stop
    $menus(1) add cascade -menu $menus(1).pause
    $menus(1) add separator
    $menus(1) add command -command exit

    set menus(1,3) [menu $menus(1).stop -tearoff 0]
    $menus(1,3) add check -variable options(stopPage) -onvalue 1 -offvalue 0
    $menus(1,3) add check -variable options(stopFile) -onvalue 1 -offvalue 0
    $menus(1,3) add command -command ::Ventana::CancelDownloading

    set menus(1,4) [menu $menus(1).pause -tearoff 0]
    $menus(1,4) add check -variable options(pausePage) -onvalue 1 -offvalue 0
    $menus(1,4) add check -variable options(pauseFile) -onvalue 1 -offvalue 0

    $menus(2) add cascade -menu $menus(2).directorios
    $menus(2) add cascade -menu $menus(2).levels
    $menus(2) add cascade -menu $menus(2).exLevels
    $menus(2) add cascade -menu $menus(2).filters
    $menus(2) add check   -variable options(resume)    -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(update)    -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(cgi)       -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(proxy)     -onvalue 1 -offvalue 0
    $menus(2) add check   -variable options(checkSize) -onvalue 1 -offvalue 0

    set options(cgi)       0
    set options(resume)    1
    set options(update)    0
    set options(checkSize) 1
    set options(external)  0

    set menus(2,0) [menu $menus(2).directorios -tearoff 0]
    $menus(2,0) add radio -variable options(dir) -value 1
    $menus(2,0) add radio -variable options(dir) -value 0
    set options(dir) 1

    set menus(2,1) [menu $menus(2).levels -tearoff 0]
    $menus(2,1) add radio -variable options(levels) -value -1
    for {set i 0} {$i<6} {incr i} {
        $menus(2,1) add radio -label $i -variable options(levels) -value $i
    }

    set menus(2,2) [menu $menus(2).exLevels -tearoff 0]
    $menus(2,2) add radio -variable options(exLevels) -value 0      \
            -command {set options(external) 0}
    for {set i 1} {$i<3} {incr i} {
        $menus(2,2) add radio -label $i -variable options(exLevels) \
            -value $i -command {set options(external) 1}
    }

    set menus(2,3) [menu $menus(2).filters -tearoff 0]
    $menus(2,3) add check -variable options(onlyHtml) -onvalue 1 -offvalue 0
    $menus(2,3) add cascade -menu $menus(2,3).images
    $menus(2,3) add command -command Herramientas::FilterFiles

    set options(levels) -1
    set options(exLevels) 0

    set menus(2,3,1) [menu $menus(2,3).images -tearoff 0]
    $menus(2,3,1) add check -variable options(onlyThumb) -onvalue 1 -offvalue 0 \
            -command "set options(noThumb) 0"
    $menus(2,3,1) add check -variable options(noThumb)   -onvalue 1 -offvalue 0 \
            -command "set options(onlyThumb) 0"

    set options(onlyThumb) 0
    set options(noThumb)   0

    $menus(3) add command -command Herramientas::PurgeFiles
    $menus(3) add command -command Herramientas::RestoreOriginals
    $menus(3) add command -command Herramientas::ConfProxy
    $menus(3) add cascade -menu $menus(3).idiomas
    if {$::DEBUG==1} {
        $menus(3) add command -command \
                "source [file join "$dirGetleft(main)" Getleft.tcl]"
    }

    set menus(3,0) [menu $menus(3).idiomas -tearoff 0]
    set i 0
    foreach lang [lsort [array names supportedLang]] {
        if [file exists [file join $dirGetleft(languages) \
                menus.$supportedLang($lang)]] {
            $menus(3,0) add command \
                    -command "ChangeLanguage $supportedLang($lang)"
            $menus(3,0) entryconfigure $i -label $lang
            incr i
        }
    }

    $menus(4) add command -command Ayuda::Manual
    $menus(4) add separator
    $menus(4) add command -command Ayuda::Changes
    $menus(4) add command -command Ayuda::Licence
    $menus(4) add separator
    $menus(4) add command -command Ayuda::About

    MenusLabels $options(lang)

    return
}

###############################################################################
#				PROGRAMA PRINCIPAL
###############################################################################

set DEBUG 0

set dirGetleft(main) [file dirname [info script]]
if [file exists /etc/getlet.conf] {
    ReadConfig /etc/getleft.conf
} elseif [file exists /usr/local/etc/getleft.conf] {
    ReadConfig /usr/local/etc/getleft.conf
} else {
    regexp {(?:.:)(.*)} "$dirGetleft(main)" nada dirGetleft(main)
    set dirGetleft(scripts)   [file join "$dirGetleft(main)" scripts]
    set dirGetleft(images)    [file join "$dirGetleft(main)" images]
    set dirGetleft(doc)       [file join "$dirGetleft(main)" doc]
    set dirGetleft(languages) [file join "$dirGetleft(main)" Languages]
}

source [file join "$dirGetleft(scripts)" Ccombinado.tcl]
source [file join "$dirGetleft(scripts)" Ventana.tcl]
source [file join "$dirGetleft(scripts)" Dialogos.tcl]
source [file join "$dirGetleft(scripts)" Herramientas.tcl]
source [file join "$dirGetleft(scripts)" Ayuda.tcl]
source [file join "$dirGetleft(scripts)" HtmlParser.tcl]

if {![winfo exists .menus]} {
    ReadSupportedLang
    ReadLangFile en

    wm title . "Getleft v 0.8.3"
    if [regexp {Windows} $tcl_platform(os)] {
         wm geometry . 300x25
    } else {
         wm geometry . 300x1
    }
    . config -menu .menus

    wm protocol . WM_DELETE_WINDOW exit

    ReadConfig
    set options(filter)            ""
    set options(map)               0
    set getleftState(downloading)  0

    bind Button <Return>   {%W invoke}
    bind Button <KP_Enter> {%W invoke}

    CreateMenus
}

